#include "htmlexport.h"
#include "fileop.h"
#include "border.h"
#include "imageutils.h"
#include "ifapp.h"
#include <qfile.h>
#include <qfileinfo.h>
#include <klocale.h>
#include <kglobal.h>
#include <kconfig.h>

#include <math.h>

HTMLExportBase::HTMLExportBase(QObject *parent, const char *name)
    : QObject(parent, name)
{
    ;
}

void HTMLExportBase::run(const QString &path,
                         const QStringList &fileList)
{
    loadSettings();

    imagesPerPage = data.rows * data.cols;
    pages = (int)ceil(((float)fileList.count())/imagesPerPage);

    qWarning("Total images: %d, images per page %d, pages: %d",
             fileList.count(), imagesPerPage, pages);

    it = fileList.begin();
    end = fileList.end();
    if(data.style == 0 || data.style == 1){ // Grid style
        // main loop for HTML
        int i;
        QFile file;
        QTextStream stream;
        for(i=1; i <= pages; ++i){
            file.setName(path + "/" + filename(i));
            if(!file.open(IO_WriteOnly)){
                // TODO error message
                qWarning("Unable to write to file %s!",
                         QString(path+"/"+filename(i)).latin1());
                return;
            }
            stream.setDevice(&file);
            writeHeader(stream);
            if(data.style == 0){
                writeGrid(stream, false);
                writePageIndex(i, stream);
            }
            else if(data.style == 1){
                writeGrid(stream, true);
                writePageIndex(i, stream);
            }
            writeFooter(stream);
            file.close();
        }
    }
    else if(data.style == 2)
        writeThumbnailFrame(path, fileList);
    else
        writePlainFrame(path, fileList);

    if(data.style != 3){
        // main loop for thumbnails
        emit setStatusBarText(i18n("Creating HTML gallery thumbnails..."));
        kifapp()->processEvents();
        for(it = fileList.begin(); it != fileList.end(); ++it){
            createThumbnail(path, *it);
        }
        emit setStatusBarText(i18n("Finished HTML gallery"));
    }
}

void HTMLExportBase::writeThumbnailFrame(const QString &path,
                                         const QStringList &fileList)
{
    QFile file;
    QTextStream stream;
    int thumbStart = data.useBannerAd ? 4 : 3; // TODO : use this on plain frames
    int tw = data.thumbWidth;
    int th = data.thumbHeight;
    if(data.borderStyle != 0){
        int type = data.borderStyle -1;
        if(type == Solid || type == Bevel || type == Liquid){
            tw += data.frameWidth*2;
            th += data.frameWidth*2;
        }
    }
    // write main html file
    file.setName(path + "/" + data.filenamePrefix + ".html");
    if(!file.open(IO_WriteOnly)){
        // TODO error message
        qWarning("Unable to write to file %s!",
                 QString(path+data.filenamePrefix+".html").latin1());
        return;
    }
    stream.setDevice(&file);
    stream << "<HTML>\n";
    if(data.useBannerAd){
        stream << "<FRAMESET ROWS=\"" << data.bannerHeight+2 << ",*\">\n";
        stream << "<FRAME NAME=\"banner\" SRC=\"" << data.filenamePrefix +
            "3.html\" MARGINHEIGHT=\"1\" FRAMEBORDER=\"0\" SCROLLING=\"NO\">\n";
    }
    stream << "<FRAMESET COLS=\"" << "150" << ",*\">\n";

    stream << "<FRAME NAME=\"browser\" SRC=\"" << data.filenamePrefix +
        QString::number(thumbStart) + ".html\">\n";
    stream << "<FRAME NAME=\"viewer\" SRC=\"" << data.filenamePrefix +
        "2.html\">\n";
    stream << "</FRAMESET>\n";
    if(data.useBannerAd){
        stream << "</FRAMESET>\n";
    }
    stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
    stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
    stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
    stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
    stream << ">" << '\n';
    stream << i18n("Viewing this page requires a frames capable browser!") << '\n';
    stream << "</BODY>\n" << "</HTML>";
    file.close();

    // write view stub html file
    file.setName(path + "/" + data.filenamePrefix + "2.html");
    if(!file.open(IO_WriteOnly)){
        // TODO error message
        qWarning("Unable to write to file %s!",
                 QString(path+data.filenamePrefix+"2.html").latin1());
        return;
    }
    stream.setDevice(&file);
    stream << "<HTML>\n";
    stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
    stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
    stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
    stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
    stream << ">" << '\n';
    stream << "</BODY>\n" << "</HTML>";
    file.close();

    // write banner html file
    if(data.useBannerAd){
        file.setName(path + "/" + data.filenamePrefix + "3.html");
        if(!file.open(IO_WriteOnly)){
            // TODO error message
            qWarning("Unable to write to file %s!",
                     QString(path+data.filenamePrefix+"3.html").latin1());
            return;
        }
        stream.setDevice(&file);
        stream << "<HTML>\n";
        stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
        stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
        stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
        stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
        stream << ">" << '\n';
        stream << "<DIV ALIGN=\"CENTER\">" << '\n';
        stream << "<A TARGET=\"_url\" HREF=\"" << data.bannerURL << "\">";
        stream << "<IMG SRC=\"" << data.bannerImageURL << "\" BORDER=\"0\">";
        stream << "</A>" << '\n';
        stream << "</DIV>" << '\n';
        stream << "</BODY>\n" << "</HTML>";
        file.close();
    }

    // write thumbnail frames
    int frames = (int)ceil(((float)fileList.count())/data.rows);
    int i;
    for(i=thumbStart; i < frames+thumbStart; ++i){
        file.setName(path + "/" + data.filenamePrefix + QString::number(i) + ".html");
        if(!file.open(IO_WriteOnly)){
            // TODO error message
            qWarning("Unable to write to file %s!",
                     QString(path+data.filenamePrefix+QString::number(i)+".html").latin1());
            return;
        }
        // header
        stream.setDevice(&file);
        stream << "<HTML>\n";
        stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
        stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
        stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
        stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
        stream << ">" << '\n';
        stream << "<DIV ALIGN=\"CENTER\">";
        stream << data.title << "<P>";
        if(frames > 1)
            stream << i18n("Page") + " "  << i-thumbStart+1 << " " +
                i18n("of") + " " << frames;
        stream << "</DIV>\n";
        // main thumbnail loop
        QFileInfo fi;
        int count;
        QString extStr, thumbStr;
        for(count=0; count < data.rows && it != end; ++count){
            fi.setFile(*it);
            extStr = fi.extension(false).lower();
            if(extStr != "png")
                thumbStr = "thb_" + fi.fileName() + ".png";
            else
                thumbStr = "thb_" + fi.fileName();
            stream << "<A HREF=\"" << fi.fileName() << "\" TARGET=\"viewer\">";
            stream << "<IMG SRC=\"" << thumbStr << "\" ";
            stream << "BORDER=\"0\" WIDTH=\"" << tw;
            stream << "\" HEIGHT=\"" << th << "\">";
            if(data.includeFilename){
                stream << "<BR>";
                stream << fi.fileName();
            }
            stream << "</A><P>\n";
            ++it;
        }
        // footer
        if(i != thumbStart){
            stream << "<A HREF=\"" << data.filenamePrefix + QString::number(i-1) + ".html";
            stream << "\" TARGET=\"browser\">";
            if(data.useBackImage)
                stream << "<IMG SRC=\"" << data.backImageURL << "\" BORDER=\"0\">";
            else
                stream << data.backBtnText;
            stream << "</A><BR>";
        }
        if(i != frames+thumbStart-1){
            stream << "<A HREF=\"" << data.filenamePrefix + QString::number(i+1) + ".html";
            stream << "\" TARGET=\"browser\">";
            if(data.useNextImage)
                stream << "<IMG SRC=\"" << data.nextImageURL << "\" BORDER=\"0\">";
            else
                stream << data.nextBtnText;
            stream << "</A><BR>";

        }
        if(data.useHome){
            stream << "<P>\n";
            stream << "<A HREF=\"" << data.homeURL << "\">";
            if(data.useHomeImage)
                stream << "<IMG SRC=\"" << data.homeImageURL << "\" BORDER=\"0\">";
            else
                stream << data.homeBtnText;
            stream << "</A>\n";

        }
        stream << "</BODY>\n" << "</HTML>";
        file.close();
    }
}

void HTMLExportBase::writePlainFrame(const QString &path,
                                     const QStringList &/*fileList*/)
{
    QFile file;
    QTextStream stream;

    // write main html file
    file.setName(path + "/" + data.filenamePrefix + ".html");
    if(!file.open(IO_WriteOnly)){
        // TODO error message
        qWarning("Unable to write to file %s!",
                 QString(path+data.filenamePrefix+".html").latin1());
        return;
    }
    stream.setDevice(&file);
    stream << "<HTML>\n";
    if(data.useBannerAd){
        stream << "<FRAMESET ROWS=\"" << data.bannerHeight+2 << ",*\">\n";
        stream << "<FRAME NAME=\"banner\" SRC=\"" << data.filenamePrefix +
            "4.html\" MARGINHEIGHT=\"1\" FRAMEBORDER=\"0\" SCROLLING=\"NO\">\n";
    }
    stream << "<FRAMESET COLS=\"" << 150 << ",*\">\n";
    stream << "<FRAME NAME=\"browser\" SRC=\"" << data.filenamePrefix +
        "2.html\">\n";
    stream << "<FRAME NAME=\"viewer\" SRC=\"" << data.filenamePrefix +
        "3.html\">\n";
    stream << "</FRAMESET>\n";
    if(data.useBannerAd){
        stream << "</FRAMESET>\n";
    }
    stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
    stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
    stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
    stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
    stream << ">" << '\n';
    stream << i18n("Viewing this page requires a frames capable browser!") << '\n';
    stream << "</BODY>\n" << "</HTML>";
    file.close();

    // write banner html file
    if(data.useBannerAd){
        file.setName(path + "/" + data.filenamePrefix + "4.html");
        if(!file.open(IO_WriteOnly)){
            // TODO error message
            qWarning("Unable to write to file %s!",
                     QString(path+data.filenamePrefix+"4.html").latin1());
            return;
        }
        stream.setDevice(&file);
        stream << "<HTML>\n";
        stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
        stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
        stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
        stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
        stream << ">" << '\n';
        stream << "<DIV ALIGN=\"CENTER\">" << '\n';
        stream << "<A TARGET=\"_url\" HREF=\"" << data.bannerURL << "\">";
        stream << "<IMG SRC=\"" << data.bannerImageURL << "\" BORDER=\"0\">";
        stream << "</A>" << '\n';
        stream << "</DIV>" << '\n';
        stream << "</BODY>\n" << "</HTML>";
        file.close();
    }

    // write view stub html file
    file.setName(path + "/" + data.filenamePrefix + "3.html");
    if(!file.open(IO_WriteOnly)){
        // TODO error message
        qWarning("Unable to write to file %s!",
                 QString(path+data.filenamePrefix+"3.html").latin1());
        return;
    }
    stream.setDevice(&file);
    stream << "<HTML>\n";
    stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
    stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
    stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
    stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
    stream << ">" << '\n';
    stream << "</BODY>\n" << "</HTML>";
    file.close();

    // write file index html file
    file.setName(path + "/" + data.filenamePrefix + "2.html");
    if(!file.open(IO_WriteOnly)){
        // TODO error message
        qWarning("Unable to write to file %s!",
                 QString(path+data.filenamePrefix+"2.html").latin1());
        return;
    }
    stream.setDevice(&file);
    stream << "<HTML>\n";
    stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
    stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
    stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
    stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
    stream << ">" << '\n';
    stream << "<DIV ALIGN=\"CENTER\">";
    stream << data.title << "<P>";
    stream << "</DIV>\n";
    QFileInfo fi;
    while(it != end){
        fi.setFile(*it);
        //stream << "<A TARGET=\"viewer\" HREF=\"" << fi.fileName() << "\">";
        stream << "<A HREF=\"" << fi.fileName() << "\" TARGET=\"viewer\">";

        stream << fi.fileName() << "</A><BR>\n";
        ++it;
    }
    if(data.useHome){
        stream << "<P>\n";
        stream << "<A HREF=\"" << data.homeURL << "\">";
        if(data.useHomeImage)
            stream << "<IMG SRC=\"" << data.homeImageURL << "\" BORDER=\"0\">";
        else
            stream << data.homeBtnText;
        stream << "</A>\n";

    }
    stream << "</BODY>\n" << "</HTML>";
    file.close();

}

void HTMLExportBase::writeGrid(QTextStream &stream, bool newwindow)
{
    // banner ad
    if(data.useBannerAd){
        stream << "<DIV ALIGN=\"CENTER\">" << '\n';
        stream << "<A TARGET=\"URL\" HREF=\"" << data.bannerURL << "\">";
        stream << "<IMG SRC=\"" << data.bannerImageURL << "\" BORDER=\"0\">";
        stream << "</A>" << '\n';
        stream << "</DIV>" << '\n';
        stream << "<P>" << '\n';
    }

    // do the title
    stream << "<H3>" << data.title << "</H3>" << '\n';
    stream << "<P>" << '\n';


    // do the image table
    stream << "<TABLE BORDER=\"" << data.htmlBorderWidth << "\">" << '\n';
    int r, c;
    QFileInfo fi;
    QString extStr, thumbStr;

    // calculate thumbnail size ahead of time
    int tw = data.thumbWidth;
    int th = data.thumbHeight;
    if(data.borderStyle != 0){
        int type = data.borderStyle -1;
        if(type == Solid || type == Bevel || type == Liquid){
            tw += data.frameWidth*2;
            th += data.frameWidth*2;
        }
    }

    for(r=0; r < data.rows; ++r){
        stream << "<TR ALIGN=\"CENTER\">\n";
        for(c=0; c < data.cols; ++c){
            if(it != end){
                fi.setFile(*it);
                extStr = fi.extension(false).lower();
                if(extStr != "png")
                    thumbStr = "thb_" + fi.fileName() + ".png";
                else
                    thumbStr = "thb_" + fi.fileName();

                stream << "<TD>";

                if(!newwindow)
                    stream << "<A HREF=\"" << fi.fileName() << "\">";
                else
                    stream << "<A TARGET=\"URL\" HREF=\"" << fi.fileName() << "\">";
                //stream << "<IMG SRC=\"" << thumbStr << "\" >";
                stream << "<IMG SRC=\"" << thumbStr << "\" ";
                stream << "BORDER=\"0\" WIDTH=\"" << tw;
                stream << "\" HEIGHT=\"" << th << "\">";
                if(data.includeFilename){
                    stream << "<BR>";
                    stream << fi.fileName();
                }
                stream << "</A>";

                stream << "</TD>\n";
                ++it;
            }

        }
        stream << "</TR>\n";
    }
    stream << "</TABLE>";
}

void HTMLExportBase::createThumbnail(const QString &path,
                                     const QString &fileName)
{
    // check for existing thumbnail
    QImage img;
    QFileInfo fi(fileName);
    QString fn = fi.fileName();
    if(data.thumbWidth == 48 && data.thumbHeight == 48){
        if(QFile::exists(path+"/.pics/small/"+fn)){
            qWarning("Found small thumbnail");
            loadImage(img, path+"/.pics/small/"+fn, "PNG");
        }
    }
    else if(data.thumbWidth == 64 && data.thumbHeight == 64){
        if(QFile::exists(path+"/.pics/med/"+fn)){
            qWarning("Found medium thumbnail");
            loadImage(img, path+"/.pics/med/"+fn, "PNG");
        }
    }
    else if(data.thumbWidth == 90 && data.thumbHeight == 90){
        if(QFile::exists(path+"/.pics/large/"+fn)){
            qWarning("Found large thumbnail");
            loadImage(img, path+"/.pics/large/"+fn, "PNG");
        }
    }
    else if(data.thumbWidth == 120 && data.thumbHeight == 120){
        if(QFile::exists(path+"/.pics/large/"+fn)){
            qWarning("Found huge thumbnail");
            loadImage(img, path+"/.pics/huge/"+fn, "PNG");
        }
    }
    if(img.isNull()){
        if(!loadImage(img, path+"/"+fn)){
            qWarning("Unable to load file: %s", fileName.latin1());
            return;
        }
    }
    // scale image if needed
    if(img.width() > data.thumbWidth || img.height() > data.thumbHeight){
        int w = img.width();
        int h = img.height();
        if(w > data.thumbWidth){
            float percent = ((float)data.thumbWidth)/w;
            w = (int)(w*percent);
            h = (int)(h*percent);
        }
        if(h > data.thumbHeight){
            float percent = ((float)data.thumbHeight)/h;
            w = (int)(w*percent);
            h = (int)(h*percent);
        }
        img = img.smoothScale(w, h);

    }
    // center on fill color
    if(img.width() < data.thumbWidth || img.height() < data.thumbHeight){
        if(img.depth() < 32)
            img = img.convertDepth(32);
        QImage dest(data.thumbWidth, data.thumbHeight, 32);
        if(data.borderStyle == 0){
            dest.fill(data.htmlBgColor.rgb());
        }
        else{
            int type = data.borderStyle-1;
            if(type == Solid || type == Bevel)
                dest.fill(data.frameFill.rgb());
            else if(type == Liquid)
                dest.fill(data.frameFg.rgb());
            else if(type == RoundCorner)
                dest.fill(data.htmlBgColor.rgb());
        }
        int x =  ((data.thumbWidth-img.width())/2);
        int y = ((data.thumbHeight-img.height())/2);
        copyQImageWithAlpha(img, dest, x, y);
        img = dest;
    }
    // do the border
    if(data.borderStyle != 0){
        if(img.depth() < 32)
            img = img.convertDepth(32);
        QImage dest;
        int type = data.borderStyle-1;
        if(type == Solid){
            KIFBorderEffect::solid(img, dest, data.frameFg, data.frameWidth);
        }
        else if(type == Bevel){
            KIFBorderEffect::bevel(img, dest, data.frameFg, data.frameBg, data.frameWidth);
        }
        else if(type == Liquid){
            KIFBorderEffect::liquid(img, dest, data.frameFg, data.htmlBgColor, data.frameWidth);
        }
        else if(type == RoundCorner){
            KIFBorderEffect::roundCorner(img, dest, data.htmlBgColor);
        }
        img = dest;
    }
    QString extStr = fi.extension(false).lower();
    QString thumbStr;
    if(extStr != "png")
        thumbStr = path + "/thb_" + fn + ".png";
    else
        thumbStr = path + "/thb_" + fn;
    if(!saveImage(img, thumbStr, "PNG")){
        // TODO Error Message
        qWarning("Unable to save %s", thumbStr.latin1());
    }

}



void HTMLExportBase::writeHeader(QTextStream &stream)
{
    stream << "<HTML>" << '\n';
    stream << "<BODY BGCOLOR=\"" << data.htmlBgColor.name() << "\" ";
    stream << "TEXT=\"" << data.htmlTextColor.name() << "\" ";
    stream << "ALINK=\"" << data.htmlLinkColor.name() << "\" ";
    stream << "VLINK=\"" << data.htmlFollowedLinkColor.name() << "\" ";
    stream << ">" << '\n';
}

void HTMLExportBase::writeFooter(QTextStream &stream)
{
    stream << "<BODY>" << '\n';
    stream << "<HTML>";
}

void HTMLExportBase::writePageIndex(int page, QTextStream &stream)
{
    if(pages == 1){
        qWarning("Only one page of HTML, no index written");
        return;
    }
    stream << "\n<P>\n";
    // home button
    if(data.useHome){
        stream << "<A HREF=\"" << data.homeURL << "\">";
        if(data.useHomeImage)
            stream << "<IMG SRC=\"" << data.homeImageURL << "\" BORDER=\"0\">";
        else
            stream << data.homeBtnText;
        stream << "</A>";
        stream << "|";
    }
    // back button
    if(page != 1){ 
        stream << "<A HREF=\"" << filename(page-1) << "\">";
        if(data.useBackImage)
            stream << "<IMG SRC=\"" << data.backImageURL << "\" BORDER=\"0\">";
        else
            stream << data.backBtnText;
        stream << "</A>";
        stream << "|";
    }
    // page index
    int i;
    for(i = 1; i <= pages; ++i){
        if(i != page)
            stream << "<A HREF=\"" << filename(i) << "\">";
        stream << QString::number(i);
        if(i != page)
            stream << "</A>";
    }
    // next button
    if(page != pages){
        stream << "|";
        stream << "<A HREF=\"" << filename(page+1) << "\">";
        if(data.useNextImage)
            stream << "<IMG SRC=\"" << data.nextImageURL << "\" BORDER=\"0\">";
        else
            stream << data.nextBtnText;
        stream << "</A>";
    }
    stream << "\n<P>\n";
}

QString HTMLExportBase::filename(int page)
{
    if(pages == 1 || page == 1)
        return(data.filenamePrefix + ".html");
    else
        return(data.filenamePrefix + QString::number(page) + ".html");
}


void HTMLExportBase::loadSettings()
{
    KConfig *config = KGlobal::config();
    QString oldGrp = config->group();
    config->setGroup("HTMLGallery");

    data.rows = config->readNumEntry("Rows", 5);
    data.cols = config->readNumEntry("Cols", 5);
    data.thumbWidth = config->readNumEntry("ThumbWidth", 90);
    data.thumbHeight = config->readNumEntry("ThumbHeight", 90);

    data.borderStyle = config->readNumEntry("BorderStyle", 0);
    data.frameFg = config->readColorEntry("FrameFg", &Qt::gray);
    data.frameBg = config->readColorEntry("FrameBg", &Qt::white);
    data.frameFill = config->readColorEntry("FrameFill", &Qt::lightGray);
    data.frameWidth = config->readNumEntry("FrameWidth", 1);

    data.style = config->readNumEntry("Style", 0);
    data.htmlBorderWidth = config->readNumEntry("HTMLBorderWidth", 0);
    data.includeFilename = config->readBoolEntry("IncludeName", false);

    data.htmlBgColor = config->readColorEntry("HTMLBg", &Qt::white);
    data.htmlTextColor = config->readColorEntry("HTMLText", &Qt::black);
    data.htmlLinkColor = config->readColorEntry("HTMLLink", &Qt::blue);
    data.htmlFollowedLinkColor = config->readColorEntry("HTMLFollowedLink", &Qt::darkMagenta);

    data.homeURL = config->readEntry("HomeURL", i18n("http://"));
    data.homeBtnText = config->readEntry("HomeBtnText", i18n("home"));
    data.homeImageURL = config->readEntry("HomeImageURL", i18n("http://"));
    data.useHomeImage = config->readBoolEntry("UseHomeImage", false);
    data.useHome = config->readBoolEntry("UseHome", false);

    data.nextBtnText = config->readEntry("NextBtnText", i18n("Next >>"));
    data.nextImageURL = config->readEntry("NextImageURL", i18n("http://"));
    data.useNextImage = config->readBoolEntry("UseNextImage", false);

    data.backBtnText = config->readEntry("BackBtnText", i18n("<< Back"));
    data.backImageURL = config->readEntry("BackImageURL", i18n("http://"));
    data.useBackImage = config->readBoolEntry("UseBackImage", false);


    data.filenamePrefix = config->readEntry("FilenamePrefix", i18n("thumbnails"));

    data.bannerURL = config->readEntry("BannerURL", i18n("http://"));
    data.bannerImageURL = config->readEntry("BannerImageURL", i18n("http://"));
    data.bannerHeight = config->readNumEntry("BannerHeight", 90);
    data.useBannerAd = config->readBoolEntry("UseBanner", false);

    data.title = config->readEntry("Title", i18n("Web Image Gallery"));

    config->sync();
    config->setGroup(oldGrp);
}

