/* Copyright (c) 2001-2011, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.test;

import com.pixelmed.dicom.ContentItem;
import com.pixelmed.dose.*;

import junit.framework.*;

import com.pixelmed.dicom.CodedSequenceItem;

public class TestCTAcquisitionParameters extends TestCase {
	
	// constructor to support adding tests to suite ...
	
	public TestCTAcquisitionParameters(String name) {
		super(name);
	}
	
	// add tests to suite manually, rather than depending on default of all test...() methods
	// in order to allow adding TestCTAcquisitionParameters.suite() in AllTests.suite()
	// see Johannes Link. Unit Testing in Java pp36-47
	
	public static Test suite() {
		TestSuite suite = new TestSuite("TestCTAcquisitionParameters");
		
		suite.addTest(new TestCTAcquisitionParameters("TestCTAcquisitionParametersConstructor_WithAllParameters"));
		suite.addTest(new TestCTAcquisitionParameters("TestCTAcquisitionParametersConstructor_Equality"));
		suite.addTest(new TestCTAcquisitionParameters("TestCTAcquisitionParametersConstructor_Equality_WithTID10014ScanningLength"));
		
		return suite;
	}
	
	protected void setUp() {
	}
	
	protected void tearDown() {
	}
	
	public void TestCTAcquisitionParametersConstructor_WithAllParameters() throws Exception {
		
		String irradiationEventUID = "1.2.3.4";
		String exposureTimeInSeconds = "1";
		String scanningLengthInMM = "750";
		String lengthOfReconstructableVolumeInMM = "650";
		String exposedRangeInMM = "760";
		String topZLocationOfReconstructableVolume= "-50";
		String topZLocationOfReconstructableVolumeExpected= "I50";
		String bottomZLocationOfReconstructableVolume = "-700";
		String bottomZLocationOfReconstructableVolumeExpected = "I700";
		String topZLocationOfScanningLength = "0";
		String topZLocationOfScanningLengthExpected = "S0";
		String bottomZLocationOfScanningLength = "-750";
		String bottomZLocationOfScanningLengthExpected = "I750";
		String frameOfReferenceUID = "1.2.3.6";
		String nominalSingleCollimationWidthInMM = "0.625";
		String nominalTotalCollimationWidthInMM = "40";
		String pitchFactor = "0.984";
		String kvp = "120";
		String tubeCurrent = "397";
		String tubeCurrentMaximum = "433";
		String exposureTimePerRotation = "0.6";
		String anatomyCodeMeaning = "Chest, Abdomen and Pelvis";
		CodedSequenceItem anatomy = new CodedSequenceItem("R-FAB56","SRT",anatomyCodeMeaning);
		CTScanType scanType = CTScanType.HELICAL;
		
		CTAcquisitionParameters ctAcquisitionParameters = new CTAcquisitionParameters(irradiationEventUID,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,lengthOfReconstructableVolumeInMM,exposedRangeInMM,topZLocationOfReconstructableVolume,bottomZLocationOfReconstructableVolume,topZLocationOfScanningLength,bottomZLocationOfScanningLength,frameOfReferenceUID,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		
		String expectToString = "\tIrradiationEventUID="+irradiationEventUID+"\tHelical\tAnatomy="+anatomy
			+"\tScanningLength="+scanningLengthInMM+" mm ["+bottomZLocationOfScanningLengthExpected+"-"+topZLocationOfScanningLengthExpected+"]"
			+"\tLengthOfReconstructableVolume="+lengthOfReconstructableVolumeInMM+" mm ["+bottomZLocationOfReconstructableVolumeExpected+"-"+topZLocationOfReconstructableVolumeExpected+"]"
			+"\tExposedRange="+exposedRangeInMM+" mm"
			+"\tFrameOfReferenceUID="+frameOfReferenceUID
			+"\tCollimation single/total="+nominalSingleCollimationWidthInMM+"/"+nominalTotalCollimationWidthInMM+" mm"+"\tPitchFactor="+pitchFactor+":1"
			+"\tKVP="+kvp+" kVP"+"\tTubeCurrent/Max="+tubeCurrent+"/"+tubeCurrentMaximum+" mA"+"\tExposure time/per rotation="+exposureTimeInSeconds+"/"+exposureTimePerRotation+" s\n";
		
		String expectToStringPretty = "\tHelical\t"+anatomyCodeMeaning
			+"\t"+scanningLengthInMM+" mm ["+bottomZLocationOfScanningLengthExpected+"-"+topZLocationOfScanningLengthExpected+"]"
			+"\t"+lengthOfReconstructableVolumeInMM+" mm ["+bottomZLocationOfReconstructableVolumeExpected+"-"+topZLocationOfReconstructableVolumeExpected+"]"
			+"\t"+exposedRangeInMM+" mm"
			+"\t"+nominalSingleCollimationWidthInMM+"/"+nominalTotalCollimationWidthInMM+" mm"+"\t"+pitchFactor+":1"
			+"\t"+kvp+" kVP"+"\t"+tubeCurrent+"/"+tubeCurrentMaximum+" mA"+"\t"+exposureTimeInSeconds+"/"+exposureTimePerRotation+" s\n";
		
		String expectGetHTMLHeader =
			  "<th>Type</th><th>Anatomy</th><th>Scanning Length mm</th><th>Reconstructable Volume mm</th><th>Exposed Range mm</th><th>Collimation Single/Total mm</th><th>Pitch Factor</th><th>kVP</th><th>Tube Current Mean/Max mA</th><th>Exposure Time/Per Rotation s</th>"
			;
		
		String expectGetHTML =
			  "<td>Helical</td><td>Chest, Abdomen and Pelvis"
			  +"</td><td>"+scanningLengthInMM+" ["+bottomZLocationOfScanningLengthExpected+"-"+topZLocationOfScanningLengthExpected+"]"
			  +"</td><td>"+lengthOfReconstructableVolumeInMM+" ["+bottomZLocationOfReconstructableVolumeExpected+"-"+topZLocationOfReconstructableVolumeExpected+"]"
			  +"</td><td>"+exposedRangeInMM
			  +"</td><td>"+nominalSingleCollimationWidthInMM+"/"+nominalTotalCollimationWidthInMM
			  +"</td><td>"+pitchFactor+":1"
			  +"</td><td>"+kvp
			  +"</td><td>"+tubeCurrent+"/"+tubeCurrentMaximum
			  +"</td><td>1/"+exposureTimePerRotation+"</td>"
			;

		assertEquals("Checking IrradiationEventUID equality",irradiationEventUID,ctAcquisitionParameters.getIrradiationEventUID());
		assertEquals("Checking Anatomy equality",anatomy,ctAcquisitionParameters.getAnatomy());
		assertEquals("Checking ExposureTimeInSeconds equality",exposureTimeInSeconds,ctAcquisitionParameters.getExposureTimeInSeconds());
		assertEquals("Checking ScanningLengthInMM equality",scanningLengthInMM,ctAcquisitionParameters.getScanningLengthInMM());
		assertEquals("Checking NominalSingleCollimationWidthInMM equality",nominalSingleCollimationWidthInMM,ctAcquisitionParameters.getNominalSingleCollimationWidthInMM());
		assertEquals("Checking NominalTotalCollimationWidthInMM equality",nominalTotalCollimationWidthInMM,ctAcquisitionParameters.getNominalTotalCollimationWidthInMM());
		assertEquals("Checking PitchFactor equality",pitchFactor,ctAcquisitionParameters.getPitchFactor());
		assertEquals("Checking KVP equality",kvp,ctAcquisitionParameters.getKVP());
		assertEquals("Checking TubeCurrent equality",tubeCurrent,ctAcquisitionParameters.getTubeCurrent());
		assertEquals("Checking TubeCurrentMaximum equality",tubeCurrentMaximum,ctAcquisitionParameters.getTubeCurrentMaximum());
		assertEquals("Checking ExposureTimePerRotation equality",exposureTimePerRotation,ctAcquisitionParameters.getExposureTimePerRotation());
		
		assertEquals("Checking toString",expectToString,ctAcquisitionParameters.toString());
		assertEquals("Checking toString pretty",expectToStringPretty,ctAcquisitionParameters.toString(true));
		assertEquals("Checking getHTML header",expectGetHTMLHeader,ctAcquisitionParameters.getHTMLTableHeaderRowFragment());
		assertEquals("Checking getHTML",expectGetHTML,ctAcquisitionParameters.getHTMLTableRowFragment());
		
		// check round trip via SR ...
		
		ContentItem srFragment = ctAcquisitionParameters.getStructuredReportFragment(null);
		CTAcquisitionParameters ctAcquisitionParameters2 = new CTAcquisitionParameters(srFragment);

		assertEquals("Checking round trip ExposureTimeInSeconds equality",exposureTimeInSeconds,ctAcquisitionParameters2.getExposureTimeInSeconds());
		assertEquals("Checking round trip ScanningLengthInMM equality",scanningLengthInMM,ctAcquisitionParameters2.getScanningLengthInMM());
		assertEquals("Checking round trip ExposedRangeInMM equality",exposedRangeInMM,ctAcquisitionParameters2.getExposedRangeInMM());
		assertEquals("Checking round trip TopZLocationOfReconstructableVolume equality",topZLocationOfReconstructableVolume,ctAcquisitionParameters2.getTopZLocationOfReconstructableVolume());
		assertEquals("Checking round trip BottomZLocationOfReconstructableVolume equality",bottomZLocationOfReconstructableVolume,ctAcquisitionParameters2.getBottomZLocationOfReconstructableVolume());
		assertEquals("Checking round trip TopZLocationOfScanningLength equality",topZLocationOfScanningLength,ctAcquisitionParameters2.getTopZLocationOfScanningLength());
		assertEquals("Checking round trip BottomZLocationOfScanningLength equality",bottomZLocationOfScanningLength,ctAcquisitionParameters2.getBottomZLocationOfScanningLength());
		assertEquals("Checking round trip FrameOfReferenceUID equality",frameOfReferenceUID,ctAcquisitionParameters2.getFrameOfReferenceUID());
		assertEquals("Checking round trip NominalSingleCollimationWidthInMM equality",nominalSingleCollimationWidthInMM,ctAcquisitionParameters2.getNominalSingleCollimationWidthInMM());
		assertEquals("Checking round trip NominalTotalCollimationWidthInMM equality",nominalTotalCollimationWidthInMM,ctAcquisitionParameters2.getNominalTotalCollimationWidthInMM());
		assertEquals("Checking round trip PitchFactor equality",pitchFactor,ctAcquisitionParameters2.getPitchFactor());
		// do NOT expect ScanType round trip equality, since though ScanType is extracted from SR by constructor, it is inserted by CTDoseAcquisition, not CTAcquisitionParameters
		//assertEquals("Checking round trip ScanType equality",scanType,ctAcquisitionParameters2.getScanType());
	}
	
	public void TestCTAcquisitionParametersConstructor_Equality() throws Exception {
		
		String irradiationEventUID1 = "1.2.3.4";
		String irradiationEventUID2 = "1.2.3.5";
		String exposureTimeInSeconds = "1";
		String scanningLengthInMM = "750";
		String nominalSingleCollimationWidthInMM = "0.625";
		String nominalTotalCollimationWidthInMM = "40";
		String pitchFactor = "0.984";
		String kvp = "120";
		String tubeCurrent = "397";
		String tubeCurrentMaximum = "433";
		String exposureTimePerRotation = "0.6";
		String anatomyCodeMeaning = "Chest, Abdomen and Pelvis";
		CodedSequenceItem anatomy = new CodedSequenceItem("R-FAB56","SRT",anatomyCodeMeaning);
		CTScanType scanType = CTScanType.HELICAL;
		
		CTAcquisitionParameters ctAcquisitionParameters1 = new CTAcquisitionParameters(irradiationEventUID1,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		CTAcquisitionParameters ctAcquisitionParameters2 = new CTAcquisitionParameters(irradiationEventUID1,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		CTAcquisitionParameters ctAcquisitionParameters3 = new CTAcquisitionParameters(irradiationEventUID2,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		
		assertTrue("Checking CTAcquisitionParameters equality",ctAcquisitionParameters1.equals(ctAcquisitionParameters2));
		assertTrue("Checking CTAcquisitionParameters equality apart from event UID",ctAcquisitionParameters1.equalsApartFromIrradiationEventUID(ctAcquisitionParameters2));
		assertTrue("Checking CTAcquisitionParameters inequality with different event UID",!ctAcquisitionParameters1.equals(ctAcquisitionParameters3));
		assertTrue("Checking CTAcquisitionParameters equality apart from different event UID",ctAcquisitionParameters1.equalsApartFromIrradiationEventUID(ctAcquisitionParameters3));
	}
	
	public void TestCTAcquisitionParametersConstructor_Equality_WithTID10014ScanningLength() throws Exception {
		
		String irradiationEventUID1 = "1.2.3.4";
		String irradiationEventUID2 = "1.2.3.5";
		String exposureTimeInSeconds = "1";
		String scanningLengthInMM = "750";
		String lengthOfReconstructableVolumeInMM = "650";
		String exposedRangeInMM = "760";
		String topZLocationOfReconstructableVolume= "-50";
		String bottomZLocationOfReconstructableVolume = "-700";
		String topZLocationOfScanningLength = "0";
		String bottomZLocationOfScanningLength = "-750";
		String frameOfReferenceUID = "1.2.3.6";
		String nominalSingleCollimationWidthInMM = "0.625";
		String nominalTotalCollimationWidthInMM = "40";
		String pitchFactor = "0.984";
		String kvp = "120";
		String tubeCurrent = "397";
		String tubeCurrentMaximum = "433";
		String exposureTimePerRotation = "0.6";
		String anatomyCodeMeaning = "Chest, Abdomen and Pelvis";
		CodedSequenceItem anatomy = new CodedSequenceItem("R-FAB56","SRT",anatomyCodeMeaning);
		CTScanType scanType = CTScanType.HELICAL;

		CTAcquisitionParameters ctAcquisitionParameters1 = new CTAcquisitionParameters(irradiationEventUID1,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,lengthOfReconstructableVolumeInMM,exposedRangeInMM,topZLocationOfReconstructableVolume,bottomZLocationOfReconstructableVolume,topZLocationOfScanningLength,bottomZLocationOfScanningLength,frameOfReferenceUID,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		CTAcquisitionParameters ctAcquisitionParameters2 = new CTAcquisitionParameters(irradiationEventUID1,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,lengthOfReconstructableVolumeInMM,exposedRangeInMM,topZLocationOfReconstructableVolume,bottomZLocationOfReconstructableVolume,topZLocationOfScanningLength,bottomZLocationOfScanningLength,frameOfReferenceUID,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		CTAcquisitionParameters ctAcquisitionParameters3 = new CTAcquisitionParameters(irradiationEventUID2,scanType,anatomy,exposureTimeInSeconds,scanningLengthInMM,lengthOfReconstructableVolumeInMM,exposedRangeInMM,topZLocationOfReconstructableVolume,bottomZLocationOfReconstructableVolume,topZLocationOfScanningLength,bottomZLocationOfScanningLength,frameOfReferenceUID,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,tubeCurrentMaximum,exposureTimePerRotation);
		
		assertTrue("Checking CTAcquisitionParameters equality",ctAcquisitionParameters1.equals(ctAcquisitionParameters2));
		assertTrue("Checking CTAcquisitionParameters equality apart from event UID",ctAcquisitionParameters1.equalsApartFromIrradiationEventUID(ctAcquisitionParameters2));
		assertTrue("Checking CTAcquisitionParameters inequality with different event UID",!ctAcquisitionParameters1.equals(ctAcquisitionParameters3));
		assertTrue("Checking CTAcquisitionParameters equality apart from different event UID",ctAcquisitionParameters1.equalsApartFromIrradiationEventUID(ctAcquisitionParameters3));
	}
	
}
