/*
 * Some functions common to many PIMPPA programs
 *
 */

#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <utime.h>

#include "pimppa.h"
#include "common.h"

/*
 * Creates an assign pattern for file "filename". The pattern
 * must have atleast "minlen" characters before the extension.
 * If yes, the pattern is assigned to "destarea", in case
 * the area is suitable for assignation. "context" is the
 * defined context of "destarea".
 *
 * Returns: 1 if assign done, 0 otherwise
 *
 */
int p_assign(MYSQL *db, char *filename, int minlen, int destarea,
			int context)
{
	char pattern[P_LEN_FILE_NAME];
	MYSQL_RES *sql_res;
		
	// Check if NOASSIGN or INCOMING is set for "destarea"
	p_query(db, "SELECT DISTINCT 1 FROM p_areas "
	            "WHERE area_id=%d AND ((area_flags & %ld) "
		    "      OR (area_flags & %ld))",
	 	    destarea, AREA_NOASSIGN, AREA_INCOMING);
	sql_res=mysql_store_result(db);
	if(sql_res)
	{
		if(mysql_num_rows(sql_res)>0)
		{
			mysql_free_result(sql_res);
			return(0);
		}
		mysql_free_result(sql_res);
	}

	
	if(p_makepattern(pattern, filename)>=minlen)
	{
		char escaped_pat[2*P_LEN_FILE_NAME+1];
		
		mysql_escape_string(escaped_pat, pattern, strlen(pattern));
	
		// Check if assign to 0 exists (==assign disabled)
		p_query(db, "SELECT 1 FROM p_assign WHERE a_pattern='%s'"
			    "	AND a_dest=0 "
			    "	AND (a_context=%d OR a_context=0)", 
					escaped_pat,
					context);
		sql_res=mysql_store_result(db);
		if(sql_res)
		{
			if(mysql_num_rows(sql_res)>0)
			{
				mysql_free_result(sql_res);
				return(0);
			}
			mysql_free_result(sql_res);
		}

		// Ok, looks good, its insert or replace.
		p_query(db, "REPLACE INTO p_assign (a_pattern, a_dest, a_context) "
			     " VALUES ('%s', %d, %d)",
				escaped_pat, destarea, context);
	
		return(1);
	}

	return(0);
}

/*
 * Checks if a string has space in it (" ")
 *
 */
int p_checkforspace(char *string)
{
	int i=0;

	if(!string)
		return(0);

	while(string[i]!=0)
	{
		if(string[i]==' ')
			return(1);
		i++;
	}

	return(0);
}

/*
 * Returns the amount of numbers in a string (before the dot)
 *
 */
int p_checkfornumber(char *string)
{
	int i=0,numbers=0,len=0;

	if(!string)
		return(0);

	for(i=0;string[i]!=0;i++);					// Go to end
	len=i;
	for(i=len;i>=0 && string[i]!='.';i--);		// Skip extension if any
	if(i==0 && string[0]!='.')					// No dot
		i=len;
	
	while(i>=0)									// Rewind
	{
		if(isdigit(string[i]))
			numbers++;
		i--;
	}

	return(numbers);
}

/*
 * Returns the amount of non-number chars in a string
 * before the last dot.
 *
 */
int p_checkforother(char *string)
{
	int i=0,letters=0;

	if(!string)
		return(0);

	while(string[i]!=0)				// Go to the end of string
		i++;

	while(i>=0 && string[i]!='.')	// Rewind until a dot or falloff
		i--;
	if(i>0)							// Skip the dot
		i--;
	
	while(i>=0)						// Look for letters
	{
		if(!isdigit(string[i]))		// Found a letter
			letters++;
		i--;
	}

	return(letters);
}

/*
 * Initializes some default values to p_misc table,
 * if they're not already set.
 *
 */
int p_initcfg(MYSQL *db)
{
	char *value;

// Filetypes? 
	
	value=p_getmisc(db, P_KEY_FILETYPES);
	if(!value)
		p_setmisc(db, P_KEY_FILETYPES, P_FILETYPES);

	value=p_getmisc(db, P_KEY_NOSPACE);
	if(!value)
		p_setmisc(db, P_KEY_NOSPACE, P_NOSPACE);

	value=p_getmisc(db, P_KEY_NEEDNUMBERS);
	if(!value)
		p_setmisc(db, P_KEY_NEEDNUMBERS, P_NEEDNUMBERS);

	value=p_getmisc(db, P_KEY_NEEDOTHERS);
	if(!value)
		p_setmisc(db, P_KEY_NEEDOTHERS, P_NEEDOTHERS);

	value=p_getmisc(db, P_KEY_STRICTMD5);
	if(!value)
		p_setmisc(db, P_KEY_STRICTMD5, P_STRICTMD5);

	value=p_getmisc(db, P_KEY_TOLOWERCASE);
	if(!value)
		p_setmisc(db, P_KEY_TOLOWERCASE, P_TOLOWERCASE);

	value=p_getmisc(db, P_KEY_BOWSERSINCE);
	if(!value)
		p_setmisc(db, P_KEY_BOWSERSINCE, P_BOWSER_DEFAULT_SINCE);

	value=p_getmisc(db, P_KEY_NEWSWAITAFTER);
	if(!value)
		p_setmisc(db, P_KEY_NEWSWAITAFTER, P_NEWS_WAIT_AFTER);

	value=p_getmisc(db, P_KEY_NEWSWAITSECS);
	if(!value)
		p_setmisc(db, P_KEY_NEWSWAITSECS, P_NEWS_WAIT_SECS);
	
	value=p_getmisc(db, P_KEY_VIEWER);
	if(!value)
		p_setmisc(db, P_KEY_VIEWER, P_VIEWER);

	value=p_getmisc(db, P_KEY_MINASSNAMELEN);
	if(!value)
		p_setmisc(db, P_KEY_MINASSNAMELEN, P_MIN_ASS_NAMELENGTH);

	value=p_getmisc(db, P_KEY_TMPDIR);
	if(!value)
		p_setmisc(db, P_KEY_TMPDIR, P_TMPDIR);
		
	return(0);
}

/*
 * Creates an "assign pattern" from a filename. 
 *
 * All numbers in filename will be replaced with "0".
 * The letters after the last encountered number and before
 * the last dot (".") are thought as indexing if there's
 * no more than two of them, and in that case replaced with "1". 
 *
 * e.g. "ab-013-f.jpg" => "ab-000-1.jpg"
 *      "ab-029-x.jpg" => "ab-000-1.jpg"
 *      "ab-ccc-1.jpg" => "ab-ccc-0.jpg"
 *      "ab-1-1ab.jpg" => "ab-0-011.jpg
 *      "ab-1-def.jpg" => "ab-0-def.jpg"
 *      "ab-1b-22.jpg" => "ab-0b-00.jpg"
 *
 * This is of course not a foolproof method, but then, what is? =)
 *
 * Output will be written to "output". Returns the
 * number of characters before the first dot.
 *
 */
int p_makepattern(char *output, char *filename)
{
	int i=0, dotfound=0;
	int letters=0;

	if(!filename || !output)
		return(0);

	while(filename[i]!=0)			// First pass forward, numbers => 0
	{
		if(isdigit(filename[i]))
			output[i]='0';
		else
			output[i]=filename[i];

		if(filename[i]=='.')		// Set new last dot location
			dotfound=i;

		i++;
	}
	output[i]=0;
	
	i=dotfound-1;						// Location of last dot
	
	while(i>=0 && isalpha(output[i]))	// Count letters
	{
		letters++;
		i--;
	}
	if(letters<=2)					// Second pass backwards, letters => 1
	{
		i=dotfound-1;

		while(i>0)
		{
			if(isalpha(output[i]))
				output[i]='1';
			else
				break;
		
			i--;
		}
	}
	
	return(dotfound);
}

/*
 * Converts a string to lowercase
 *
 */
int p_strtolc(char *string)
{
	int i=0;

	if(!string)
		return(0);

	while(string[i]!=0)
	{
		string[i]=tolower(string[i]);
		i++;
	}

	return(1);
}

/*
 * Connect to the PIMPPA database.
 *
 */
MYSQL *p_connect(void)
{
	MYSQL *db;
	
	db=mysql_init(NULL);
	if(!db)
	{
		fprintf(stderr, "mysql_init() failure\n");
		return(NULL);
	}

	mysql_options(db, MYSQL_READ_DEFAULT_GROUP, "pimppa");

	mysql_real_connect(db,NULL,NULL,NULL,P_DBASE,0,NULL,0);
	if(mysql_error(db)[0])
	{
		fprintf(stderr, "%s\n", mysql_error(db));
		return(NULL);
	}
	/*
	mysql_select_db(db, P_DBASE);
	if(mysql_error(db)[0])
	{
		fprintf(stderr, "%s\n", mysql_error(db));
		mysql_close(db);
		return(0);
	}
	*/
	
	return(db);
}

/*
 * Internal "cp" command to copy files from directory to another.
 * Probably inefficient as hell.
 *
 */
int p_cp(char *src, char *dest)
{
	char buffer[P_CP_BUFFSIZE];
	char truedest[PATH_MAX];
	struct utimbuf origtimes;
	struct stat st;
	int oldfd, newfd, cnt;

	if(!src || !dest)
	{
		fprintf(stderr, "p_cp(): src or dest is null\n");
		return(0);
	}

	if(!src[0] || !dest[0])
	{
		fprintf(stderr, "p_cp(): src[0] or dest[0] is null!\n");
		return(0);
	}
	
	oldfd=open(src, O_RDONLY);
	if(oldfd==-1)
	{
		fprintf(stderr, "p_cp(): Unable to open %s\n", src);
		return(0);
	}
	
	if(fstat(oldfd, &st)!=0)
	{
		fprintf(stderr, "p_cp(): Unable to stat oldfd!\n");
		close(oldfd);
		return(0);
	}

	origtimes.actime=st.st_atime;
	origtimes.modtime=st.st_mtime;

	// Check if destination is a directory
	if(stat(dest, &st)==0 && S_ISDIR(st.st_mode))
	{
//		printf("Basename: %s\n", p_basename(src));
	
		if(!p_checkp(dest))
			sprintf(truedest, "%s/%s", dest, p_basename(src));
		else
			sprintf(truedest, "%s%s", dest, p_basename(src));
	}
	else
		strcpy(truedest, dest);

//	printf("newdest: %s\n", truedest);

	newfd=open(truedest, O_WRONLY|O_CREAT,0644);
	if(newfd==-1)
	{
		fprintf(stderr, "p_mv(): Unable to open %s\n", truedest);
		close(oldfd);
		return(0);
	}

	while((cnt=read(oldfd, buffer, P_CP_BUFFSIZE)))
	{
		write(newfd, buffer, cnt);
	}
	
	close(newfd);
	utime(truedest, &origtimes);	// Restore times
	
	close(oldfd);

	return(1);
}

/*
 * Internal "mv" command. Relies on the "p_cp". 
 *
 */
int p_mv(char *src, char *dest)
{
	if(rename(src,dest)==-1)		// Try this first
	{
		if(!p_cp(src, dest))
			return(0);
	
		unlink(src);
	}

	return(1);
}

/*
 * Returns the filename component of a full path
 *
 */
char *p_basename(char *src)
{
	int i;

	if(!src || !src[0])
		return(NULL);

	for(i=strlen(src);i>=0;i--)
	{
		if(src[i]=='/')
			return(&src[i+1]);
	}

	return(NULL);
}

/* 
 * Returns the first component in "path" in "head"
 * and the rest in "tail". 
 *
 */
int p_pathcomp(char *path, char *head, char *tail)
{
	int i=0;

	if(!path || !path[0])
		return(0);

	if(!head || !tail)	
		return(0);
		
	while(path[i]!=':')
	{
		head[i]=path[i];
		i++;
	}
	head[i]=0;

	strcpy(tail, &path[i+1]);

	return(1);
}

/*
 * Gets the "misc_data" field from "p_misc" for given "key".
 *
 */
char *p_getmisc(MYSQL *db, char *key)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	char key_esc[256];
	static char result[1024];
	int retval=0;

	result[0]=0;

	if(strlen(key)>32)
		return(NULL);
		
	mysql_escape_string(key_esc, key, strlen(key));

	p_query(db, "SELECT misc_data FROM p_misc WHERE misc_key='%s'",
			key_esc);
	sql_res=mysql_store_result(db);
	if(sql_res)
	{
		if((sql_row=mysql_fetch_row(sql_res)))
		{
			strcpy(result, sql_row[0]);
			retval=1;
		}
		mysql_free_result(sql_res);	
	}

	if(retval)
		return(result);
	else
		return(NULL);
}

/*
 * Sets the "misc_data" field from "p_misc" for given "key".
 *
 */
int p_setmisc(MYSQL *db, char *key, char *data)
{
	char key_esc[128+1];
	char data_esc[512+1];

	if(strlen(key)>32 || strlen(data)>256)
		return(0);
		
	mysql_escape_string(data_esc, data, strlen(data));
	mysql_escape_string(key_esc, key, strlen(key));

	p_query(db, "REPLACE INTO p_misc (misc_key,misc_data) "
		    "VALUES ('%s','%s')",
	     	    key_esc,data_esc);
	
	return(1);
}

/*
 * Performs a database query. Returns 0 on failure, 1 on success.
 *
 */
int p_query(MYSQL *db, const char * format,...)
{
	va_list args;
	char query[QUERY_MAX];

	va_start(args, format);
	vsnprintf(query, QUERY_MAX, format, args);
	va_end(args);

//	fprintf(stderr, "DEBUG: %s\n", query);

	mysql_query(db, query);
	if(mysql_error(db)[0])
	{
		fprintf(stderr, "SQL error %d: %s\n", mysql_errno(db), mysql_error(db));
		if(mysql_errno(db)!=ER_PARSE_ERROR)
		{
			fprintf(stderr, "PIMPPA safety measure: Suicide. :(\n");
			exit(42);
		}
		return(0);
	}
		
	return(1);
}

/*
 * Return 1 if *path ends in '/', 0 otherwise
 *
 */
int p_checkp(char *path)
{
	if(!path)
		return(0);

	if(path[strlen(path)-1]=='/')
		return(1);
	else
		return(0);
}

/*
 * Checks if a given md5sum exists in database for some file
 * 
 */
int p_md5check(char *md5sum)
{
	MYSQL *db;
	MYSQL_RES *sql_res;
	char md5sum_esc[16*2+1];
	
	db=p_connect();
	if(!db)
		return(0);

	mysql_escape_string(md5sum_esc, md5sum, 16);
	
	p_query(db, "SELECT 1 FROM p_files "
		    "WHERE file_md5sum='%s'",
	  	    md5sum_esc);
	sql_res=mysql_store_result(db);
	if(sql_res)
	{
		if(mysql_num_rows(sql_res)>0)
		{
			mysql_free_result(sql_res);
			mysql_close(db);
	
			return(1);
		}
		mysql_free_result(sql_res);
	}

	mysql_close(db);

	return(0);
}
	
/*
 * Replacement for system(). Goes through $PATH.
 *
 */
/*
int p_exec(char *cmd, char *args[])
{
	pid_t mypid;

	mypid=fork();
	if(mypid==0)
	{
		char head[10*PATH_MAX];
		char tail[10*PATH_MAX];
		char *ptr=getenv("PATH");

		p_pathcomp(ptr, head, tail);
		sprintf(args[0], "%s/%s", head, cmd);

		while(execv(args[0], args)==-1)
		{
			if(!p_pathcomp(tail, head, tail))
				break;
				
			sprintf(args[0], "%s/%s", head, cmd);
		}
		exit(-1);
	}
	return(mypid);
}
*/

/*
 * Checks if a filename is valid in the current settings
 *
 * 1 == ok, 0 == not ok
 *
 */
int p_checkfilename(MYSQL *db, char *filename)
{
	static int neednumbers, needothers, nospace;
	static int loaded=0;

	// Load the preferences first, if not loaded
	if(!loaded)
	{
		char *value;

	    value=p_getmisc(db, P_KEY_NOSPACE);
	    if(value)
	        nospace=atoi(value);
	    else
	    {
        	nospace=atoi(P_NOSPACE);
	        p_setmisc(db, P_KEY_NOSPACE, P_NOSPACE);
	    }

	    value=p_getmisc(db, P_KEY_NEEDNUMBERS);
	    if(value)
	        neednumbers=atoi(value);
	    else
	    {
        	neednumbers=atoi(P_NEEDNUMBERS);
	        p_setmisc(db, P_KEY_NEEDNUMBERS, P_NEEDNUMBERS);
	    }

	    value=p_getmisc(db, P_KEY_NEEDOTHERS);
	    if(value)
	        needothers=atoi(value);
	    else
	    {
        	needothers=atoi(P_NEEDOTHERS);
	        p_setmisc(db, P_KEY_NEEDOTHERS, P_NEEDOTHERS);
	    }
		loaded=1;
	}
	
	if(neednumbers)
	{
   		if(p_checkfornumber(filename)<neednumbers) // Not enough numbers!
			return(0);
    }
	
	if(needothers)          // How about a letter or similar?
	{
		if(p_checkforother(filename)<needothers) // Not enough!
			return(0);
	}

	if(nospace)
	{
		if(p_checkforspace(filename)) // It has a space!
			return(0);
	}

	return(1);
}

/*
 * Tries to find destination area for filename based on 
 * rules and assign patterns
 *
 * Returns: 0 if not known, else the presumed area
 *
 */
int p_getdest(char *filename, int context)
{
	MYSQL *db;
	MYSQL_ROW sql_row;
	MYSQL_RES *sql_res;
	char pattern[P_LEN_FILE_NAME+1];
	char escaped_fn[P_LEN_FILE_NAME*2+1];
	char *value;
	int minassnamelen;
	int destarea_id=0;
	
	db=p_connect();
	if(!db)
		return(0);

	// Check for assign
	value=p_getmisc(db, P_KEY_MINASSNAMELEN);
	if(value)
		minassnamelen=atoi(value);
	else
		minassnamelen=atoi(P_MIN_ASS_NAMELENGTH);

	if(p_makepattern(pattern,filename)>=minassnamelen)
	{
		char escaped_pat[2*P_LEN_FILE_NAME+1];

		mysql_escape_string(escaped_pat, pattern, strlen(pattern));
        
        p_query(db, "SELECT a_dest,a_context FROM p_assign,p_areas "
                    "WHERE a_pattern='%s' AND (a_context=%d OR a_context=0) "
      		    "  AND a_dest=area_id " 
    		    "  AND NOT ((area_flags & %ld) OR (area_flags & %ld)) "
        	    "ORDER BY a_context DESC", 
		 escaped_pat, context,
		 AREA_INCOMING, AREA_NOASSIGN);
        sql_res=mysql_store_result(db);
        if(sql_res)
        {
           if((sql_row=mysql_fetch_row(sql_res)))
              destarea_id=atoi(sql_row[0]);
           mysql_free_result(sql_res);
           if(destarea_id!=0) {
	         mysql_close(db);
	         return(destarea_id);
           }
      	}
	}
	
	// Check for a specific rule
	mysql_escape_string(escaped_fn, filename, strlen(filename));
	p_query(db, "SELECT r_target FROM p_rules " 
		    "WHERE (r_context=%d OR r_context=0) "
		    "  AND r_type=%d "
			"  AND '%s' REGEXP r_rule "
		    "ORDER BY r_context DESC",
		context, 
		RULE_FILENAME,
		escaped_fn);
	sql_res=mysql_store_result(db);
	if(sql_res)
	{
		if((sql_row=mysql_fetch_row(sql_res)))
		{
			destarea_id=atoi(sql_row[0]);
		}
		mysql_free_result(sql_res);
		
		if(destarea_id!=0) {
			mysql_close(db);
			return(destarea_id);
		}
	}

	mysql_close(db);
	return(0);
}

char *Results[]={"File ok",
    "Filename length is larger than allowed",
    "Invalid filename",
    "File exists",
    "Assigns tell to delete",
    "Nazi tells to delete (assign unknown)"};

/** 
 * Checks subject contains a known key pattern from p_rules
 *
 * Returns: target area id, or 0 if not known
 *
 **/
int p_checksubject(MYSQL *db, int areaid, char * subject)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	char * escaped_subject;
	int retval=0;
	int context=0;

    if(subject==NULL || subject[0]==0)
	  return 0;

    escaped_subject = malloc(strlen(subject)*2+1);

	p_query(db, "SELECT area_context "
		    "FROM p_areas WHERE area_id=%d",
    		areaid);
	sql_res=mysql_store_result(db);
	if(sql_res) {
		if((sql_row=mysql_fetch_row(sql_res))) {
			context=atoi(sql_row[0]);
		}
		mysql_free_result(sql_res);
	}

    mysql_escape_string(escaped_subject, subject, strlen(subject));
	p_query(db, "SELECT r_target FROM p_rules "
	            "WHERE (r_context=%d OR r_context=0) "
				"  AND r_type = %d "
				"  AND '%s' REGEXP r_rule",
				context,
				RULE_KEYWORD,
				escaped_subject);
    sql_res=mysql_store_result(db);
	if(sql_res) {
	  if((sql_row = mysql_fetch_row(sql_res))) 
	    retval=atoi(sql_row[0]);
	  mysql_free_result(sql_res);
	}

	free(escaped_subject);

	return(retval);
}

/*
 * Checks if the given file 'filename' is valid for 'areaid',
 * including filename check, duplicate check and assign/rule check.
 *
 */
int p_checkfile(MYSQL *db, int areaid, char *filename, unsigned long options)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	int dval;
	int context=0;
	
	if(strlen(filename)>=P_LEN_FILE_NAME)
		return(RESULT_LONG_FILENAME);
	
	if(!p_checkfilename(db, filename))
		return(RESULT_INVALID_FILENAME);

	dval=p_dupecheck(db, filename, areaid);
	if(dval>=0 && dval!=INTEG_FAILED)
		return(RESULT_FILE_EXISTS);

	p_query(db, "SELECT area_context "
		    "FROM p_areas WHERE area_id=%d",
    		areaid);
	sql_res=mysql_store_result(db);
	if(sql_res)
	{
		if((sql_row=mysql_fetch_row(sql_res)))
		{
			context=atoi(sql_row[0]);
		}
		mysql_free_result(sql_res);
	}
	
	dval=p_getdest(filename,context);
	if(dval<0)
		return(RESULT_ASSIGN_DELETE);
	
	if((options & OPT_NAZI) && dval==0)
		return(RESULT_NAZI_KILL);

	return(RESULT_OK);
}

int getlastcomp(char *string);
int dupecheck_single(MYSQL *src_db, char *escaped_fn, int area_id);
int dupecheck_regexp(MYSQL *src_db, char *escaped_fn, char *regexp);
int dupecheck(MYSQL *src_db, char *filename, int area_id);

/*
 * Returns the last component after a dot in a string with the format
 *
 * blaa.bloo.blii.bluup  == bluup 
 *
 */
int getlastcomp(char *string)
{
	int i;

	i=strlen(string);
	while(i>0 && string[i]!='.')
		i--;

	if(i>0)
		return(i+1);
	else
		return(0);
}

/*
 * Returns file integrity value if file exists on "area_id", 
 * or on any area that is defined in p_areas -table as that 
 * areas immediate target areas.
 *
 * -1 is returned if the file was not found
 *
 */
int p_dupecheck(MYSQL *src_db, char *filename, int area_id)
{
	MYSQL_ROW sql_row;
	MYSQL_RES *sql_res;
	int retval=-1;

	char transname[P_LEN_FILE_NAME+1];
	char *transptr;

/***** Check normal filename ****************/
	
	retval=dupecheck(src_db, filename, area_id);
	if(retval>=0)
		return(retval);

/***** Dupecheck transformed filename *******/

	strcpy(transname, filename);
	transptr=strrchr(transname, '.');		// Next char from '.'
	if(transptr)
	{
		char ext[PATH_MAX];
		char escaped_transn[2*P_LEN_FILE_NAME+1];

		transptr++;
		ext[0]=0;
		
		mysql_escape_string(escaped_transn, transptr, strlen(transptr));

		p_query(src_db, "SELECT type_transto "
				"FROM p_types "
				"WHERE type_ext='%s' "
				"  AND type_transcmd<>'' "
				"  AND type_transto<>''",
			escaped_transn);
		if(mysql_error(src_db)[0])
		{
			fprintf(stderr, "Err: %s\n", mysql_error(src_db));
			return(retval);
		}
		sql_res=mysql_store_result(src_db);
		if(sql_res)
		{
			if((sql_row=mysql_fetch_row(sql_res)))
				strcpy(ext, sql_row[0]);
				
			mysql_free_result(sql_res);
		}

		if(ext[0])
		{
			strcpy(transptr, ext);				// Attach new extension
	
			if(strcmp(transname, filename)!=0)  // Name changed?
			{
//				printf("	=> %s\n", transname);

				retval=dupecheck(src_db, transname, area_id);
				if(retval>=0)								// Exists
					return(retval);
			}
		}
	}

	return(retval);
}

int dupecheck(MYSQL *src_db, char *filename, int area_id)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	char escaped_fn[PATH_MAX];
	int retval=-1;

	mysql_escape_string(escaped_fn, filename, strlen(filename));

	if(area_id==-1)						// Check from all areas
	{
//		printf("	DEBUG: Dupechecking %s from all...\n", filename);

		p_query(src_db, "SELECT file_integ FROM p_files " 
				"WHERE file_name='%s' ", 
			escaped_fn);
		sql_res=mysql_store_result(src_db);
		if(sql_res)
		{
			if(mysql_num_rows(sql_res)>0)		// Exists
			{
				sql_row=mysql_fetch_row(sql_res);
				retval=atoi(sql_row[0]);
				mysql_free_result(sql_res);
				
				return(retval);
			}

			mysql_free_result(sql_res);
		}
		return(retval);
	}

/************ Specific area and its targets *****************/

	retval=dupecheck_single(src_db, escaped_fn, area_id);
	if(retval>=0)
		return(retval);

	p_query(src_db, "SELECT area_targets FROM p_areas "
		        "WHERE area_id=%d AND (area_flags & %ld)",
		area_id, AREA_INCOMING);
	sql_res=mysql_store_result(src_db);
	if(sql_res)	
	{
		if(mysql_num_rows(sql_res)<=0)
		{
			mysql_free_result(sql_res);
			return(retval);
		}
		
		sql_row=mysql_fetch_row(sql_res);

		retval=dupecheck_regexp(src_db, escaped_fn, sql_row[0]);
			
		mysql_free_result(sql_res);
		
		if(retval>=0)
			return(retval);
	}

	return(retval);
}

/*
 * Dupechecks from a single area, no target areas noted
 *
 */
int dupecheck_single(MYSQL *src_db, char *escaped_fn, int area_id)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	int retval=-1;

//	printf("	DEBUG: Dupechecking %s from %d\n", escaped_fn, area_id);

//	printf("SELECT 1 FROM p_files
	p_query(src_db, "SELECT file_integ FROM p_files "
		        "WHERE file_name='%s' AND file_area=%d",
	      escaped_fn, area_id);
	sql_res=mysql_store_result(src_db);
	if(sql_res)	
	{
		if(mysql_num_rows(sql_res)>0)		// Exists
		{
			sql_row=mysql_fetch_row(sql_res);
			retval=atoi(sql_row[0]);
			
			mysql_free_result(sql_res);
			
			return(retval);
		}
		mysql_free_result(sql_res);
	}

	return(retval);
}

/*
 * Dupechecks from regexp matching areas
 *
 */
int dupecheck_regexp(MYSQL *src_db, char *escaped_fn, char *regexp)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	int retval=-1;

//	printf("	DEBUG: RegExpDChecking %s from %s\n", escaped_fn, regexp);

	p_query(src_db, "SELECT file_integ FROM p_files "
	 	        "WHERE file_name='%s' "
		 	"  AND file_area RLIKE '%s'",
	        escaped_fn, regexp);
	sql_res=mysql_store_result(src_db);
	if(sql_res)	
	{
		if(mysql_num_rows(sql_res)<=0)		// Exists
		{
			mysql_free_result(sql_res);
			return(retval);
		}

		sql_row=mysql_fetch_row(sql_res);
		retval=atoi(sql_row[0]);
		
		mysql_free_result(sql_res);
	
		return(retval);
	}

	return(retval);
}

