/***************************************************************************
 *   Copyright (C) 2007-2008 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "custom_prog.h"

#include "devices/base/device_group.h"
#include "devices/list/device_list.h"
#include "libgui/main_global.h"
#include "libgui/project_manager.h"

//-----------------------------------------------------------------------------
const CustomProgrammer::Action::Data CustomProgrammer::Action::DATA[Nb_Types] = {
  { "read",        I18N_NOOP("Read")        },
  { "erase",       I18N_NOOP("Erase")       },
  { "program",     I18N_NOOP("Program")     },
  { "verify",      I18N_NOOP("Verify")      },
  { "blank_check", I18N_NOOP("Blank Check") },
  { "run",         I18N_NOOP("Run")         },
  { "stop",        I18N_NOOP("Stop")        }
};

//-----------------------------------------------------------------------------
QString CustomProgrammer::Config::command(Action action)
{
  Config config;
  return config.readEntry(action.key(), QString::null);
}

void CustomProgrammer::Config::writeCommand(Action action, const QString &command)
{
  Config config;
  config.writeEntry(action.key(), command);
}

//-----------------------------------------------------------------------------
CustomProgrammer::ProcessManager::ProcessManager(::CustomProgrammer::Base& base)
: Log::Base(&base), _base(base), _process(0)
{
}

void CustomProgrammer::ProcessManager::logLine(::Process::OutputType type, const QString &s)
{
  Log::LineType ltype = (type==::Process::OutputType::Stderr ? Log::LineType::SoftError : Log::LineType::Normal);
  log(ltype, s, Log::Delayed); // Delayed => critical!!
}

bool CustomProgrammer::ProcessManager::execute(const QString &command)
{
  delete _process;
  _process = new ::Process::LineSignal;
  _process->setUseShell(true);
  connect(_process, SIGNAL(logLine(::Process::OutputType, const QString &)), SLOT(logLine(::Process::OutputType, const QString &)));
  connect(_process, SIGNAL(done(int)), SLOT(done(int)));
  _process->setup(command, QStringList(), false);
  PURL::Url url = Main::projectManager().projectUrl();
  if ( !url.isEmpty() ) {
    _process->setWorkingDirectory(url.directory());
  }
  log(Log::LineType::Command, _process->arguments().join(" "));
  if (!_process->start(0)) { // no timeout
    log(Log::LineType::Error, i18n("*** Error executing command ***"));
    return false;
  }
  _base.setState(::Programmer::Running);
  return true;
}

void CustomProgrammer::ProcessManager::done(int code)
{
  if ( code == 0 ) {
    log(Log::LineType::Information, i18n("*** Success ***"), Log::Delayed);
  } else {
    log(Log::LineType::Error, i18n("*** Exited with status: %1 ***").arg(code), Log::Delayed);
  }
  _base.setState(::Programmer::Stopped);
}

void CustomProgrammer::ProcessManager::abort()
{
  log(Log::LineType::Error, i18n("*** Aborted ***"), Log::Delayed);
  _base.setState(::Programmer::Stopped);
  delete _process;
  _process = 0;
}

//-----------------------------------------------------------------------------
CustomProgrammer::Base::Base(const ::Programmer::Group &group, const Device::Data *data)
 : Programmer::Generic(group, data), _manager(*this)
{
}

bool CustomProgrammer::Base::execute(Action action, Device::Memory *memory, const Device::MemoryRange *range)
{
  Config config;
  QString cmd = config.command(action).stripWhiteSpace();
  if (cmd.isEmpty()) {
    log(Log::LineType::Error, i18n("There is no command specified for the \"%1\" action.").arg(action.label()));
    return false;
  }
  return _manager.execute(cmd);
}

//-----------------------------------------------------------------------------
void CustomProgrammer::Group::initSupported()
{
  QStringList names = Device::lister().supportedDevices();
  QStringList::const_iterator it;
  for (it=names.begin(); it!=names.end(); ++it) addDevice(*it, Device::lister().data(*it), ::Group::Support::Tested);
}
