#!/usr/bin/env python

from distutils.core import setup, Command
from distutils.dist import Distribution
from distutils.command.install import install
from distutils.command.build import build
from distutils.command.clean import clean
from distutils.dir_util import remove_tree
import glob
import sys
import os.path
from picard import APPNAME, APPVERSION

args = {}

def generate_file(infilename, outfilename, variables):
    f = file(infilename, "rt")
    content = f.read()
    f.close()
    content = content % variables
    f = file(outfilename, "wt")
    f.write(content)
    f.close()

class my_build(build):
    
    user_options = build.user_options + [
        ('build-locales=', 'd', "build directory for locale files"),
    ]

    sub_commands = build.sub_commands + [
        ('build_locales', None),
    ]
    
    def initialize_options(self):
        build.initialize_options(self)
        self.build_locales = None
    
    def finalize_options(self):
        build.finalize_options(self)
        if self.build_locales is None:
            self.build_locales = os.path.join(self.build_base, 'locale')

class build_locales(Command):
    
    description = 'build locale files'
    
    user_options = [
        ('build-dir=', 'd', "directory to build to"),
    ]

    def initialize_options(self):
        self.build_dir = None

    def finalize_options (self):
        self.set_undefined_options('build',
                ('build_locales', 'build_dir'))
        self.locales = self.distribution.locales

    def run(self):
        for domain, locale, po in self.locales:
            path = os.path.join(self.build_dir, locale, 'LC_MESSAGES')
            mo = os.path.join(path, '%s.mo' % domain)
            self.mkpath(path)
            self.spawn(['msgfmt', '-o', mo, po])
    
class my_clean(clean):
    
    def run(self):
        clean.run(self)
        self.run_command('clean_locales')
        
class clean_locales(Command):
    
    description = 'clean locale files'

    user_options = [
        ('build-dir=', 'd', "build directory for locales"),
    ]
    
    def initialize_options(self):
        self.build_dir = None

    def finalize_options (self):
        self.set_undefined_options('build_locales', ('build_dir', 'build_dir'))

    def run(self):
        if os.path.exists(self.build_dir):
            remove_tree(self.build_dir, dry_run=self.dry_run)
            
class install_locales(Command):

    description = "install locale files"
    
    user_options = [
        ('install-dir=', 'd', "directory to install locale files to"),
        ('build-dir=','b', "build directory (where to install from)"),
        ('force', 'f', "force installation (overwrite existing files)"),
        ('skip-build', None, "skip the build steps"),
    ]

    boolean_options = ['force', 'skip-build']
        
    def initialize_options(self):
        self.install_dir = None
        self.build_dir = None
        self.force = 0
        self.skip_build = None
        self.outfiles = []

    def finalize_options(self):
        self.set_undefined_options('build', ('build_locales', 'build_dir'))
        self.set_undefined_options('install',
                                   ('install_locales', 'install_dir'),
                                   ('force', 'force'),
                                   ('skip_build', 'skip_build'),
                                  )

    def run(self):
        if not self.skip_build:
            self.run_command('build_locales')
        self.outfiles = self.copy_tree(self.build_dir, self.install_dir)
            
    def get_inputs(self):
        return self.locales or []

    def get_outputs(self):
        return self.outfiles

class my_install(install):
    
    user_options = install.user_options + [
        ('install-locales=', None,
         "installation directory for locales"),
    ]
    
    sub_commands = install.sub_commands + [
        ('install_locales', None),
    ]
    
    def initialize_options(self):
        install.initialize_options(self)
        self.install_locales = None
        self.localedir = None
    
    def finalize_options(self):
        install.finalize_options(self)
        if self.install_locales is None:
            self.install_locales = '$base/share/locale'
            self._expand_attrs(['install_locales'])
        self.install_locales = os.path.normpath(self.install_locales)
        self.localedir = self.install_locales
        if self.root is not None:
            self.change_roots('locales')
        
    def run(self):
        generate_file('scripts/picard.in', 'scripts/picard', {'localedir': self.localedir})
        install.run(self)
        
args['cmdclass'] = {
    'build': my_build,
    'build_locales': build_locales,
    'clean': my_clean,
    'clean_locales': clean_locales,
    'install': my_install,
    'install_locales': install_locales,
    }

Distribution.locales = None

try:
    from py2exe.build_exe import py2exe
except:
    py2exe = None
else:
    class bdist_nsis(py2exe):
        def run(self):
            self.distribution.data_files = [(a.replace('share/locale', 'locale'), b) for (a,b) in self.distribution.data_files]
            generate_file('scripts/picard.py2exe.in', 'scripts/picard', {})
            self.distribution.data_files.append(("", glob.glob("*.dll")))
            self.distribution.data_files.append(("plugins", glob.glob("plugins/*.tpp")))
            
            py2exe.run(self)
            
            print "*** creating the NSIS setup script ***"
            pathname = "installer/picard-setup.nsi"
            generate_file(pathname + ".in", pathname, {'name': APPNAME, 'version': APPVERSION})
        
            print "*** compiling the NSIS setup script ***"
            import ctypes
            res = ctypes.windll.shell32.ShellExecuteA(0, "compile", pathname, None, None, 0)
            if res < 32:
                raise RuntimeError, "ShellExecute failed, error %d" % (res,)
    args['cmdclass']['bdist_nsis'] = bdist_nsis
    
    manifest = '''
<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<assembly xmlns="urn:schemas-microsoft-com:asm.v1" manifestVersion="1.0">
<assemblyIdentity
    version="0.0.0.0"
    processorArchitecture="x86"
    name="%(name)s"
    type="win32"
/>
<description>%(name)s</description>
<dependency>
    <dependentAssembly>
        <assemblyIdentity
            type="win32"
            name="Microsoft.Windows.Common-Controls"
            version="6.0.0.0"
            processorArchitecture="X86"
            publicKeyToken="6595b64144ccf1df"
            language="*"
        />
    </dependentAssembly>
</dependency>
</assembly>
''' % dict(name=APPNAME)
    
    args['windows'] = {
        'script': 'scripts/picard',
        'icon_resources': [(1, 'rc/picard.ico')],
        'other_resources': [(24, 1, manifest)]
        },
        
    args['options'] = {
        'py2exe': {'packages': ['encodings'], 'optimize': 2}, 
        'win32fil': {'packages': ['win32api']},
        'win32api': {'packages': ['win32api']},
        'win32con': {'packages': ['win32api']}
        }
        
args['locales'] = [('picard', os.path.split(po)[1][:-3], po) for po in glob.glob('po/*.po')]

data_files = []

if py2exe == None:
    data_files.append(('share/icons', ('rc/picard16.png', 'rc/picard32.png')))
    data_files.append(('share/applications', ('picard.desktop',)))

setup(
    name='picard',
    version=APPVERSION,
    description="The next generation MusicBrainz tagger",
    url="http://wiki.musicbrainz.org/PicardTagger",
    package_dir={'picard': 'picard'},
    packages=['picard', 'picard.ui', 'picard.ui.options', 'picard.browser'],
    scripts=['scripts/picard'],
    data_files=data_files,
    **args
    )

