# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK *****

import wx, sys, re, os
from picard import coverartcache, events, util, debug
from tunepimp import tunepimp, metadata, track
from options import optionsdialog
import albumpanel, images, toolbar, infopanel
from picard.browser import filelookup

localTrackEditId = wx.NewId()
localAlbumComboId = wx.NewId()
localArtistComboId = wx.NewId()
localTrackNumEditId = wx.NewId()
localDateEditId = wx.NewId()
localTrackNumSpinId = wx.NewId()
localLookupButtonId = wx.NewId()

serverLookupButtonId = wx.NewId()

class MetadataPanel(wx.Panel):

    curTrack = None
    hasChanged = False

    def __init__(self, parent, config, albumPanel, infoPanel):
        wx.Panel.__init__(self, parent, -1, wx.DefaultPosition, wx.DefaultSize, wx.SIMPLE_BORDER)
        self.frame = parent
        self.tunePimp = config.getTunePimp()
        self.config = config
        self.albumPanel = albumPanel
        self.infoPanel = infoPanel

        wx.EVT_BUTTON(self, localLookupButtonId, self.OnLocalLookupButton)
        wx.EVT_BUTTON(self, serverLookupButtonId, self.OnServerLookupButton)
        wx.EVT_TEXT_ENTER(self, localAlbumComboId, self.OnEnter)
        wx.EVT_TEXT_ENTER(self, localArtistComboId, self.OnEnter)
        wx.EVT_TEXT_ENTER(self, localTrackEditId, self.OnEnter)
        wx.EVT_SPINCTRL(self, localTrackNumSpinId, self.OnTrackNumChanged)
        wx.EVT_TEXT(self, localDateEditId, self.OnChanged)
        wx.EVT_TEXT(self, localAlbumComboId, self.OnChanged)
        wx.EVT_TEXT(self, localArtistComboId, self.OnChanged)
        wx.EVT_TEXT(self, localTrackEditId, self.OnChanged)

        # Setup the notebook and button panel -----------------------------------------------
        self.nb = wx.Notebook(self, -1, wx.Point(0,0), wx.Size(0,0))
        #self.nbs = wx.NotebookSizer(self.nb)
        self.localPanel = wx.Panel(self.nb, -1, style = wx.NO_BORDER | wx.TAB_TRAVERSAL)
        self.nb.AddPage(self.localPanel, _("Local Metadata"))
        self.serverPanel = wx.Panel(self.nb, -1, style = wx.NO_BORDER | wx.TAB_TRAVERSAL)
        self.nb.AddPage(self.serverPanel, _("MusicBrainz Metadata"))

        # Create all the controls to set the tab order properly ----------------------------
        sizer = wx.BoxSizer(wx.VERTICAL)

        # local controls
        self.localArtistCombo = wx.TextCtrl(self.localPanel, -1, "", wx.DefaultPosition, 
                                      wx.DefaultSize, style=wx.TE_PROCESS_ENTER)
        self.localAlbumCombo = wx.TextCtrl(self.localPanel, -1, "", wx.DefaultPosition, 
                                      wx.DefaultSize, style=wx.TE_PROCESS_ENTER)
        
        #self.localArtistCombo = wx.ComboBox(self.localPanel, localArtistComboId, "", wx.DefaultPosition, 
        #                              wx.DefaultSize, ['Recent artists will eventually be listed here.'], 
        #                              wx.CB_DROPDOWN | wx.TE_PROCESS_ENTER)
        #self.localAlbumCombo = wx.ComboBox(self.localPanel, localAlbumComboId, "", wx.DefaultPosition, 
        #                              wx.DefaultSize, ['Recent albums will eventually be listed here.'], 
        #                              wx.CB_DROPDOWN | wx.TE_PROCESS_ENTER)
        self.localTrackEdit = wx.TextCtrl(self.localPanel, localTrackEditId, style=wx.TE_PROCESS_ENTER)

        self.localTrackNumSpin = wx.SpinCtrl(self.localPanel, localTrackNumSpinId, "", 
                                         min=1, max=99, size=(50, -1), style = wx.TE_PROCESS_ENTER)
        self.localDurationEdit = wx.TextCtrl(self.localPanel, -1, "", size=(50, -1), style=wx.TE_READONLY)
        self.localDateEdit = wx.TextCtrl(self.localPanel, localDateEditId, "", size=(80, -1))
        self.localLookupButton = wx.Button(self.localPanel, localLookupButtonId, _("Lookup"))

        # Server controls
        self.serverArtistEdit = wx.TextCtrl(self.serverPanel, -1, "", wx.DefaultPosition, 
                                      wx.DefaultSize, style=wx.TE_READONLY)
        self.serverAlbumEdit = wx.TextCtrl(self.serverPanel, -1, "", wx.DefaultPosition, 
                                      wx.DefaultSize, style=wx.TE_READONLY)
        self.serverTrackEdit = wx.TextCtrl(self.serverPanel, -1, style=wx.TE_READONLY)
        self.serverTrackNumEdit = wx.TextCtrl(self.serverPanel, -1, "", size=(50, -1), style=wx.TE_READONLY)
        self.serverDurationEdit = wx.TextCtrl(self.serverPanel, -1, "", size=(50, -1), style=wx.TE_READONLY)
        self.serverDateEdit = wx.TextCtrl(self.serverPanel, -1, "", size=(80, -1), style=wx.TE_READONLY)
        self.serverLookupButton = wx.Button(self.serverPanel, serverLookupButtonId, _("Lookup"))

        # Setup the local notebook page -----------------------------------------------------

        hSizer = wx.BoxSizer(wx.HORIZONTAL)
        hSizer.Add(self.localTrackNumSpin, 1, 0)
        hSizer.Add(wx.StaticText(self.localPanel, -1, _("Time") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 3)
        hSizer.Add(self.localDurationEdit, 1, 0)
        hSizer.Add(wx.StaticText(self.localPanel, -1, _("Release Date") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 3)
        hSizer.Add(self.localDateEdit, 1, 0)
        hSizer.Add(self.localLookupButton, 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 10)

        vSizer = wx.FlexGridSizer(4, 2, 3, 3)
        vSizer.AddMany([
            (wx.StaticText(self.localPanel, -1, _("Artist") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (self.localArtistCombo, 0, wx.EXPAND),
            (wx.StaticText(self.localPanel, -1, _("Album") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (self.localAlbumCombo, 0, wx.EXPAND),
            (wx.StaticText(self.localPanel, -1, _("Track") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (self.localTrackEdit, 0, wx.EXPAND),
            (wx.StaticText(self.localPanel, -1, _("Track Num") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (hSizer, 1, wx.EXPAND)
        ])
        vSizer.AddGrowableCol(1)
        sizer.Add(vSizer, 0, wx.EXPAND | wx.ALL, border=3)

        self.localPanel.SetSizer(sizer)
        self.localPanel.SetAutoLayout(True)
        self.localPanel.Fit()

        # Setup the server notebook page -----------------------------------------------------

        sizer = wx.BoxSizer(wx.VERTICAL)

        hSizer = wx.BoxSizer(wx.HORIZONTAL)
        hSizer.Add(self.serverTrackNumEdit, 1, 0)
        hSizer.Add(wx.StaticText(self.serverPanel, -1, _("Time") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 3)
        hSizer.Add(self.serverDurationEdit, 1, 0)
        hSizer.Add(wx.StaticText(self.serverPanel, -1, _("Release Date") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 3)
        hSizer.Add(self.serverDateEdit, 1, 0)
        hSizer.Add(self.serverLookupButton, 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 10)

        vSizer = wx.FlexGridSizer(4, 2, 3, 3)
        vSizer.AddMany([
            (wx.StaticText(self.serverPanel, -1, _("Artist") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (self.serverArtistEdit, 1, wx.EXPAND),
            (wx.StaticText(self.serverPanel, -1, _("Album") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (self.serverAlbumEdit, 1, wx.EXPAND),
            (wx.StaticText(self.serverPanel, -1, _("Track") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (self.serverTrackEdit, 1, wx.EXPAND),
            (wx.StaticText(self.serverPanel, -1, _("Track Num") + ":"), 0, wx.ALIGN_RIGHT | wx.ALIGN_CENTER_VERTICAL),
            (hSizer, 1, wx.EXPAND)
        ])
        vSizer.AddGrowableCol(1)
        sizer.Add(vSizer, 10, wx.EXPAND | wx.ALL, border=3)

        self.serverPanel.SetSizer(sizer)
        self.serverPanel.SetAutoLayout(True)
        self.serverPanel.Fit()

        # Set the main sizer ------------------------------------------------------------------

        main = wx.BoxSizer(wx.HORIZONTAL)
        main.Add(self.nb, 1, wx.ALL, 5)
        main.SetSizeHints(self)
        self.SetSizer(main)
        self.Fit()

        self.localLookupButton.Enable(False)
        self.serverLookupButton.Enable(False)
        self.enableLocalControls(False)
   
    def OnServerLookupButton(self, event):
        """The user has clicked the "Server" lookup button"""

        sel = self.albumPanel.getCurrentSelection()
        debug.debug("OnServerLookupButton: %s" % repr(sel), "mdata.lookup")
        if not len(sel):
           return
        sel = sel[0]

        lookup = filelookup.FileLookup(self.frame, self.config.settingServer, self.config.settingServerPort, 
                                       self.frame.getTaggerPort())

        if sel.__class__.__name__ == "Album":
            lookup.albumLookup(sel.getId())
        elif sel.__class__.__name__ == "Track":
            lookup.trackLookup(sel.getId())
        else:
            debug.debug("OnServerLookupButton unhandled: %s" % repr(sel), "mdata.lookup.unhandled")

    def OnLocalLookupButton(self, event):
        """The user has clicked the "Local" lookup button"""

        sel = self.albumPanel.getCurrentSelection()
        debug.debug("OnLocalLookupButton: %s" % repr(sel), "mdata.lookup")
        if not len(sel):
           return
        sel = sel[0]

        lookup = filelookup.FileLookup(self.frame, self.config.settingServer, self.config.settingServerPort, 
                                       self.frame.getTaggerPort())

        if sel.__class__.__name__ == "Album":
            if not sel.isSpecial():
                lookup.albumLookup(sel.getId())
       
        elif sel.__class__.__name__ == "Track":
            tpTrack = self.tunePimp.getTrack(sel.getFileId());
            tpTrack.lock()
            ldata = tpTrack.getLocalMetadata()
            puid = tpTrack.getPUID()
            fileName = tpTrack.getFileName()
            tpTrack.unlock()
            self.tunePimp.releaseTrack(tpTrack)

            lookup.tagLookup(self.localArtistCombo.GetValue(),
                             self.localAlbumCombo.GetValue(),
                             self.localTrackEdit.GetValue(),
                             self.localTrackNumSpin.GetValue(),
                             ldata.duration, fileName, puid)

        elif sel.__class__.__name__ == "TPFile":
            tpTrack = sel.getTrack()
            try:
                ldata = tpTrack.getLocalMetadata()
                puid = tpTrack.getPUID()
                fileName = tpTrack.getFileName()
            finally:
                sel.releaseTrack(tpTrack)

            lookup.tagLookup(self.localArtistCombo.GetValue(),
                             self.localAlbumCombo.GetValue(),
                             self.localTrackEdit.GetValue(),
                             self.localTrackNumSpin.GetValue(),
                             ldata.duration, fileName, puid)

        elif sel.__class__.__name__ == "ClusterAlbum":
            lookup.tagLookup(self.localArtistCombo.GetValue(),
                             self.localAlbumCombo.GetValue(), u"", 0, 0, u"", u"")

        else:
            debug.debug("OnLocalLookupButton unhandled: %s" % repr(sel), "mdata.lookup.unhandled")

    def OnChanged(self, event):
        """One of the local text fields has changed"""

        if self.curTrack:
            debug.debug("mdatapanel OnChanged", "mdata.event")
            self.hasChanged = True

    def OnTrackNumChanged(self, event):
        """The user has changed the local track number"""

        if self.curTrack:
            debug.debug("mdatapanel OnTrackNumChanged", "mdata.event")
            self.hasChanged = True

    def OnEnter(self, event):
        """The user has pressed Enter in one of the local data fields"""

        self.saveTrackInfo()
        event.Skip()

    def saveTrackInfo(self):
        """Save the current local data to the current object (e.g. a track)
        
        Called by OnEnter, or the 'save' button in the toolbar"""

        if self.curTrack:
            self.writeLocalDataToObject(self.curTrack)

    # Why is this sub in this class?
    def showOptions(self, pageToShow=0):
        """Show the options dialog.  Called by parent frame"""

        dlg = optionsdialog.OptionsDialog(self.frame, self.config, pageToShow)
        dlg.ShowModal()
        dlg.Destroy()
        wx.PostEvent(self.frame, events.SettingsChangedEvent())
        wx.WakeUpIdle()

    ########################################################################
    # These methods are all to do with writing the data back to the data object
    ########################################################################

    def writeLocalDataToObject(self, curTrack):
        """Write the local data back to the object it came from"""

        debug.debug("writeLocalDataToObject %s" % repr(curTrack), "mdata.write")
        n = curTrack.__class__.__name__

        if not self.hasChanged or not curTrack:
            debug.debug("writeLocalDataToObject no change / no object", "mdata.write")
            return

        if n == "Track": self.writeLocalDataToTrack(curTrack)
        elif n == "TPFile": self.writeLocalDataToTPFile(curTrack)
        elif n == "Album": pass
        elif n == "AlbumUnmatched": pass
        elif n == "ClusterAlbum": pass
        else:
            debug.debug("writeLocalDataToObject %s unhandled" % repr(item), "mdata.write.unhandled")

    def writeLocalDataToTrack(self, tr):
        """Write the local data back to a Track object"""

        # This code should proably be moved to Track.py
        fileId = tr.getFileId()
        tpTrack = self.tunePimp.getTrack(fileId)
        tpTrack.lock()
        hasChanged = tpTrack.hasChanged()
        mdata = tpTrack.getServerMetadata()
        mdata.artist = self.localArtistCombo.GetValue()
        mdata.album = self.localAlbumCombo.GetValue()
        mdata.track = self.localTrackEdit.GetValue()
        mdata.trackNum = self.localTrackNumSpin.GetValue()
        self.setReleaseDate(mdata, self.localDateEdit.GetValue())
        tpTrack.setServerMetadata(mdata)
        tpTrack.unlock()
        self.tunePimp.releaseTrack(tpTrack)

        tpfile = self.frame.tpmanager.findFile(fileId)
        al = tr.getAlbum()
        al.checkTrackUnsaved(tpfile)

        self.hasChanged = False

        wx.PostEvent(self.frame, events.UpdateFileEvent(fileId))
        wx.WakeUpIdle()

    def writeLocalDataToTPFile(self, tpfile):
        """Write the local data back to a TPFile object"""

        fileId = tpfile.getFileId()
        tpTrack = tpfile.getTrack()

        try:
            mdata = tpTrack.getLocalMetadata()
            mdata.artist = self.localArtistCombo.GetValue()
            mdata.album = self.localAlbumCombo.GetValue()
            mdata.track = self.localTrackEdit.GetValue()
            mdata.trackNum = self.localTrackNumSpin.GetValue()
            self.setReleaseDate(mdata, self.localDateEdit.GetValue())
            tpTrack.setLocalMetadata(mdata)
        finally:
            tpfile.releaseTrack(tpTrack)

        al = tpfile.getAlbum()
        if al.__class__.__name__ == "Album" and not al.isSpecial():
            al.checkTrackUnsaved(tpfile)

        self.hasChanged = False

        wx.PostEvent(self.frame, events.UpdateFileEvent(fileId))
        wx.WakeUpIdle()

    ########################################################################

    def enableLocalControls(self, state):
        """Set the local controls to all enabled / all disabled"""

        self.localArtistCombo.Enable(state)
        self.localAlbumCombo.Enable(state)
        self.localTrackEdit.Enable(state)
        self.localTrackNumSpin.Enable(state)
        self.localDurationEdit.Enable(state)
        self.localDateEdit.Enable(state)

    def enableServerControls(self, state):
        """Set the server controls to all enabled / all disabled"""

        self.serverArtistEdit.Enable(state)
        self.serverAlbumEdit.Enable(state)
        self.serverTrackEdit.Enable(state)
        self.serverTrackNumEdit.Enable(state)
        self.serverDurationEdit.Enable(state)
        self.serverDateEdit.Enable(state)

    def clearLocalControls(self):
        """Clear all the local controls"""

        self.localArtistCombo.SetValue("")
        self.localAlbumCombo.SetValue("")
        self.localTrackEdit.Clear()
        self.localDurationEdit.Clear()
        self.localTrackNumSpin.SetValue(0)
        self.localDateEdit.Clear()

    def clearServerControls(self):
        """Clear all the server controls"""

        self.serverArtistEdit.Clear()
        self.serverAlbumEdit.Clear()
        self.serverTrackEdit.Clear()
        self.serverTrackNumEdit.Clear()
        self.serverDurationEdit.Clear()
        self.serverDateEdit.Clear()

    ########################################################################
    # These methods are all to do with changing the current data object
    ########################################################################

    def showItemDetails(self, item):
        """Show the metadata for a given object.  Called by the parent frame."""
        
        debug.debug("showItemDetails %s" % repr(item), "mdata.show")

        if item != self.curTrack:
           self.writeLocalDataToObject(self.curTrack)

        self.curTrack = None
        self.clearLocalControls()
        self.clearServerControls()
        self.infoPanel.setStatusText("")
        self.frame.setStatusText("")

        # TODO handle TPFile
        if item.__class__.__name__ == "Album":
            self.showAlbumDetails(item)
        elif item.__class__.__name__ == "Track":
            self.showTrackDetails(item)
        elif item.__class__.__name__ == "TPFile":
            self.showTPFileDetails(item)
        elif item.__class__.__name__ == "ClusterAlbum":
            self.showClusterDetails(item)
        elif item.__class__.__name__ == "AlbumUnmatched":
            self.showAlbumUnmatchedDetails(item)
        else:
            self.localLookupButton.Enable(False)
            self.serverLookupButton.Enable(False)
            self.enableServerControls(False)
            self.enableLocalControls(False)
            debug.debug("showItemDetails %s unhandled" % repr(item), "mdata.show.unhandled")

    def showAlbumUnmatchedDetails(self, al):
        """Set the current data object to an 'AlbumUnmatched' object"""

        self.serverLookupButton.Enable(False)
        self.localLookupButton.Enable(False)

    def showAlbumDetails(self, al):
        """Set the current data object to an 'Album' object"""

        self.clearLocalControls()
        self.clearServerControls()
        self.enableServerControls(False)
        self.enableLocalControls(False)

        if al.isSpecial():
            self.serverLookupButton.Enable(False)
            self.localLookupButton.Enable(False)
            return

        self.serverLookupButton.Enable(True)
        self.localLookupButton.Enable(True)

        self.localArtistCombo.SetValue(al.getArtist().getName())
        self.serverArtistEdit.SetValue(al.getArtist().getName())
        self.serverArtistEdit.Enable(True)
        self.localAlbumCombo.SetValue(al.getName())
        self.serverAlbumEdit.SetValue(al.getName())
        self.serverAlbumEdit.Enable(True)
        self.serverTrackEdit.SetValue("")
        self.serverTrackNumEdit.SetValue("")
        self.serverDurationEdit.SetValue(al.getDurationString())
        self.infoPanel.setStatusText(_("%d of %d tracks linked") %
                                     (al.getNumLinkedTracks(),
                                      al.getNumTracks()))
        self.serverDateEdit.SetValue(al.getReleaseDate())

        self.infoPanel.setInfo(al)

    def showClusterDetails(self, al):
        """Set the current data object to a 'ClusterAlbum' object"""

        self.localLookupButton.Enable(True)
        self.serverLookupButton.Enable(False)
        self.clearLocalControls()
        self.clearServerControls()
        self.enableServerControls(False)
        self.enableLocalControls(False)

        self.localArtistCombo.SetValue(al.getArtistName())
        self.localArtistCombo.Enable(True)
        self.localAlbumCombo.SetValue(al.getName())
        self.localAlbumCombo.Enable(True)
        self.localDurationEdit.SetValue("")
        self.localTrackEdit.SetValue("")
        self.localTrackNumSpin.SetValue(0)
        self.localDateEdit.SetValue("")

    def showTrackDetails(self, tr):
        """Set the current data object to a 'Track' object"""

        self.hasChanged = False

        fileId = tr.getFileId()
        if tr.getArtist():
            self.serverArtistEdit.SetValue(tr.getArtist().getName())
            self.serverAlbumEdit.SetValue(tr.getAlbum().getName())
            self.serverTrackEdit.SetValue(tr.getName())
            self.serverTrackNumEdit.SetValue(str(tr.getNum()))
            self.serverDurationEdit.SetValue(tr.getDurationString())
            self.serverDateEdit.SetValue(tr.getAlbum().getReleaseDate())
            self.serverLookupButton.Enable(True)
        else:
            self.serverLookupButton.Enable(False)

        if fileId >= 0:

            tpTrack = self.tunePimp.getTrack(fileId);
            tpTrack.lock()
            mdata = tpTrack.getLocalMetadata()
            status = tpTrack.getStatus()
            error = ""
            if status == tunepimp.eError:
                error = tpTrack.getTrackError()
            #if not mdata.artist and not mdata.album and not mdata.track:
            #    mdata = tpTrack.getLocalMetadata()

            fileName = tpTrack.getFileName()
            tpTrack.unlock()
            self.tunePimp.releaseTrack(tpTrack)

            self.enableLocalControls(True)
            self.enableServerControls(True)

            self.localLookupButton.Enable(True)

            self.localArtistCombo.SetValue(mdata.artist)
            self.localAlbumCombo.SetValue(mdata.album)
            self.localTrackEdit.SetValue(mdata.track)
            self.localDurationEdit.SetValue(u"%d:%02d" % ((mdata.duration / 60000), 
                                                          ((mdata.duration % 60000) / 1000)))
            self.localTrackNumSpin.SetValue(mdata.trackNum)
            self.localDateEdit.SetValue(self.getReleaseDate(mdata))
     
            if tr.isUnmatched():
                self.infoPanel.setStatusText("")
            else:
                self.infoPanel.setStatusText(_("%d%% similar") % int(tr.getSimilarity()))
                
            if status == tunepimp.eError:
                self.frame.setStatusText("%s (%s)" % (fileName, error))
            else:
                self.frame.setStatusText(fileName)

            self.curTrack = tr

        else:

            self.enableLocalControls(False)
            self.enableServerControls(True)

            self.localLookupButton.Enable(False)
            self.frame.setStatusText("")
            self.infoPanel.setStatusText("")

    def showTPFileDetails(self, tpfile):
        """Set the current data object to a 'TPFile' object"""

        self.hasChanged = False

        fileId = tpfile.getFileId()
        tpTrack = tpfile.getTrack()

        try:
            mdata = tpTrack.getLocalMetadata()
            status = tpTrack.getStatus()
            error = ""
            if status == tunepimp.eError:
                error = tpTrack.getTrackError()
            fileName = tpTrack.getFileName()
            puid = tpTrack.getPUID()
        finally:
            tpfile.releaseTrack(tpTrack)

        self.enableLocalControls(True)
        self.localLookupButton.Enable(True)

        self.enableServerControls(False)
        self.serverLookupButton.Enable(True)

        self.localArtistCombo.SetValue(mdata.artist)
        self.localAlbumCombo.SetValue(mdata.album)
        self.localTrackEdit.SetValue(mdata.track)
        self.localDurationEdit.SetValue(u"%d:%02d" % ((mdata.duration / 60000), 
                                                      ((mdata.duration % 60000) / 1000)))
        self.localTrackNumSpin.SetValue(mdata.trackNum)
        self.localDateEdit.SetValue(self.getReleaseDate(mdata))
        self.frame.setStatusText(fileName)
 
        statusText = fileName
        if puid:
            statusText += " (PUID: %s)" % (puid,)
        elif puid != None:
            statusText += " (" + _("No PUID") + ")"
            
        if status == tunepimp.eError:
            statusText += " (%s)" % (error,)
        elif status == tunepimp.ePUIDCollision:
            statusText += " (" + _("PUID Collision") + ")"
            
        self.frame.setStatusText(statusText)

        self.curTrack = tpfile

    def getReleaseDate(self, mdata):
        releaseDate = u""
        if mdata.releaseYear:
            releaseDate += u"%04d" % mdata.releaseYear
            if mdata.releaseMonth:
                releaseDate += u"-%02d" % mdata.releaseMonth
                if mdata.releaseDay:
                    releaseDate += u"-%02d" % mdata.releaseDay
        return releaseDate

    def setReleaseDate(self, mdata, releaseDate):
        mdata.releaseYear = 0
        mdata.releaseMonth = 0
        mdata.releaseDay = 0
        date = releaseDate.split("-")
        dateLen = len(date)
        if dateLen > 0:
            try:
                mdata.releaseYear = int(date[0])
            except ValueError:
                mdata.releaseYear = 0
            if dateLen > 1:
                try:
                    mdata.releaseMonth = int(date[1])
                except ValueError:
                    mdata.releaseMonth = 0
                if dateLen > 2:
                    try:
                        mdata.releaseDay = int(date[2])
                    except ValueError:
                        mdata.releaseDay = 0
        if mdata.releaseMonth > 12:
            mdata.releaseMonth = 12
        if mdata.releaseDay > 31:
            mdata.releaseDay = 31
        
#
