# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK *****

import albummanager, debug
from tunepimp import tunepimp

try:
    tunepimp.eTooManyPUIDs
except AttributeError:
    tunepimp.eTooManyPUIDs = tunepimp.eTooManyTRMs
    tunepimp.ePUIDLookup = tunepimp.eTRMLookup
    tunepimp.ePUIDCollision = tunepimp.eTRMCollision
    from tunepimp import track
    track.track.getPUID = lambda self: None

callbackEnumDict = {
    tunepimp.eFileAdded         : "eFileAdded",
    tunepimp.eFileChanged       : "eFileChanged",
    tunepimp.eFileRemoved       : "eFileRemoved",
    tunepimp.eWriteTagsComplete : "eWriteTagsComplete",
}

errorEnumDict = {
    tunepimp.eOk                : "eOk",
    tunepimp.eTooManyPUIDs       : "eTooManyPUIDs",
    tunepimp.eNoUserInfo        : "eNoUserInfo",
    tunepimp.eLookupError       : "eLookupError",
    tunepimp.eSubmitError       : "eSubmitError",
    tunepimp.eInvalidIndex      : "eInvalidIndex",
    tunepimp.eInvalidObject     : "eInvalidObject",
}

fileStatusEnumDict = {
    tunepimp.eMetadataRead      : "eMetadataRead",
    tunepimp.ePending           : "ePending",
    tunepimp.eUnrecognized      : "eUnrecognized",
    tunepimp.eRecognized        : "eRecognized",
    tunepimp.ePUIDLookup         : "ePUIDLookup",
    tunepimp.ePUIDCollision      : "ePUIDCollision",
    tunepimp.eFileLookup        : "eFileLookup",
    tunepimp.eUserSelection     : "eUserSelection",
    tunepimp.eVerified          : "eVerified",
    tunepimp.eSaved             : "eSaved",
    tunepimp.eDeleted           : "eDeleted",
    tunepimp.eError             : "eError",
}

resultTypeEnumDict = {
    tunepimp.eNone              : "eNone",
    tunepimp.eArtistList        : "eArtistList",
    tunepimp.eAlbumList         : "eAlbumList",
    tunepimp.eTrackList         : "eTrackList",
    tunepimp.eMatchedTrack      : "eMatchedTrack",
}

albumTypeEnumDict = {
    tunepimp.eAlbumType_Album           : "eAlbumType_Album",
    tunepimp.eAlbumType_Single          : "eAlbumType_Single",
    tunepimp.eAlbumType_EP              : "eAlbumType_EP",
    tunepimp.eAlbumType_Compilation     : "eAlbumType_Compilation",
    tunepimp.eAlbumType_Soundtrack      : "eAlbumType_Soundtrack",
    tunepimp.eAlbumType_Spokenword      : "eAlbumType_Spokenword",
    tunepimp.eAlbumType_Interview       : "eAlbumType_Interview",
    tunepimp.eAlbumType_Audiobook       : "eAlbumType_Audiobook",
    tunepimp.eAlbumType_Live            : "eAlbumType_Live",
    tunepimp.eAlbumType_Remix           : "eAlbumType_Remix",
    tunepimp.eAlbumType_Other           : "eAlbumType_Other",
    tunepimp.eAlbumType_Error           : "eAlbumType_Error",
}

albumStatusEnumDict = {
    tunepimp.eAlbumStatus_Official      : "eAlbumStatus_Official",
    tunepimp.eAlbumStatus_Promotion     : "eAlbumStatus_Promotion",
    tunepimp.eAlbumStatus_Bootleg       : "eAlbumStatus_Bootleg",
}

def labelEnumValue(value, dict):
    try:
        return "%d (%s)" % (value, dict[value])
    except LookupError:
        return "%d (?)" % value

def labelCallbackEnum   (value): return labelEnumValue(value, callbackEnumDict  )
def labelErrorEnum      (value): return labelEnumValue(value, errorEnumDict     )
def labelFileStatusEnum (value): return labelEnumValue(value, fileStatusEnumDict)
def labelResultTypeEnum (value): return labelEnumValue(value, resultTypeEnumDict)
def labelAlbumTypeEnum  (value): return labelEnumValue(value, albumTypeEnumDict )
def labelAlbumStatusEnum(value): return labelEnumValue(value, albumStatusEnumDict)

class TPManager(object):
    """
    Allows access to the tunepimp object, and keeps track of TPFile objects
    """

    def __init__(self, tunepimp):
        self.tunepimp = tunepimp
        self.files = {} # fileId => TPFile object

    def getTunepimp(self):
        return self.tunepimp

    def setAlbumManager(self, albummanager):
        self.albummanager = albummanager
    def getAlbumManager(self):
        return self.albummanager

    def fileAdded(self, fileId):
        """
        Tunepimp has told us that a file has been added.  Create, store and
        return a TPFile object to represent that file.  If the file already
        exists, return the TPFile without changing anything.
        """

        # TP can apparently generate repeated "eFileAdded" events for the same
        # file.  Is this a bug?
        try:
            return self.files[fileId]
        except KeyError:
            assert(fileId not in self.files)
            tpfile = TPFile(self, fileId)
            self.files[fileId] = tpfile
            return tpfile

    def fileRemoved(self, fileId):
        """
        Tunepimp has told us that a file has been removed.  Remove the
        corresponding TPFile object.
        """

        if self.files.has_key(fileId):
            file = self.files[fileId]
            file.removeFromAlbum()
            del self.files[fileId]

    def findFile(self, fileId):
        """Given a tunepimp file ID, fetch the TPFile object"""
        return self.files[fileId]

    def getAllFiles(self):
        """Returns the dictionary of all tunepimp files (fileId: TPFile)"""
        return dict(self.files)

    def dumpState(self, w):
        """Dump the tunepimp global settings, and all TPFile objects, using a NestedDumper"""

        w(u"tpmanager = %s" % ( repr(self) ))
        w = w.nest()
        w(u"tunepimp = %s" % ( repr(self.tunepimp) ))
        w(u"files = %s" % ( repr(self.files) ))
        self.dumptpglobal(w)
        # TODO check that self.files matches tp fileid list

        ids = self.tunepimp.getFileIds()
        if self.files.keys() == ids:
            w(u"(self.files matches self.tunepimp.getFileIds)")
        else:
            w(u"MISMATCH!  ids = %s" % ( ids ))

        for f in self.files.values():
            f.dumpState(w)

    def dumptpglobal(self, w):
        """Dump the tunepimp object's global settings using a NestedDumper"""

        tp = self.tunepimp
        w(u"AllowedChars: %s" % ( repr(tp.getAllowedFileCharacters()) ))
        w(u"AnalyzerPriority: %s" % ( repr(tp.getAnalyzerPriority()) ))
        w(u"AutoRemoveSavedFiles: %s" % ( repr(tp.getAutoRemovedSavedFiles()) ))
        w(u"AutoSaveThreshold: %s" % ( repr(tp.getAutoSaveThreshold()) ))
        w(u"ClearTags: %s" % ( repr(tp.getClearTags()) ))
        w(u"Debug: %s" % ( repr(tp.getDebug()) ))
        w(u"DestDir: %s" % ( repr(tp.getDestDir()) ))
        w(u"Error: %s" % ( repr(tp.getError()) ))
        w(u"FileMask: %s" % ( repr(tp.getFileMask()) ))
        w(u"ID3V1: %s" % ( repr(tp.getWriteID3v1()) ))
        w(u"MaxFileNameLen: %s" % ( repr(tp.getMaxFileNameLen()) ))
        w(u"MoveFiles: %s" % ( repr(tp.getMoveFiles()) ))
        w(u"NumFiles: %s" % ( repr(tp.getNumFiles()) ))
        w(u"NumUnsavedItems: %s" % ( repr(tp.getNumUnsavedItems()) ))
        w(u"Proxy: %s" % ( repr(tp.getProxy()) ))
        w(u"RenameFiles: %s" % ( repr(tp.getRenameFiles()) ))
        w(u"Server: %s" % ( repr(tp.getServer()) ))
        w(u"Status: %s" % ( repr(tp.getStatus()) ))
        w(u"Supported extensions are %s" % ( repr(tp.getSupportedExtensions()) ))
        w(u"TopSrcDir: %s" % ( repr(tp.getTopSrcDir()) ))
        w(u"VAFileMask: %s" % ( repr(tp.getVariousFileMask()) ))

class TPFile(object):
    """
    Represents a file in tunepimp, and also its status within the tagger -
    e.g. linked to an album track, on an album but unmatched, etc.
    """

    def __init__(self, manager, fileId):
        self.manager = manager
        self.id = fileId
        self.album = None
        self.status = None
        self.puid = None
        debug.debug("file #%d added to tunepimp" % self.getFileId(), "tpfile.init")
        pending = self.manager.getAlbumManager().get(albummanager.pendingFiles)
        self.moveToAlbumAsUnlinked(pending)

    def getFileId(self):
        return self.id

    def getAlbum(self):
        return self.album

    def setStatus(self, status):
        self.status = status
        
    def getStatus(self):
        return self.status
        
    def setPUID(self, puid):
        self.puid = puid
        
    def getPUID(self):
        return self.puid
        
    # All manipulation of files vs tracks (link, unlink, move to album etc) is
    # done via the following methods:

    def moveToCluster(self, cluster):
        """Moves the file to a given album cluster"""

        self.removeFromAlbum()
        debug.debug("file #%d moving to cluster %s" % (self.getFileId(), cluster.getId()),
            "tpfile.move")
        self.album = cluster
        self.album.addFile(self)

    def moveToAlbumAsUnlinked(self, album):
        """Moves the file to a given album as 'unmatched'"""

        self.removeFromAlbum()
        debug.debug("file #%d moving to album %s as unlinked" % (self.getFileId(), album.getId()),
            "tpfile.move")
        self.album = album
        self.album.addUnlinkedFile(self)

    def moveToAlbumAsLinked(self, album, seq):
        """Moves the file to a given album, matching it to the track indicated
        by seq (0==first track)"""

        self.removeFromAlbum()
        debug.debug("file #%d moving to album %s as seq #%d" % (self.getFileId(), album.getId(), seq),
            "tpfile.move")
        self.album = album
        self.album.addLinkedFile(self, seq)

    def removeFromAlbum(self):
        """Removes the file from whatever album it may be on.  Does nothing if
        the file is not currently on an album"""

        if self.album is None:
            return
        debug.debug("file #%d being removed from album %s" % (self.getFileId(), self.album.getId()),
            "tpfile.move")
        self.album.removeFile(self)
        self.album = None

    def removeFromSystem(self):
        """Removes the file from the system entirely (first removes it from
        any album it may be on, then requests that tunepimp drop the file)"""

        self.removeFromAlbum()
        debug.debug("file #%d being removed from tunepimp" % self.getFileId(),
            "tpfile.remove")
        self.manager.tunepimp.remove(self.getFileId())

    # End of file manipulation methods

    def dumpState(self, w):
        """Dump this file's state using a NestedDumper"""

        def ShowMetadata(m, w):
            w(u"metadata object: %s" % ( repr(m) ))
            w(u"artist: %s" % ( repr(m.artist) ))
            w(u"sortName: %s" % ( repr(m.sortName) ))
            w(u"album: %s" % ( repr(m.album) ))
            w(u"track: %s" % ( repr(m.track) ))
            w(u"trackNum: %s" % ( repr(m.trackNum) ))
            w(u"totalInSet: %s" % ( repr(m.totalInSet) ))
            w(u"variousArtist: %s" % ( repr(m.variousArtist) ))
            w(u"nonAlbum: %s" % ( repr(m.nonAlbum) ))
            w(u"artistId: %s" % ( repr(m.artistId) ))
            w(u"albumId: %s" % ( repr(m.albumId) ))
            w(u"trackId: %s" % ( repr(m.trackId) ))
            w(u"filePUID: %s" % ( repr(m.filePUID) ))
            w(u"albumArtistId: %s" % ( repr(m.albumArtistId) ))
            w(u"albumArtist: %s" % ( repr(m.albumArtist) ))
            w(u"albumArtistSortName: %s" % ( repr(m.albumArtistSortName) ))
            w(u"duration: %s" % ( repr(m.duration) ))
            w(u"albumType: %s" % ( labelAlbumTypeEnum(m.albumType) ))
            w(u"albumStatus: %s" % ( labelAlbumStatusEnum(m.albumStatus) ))
            w(u"fileFormat: %s" % ( repr(m.fileFormat) ))
            w(u"releaseYMD: %s" % ( repr((m.releaseYear, m.releaseMonth, m.releaseDay)) ))
            w(u"releaseCountry: %s" % ( repr(m.releaseCountry) ))

        tr = self.getTrack()
        try:
            tr.lock()
            w(u"File #%d" % self.id)
            w = w.nest()
            if self.album is None:
                w(u"Album: none")
            else:
                w(u"Album: %s" % self.album.getId())
            w(u"Status: %s" % ( labelFileStatusEnum(tr.getStatus()) ))
            w(u"Filename: %s" % ( repr(tr.getFileName()) ))
            w(u"PUID: %s" % ( repr(tr.getPUID()) ))
            w(u"Error: %s" % ( repr(tr.getTrackError()) ))
            w(u"Similarity: %s" % ( repr(tr.getSimilarity()) ))
            w(u"hasChanged: %s" % ( repr(tr.hasChanged()) ))
            w(u"Local metadata:")
            ShowMetadata(tr.getLocalMetadata(), w.nest())
            w(u"Server metadata:")
            ShowMetadata(tr.getServerMetadata(), w.nest())
        finally:
            tr.unlock()
        self.releaseTrack(tr)

    def getTrack(self):
        """Get and lock the tunepimp track for this file (see also releaseTrack)"""

        tr = self.manager.tunepimp.getTrack(self.id)
        tr.lock()
        return tr

    def releaseTrack(self, tr):
        """Unlock and releasea tunepimp track for this file (see also getTrack)"""

        tr.unlock()
        self.manager.tunepimp.releaseTrack(tr)

