# ***** BEGIN LICENSE BLOCK ***** 
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED,G AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK ***** 

try:
    import wxversion
    wxversion.select(["2.6", "2.7"])
except:
    pass

import sys
import os
import copy
import wx
import atexit
import time
import traceback
import gettext
import locale

import __builtin__
__builtin__.__dict__['_'] = lambda string: string
__builtin__.__dict__['N_'] = lambda string: string

from tunepimp import tunepimp
from musicbrainz2.disc import readDisc, getSubmissionUrl

from picard import APPNAME, APPVERSION, VENDORNAME
from picard import events, coverartcache, albummanager, util, debug, tpmanager, cluster, wpath, cuesheet, playlist, puidmanager
from picard.browser import browser, filelookup, launch
from picard.ui import mdatapanel, albumpanel, images, fileselection, toolbar, coverartpanel, infopanel, nagdialog
from picard.usercheck import UserCheckThread
 
# TODO: Add command line arg support

class TaggerContext(object):
    def __init__(self):
        pass

def shutdownFrame(frame):
    try:
        frame.shutdown()
    except:
        pass

class TaggerFrame(wx.Frame):

    splitterId = wx.NewId()
    albumPanelId = wx.NewId()
    fileSelectionPanelId = wx.NewId()
    dirControlId = wx.NewId()
    infoSplitterId = wx.NewId()

    fileMenuId = wx.NewId()
    fileMenuAddFileId = wx.NewId()
    fileMenuAddDirId = wx.NewId()
    fileMenuSaveId = wx.NewId()
    fileMenuDumpId = wx.NewId()
    fileMenuExitId = wx.NewId()
    fileMenuOptionsId = wx.NewId()
    editMenuId = wx.NewId()
    editMenuCopyId = wx.NewId()
    editMenuPasteId = wx.NewId()
    helpMenuId = wx.NewId()
    helpMenuHelpId = wx.NewId()
    helpMenuQuickStartId = wx.NewId()
    helpMenuAboutId = wx.NewId()

    currentAsin = u""
    taggerPort = -1

    tunePimp = None
    timer = None
    timerId = wx.NewId()
    maxNotificationsPerPass = 15
    timerDelay = 20    # in ms

    def __init__(self, parent, id, title, config):

        self.config = config
        self.wpath = wpath.wpath(config)
        self.config.setTagger(self)
        pos = wx.DefaultPosition
        size = wx.Size(800, 600)
        try:
            (x, y, width, height) = self.config.persistWindowGeometry.split(u",")
            pos = wx.Point(int(x),int(y))
            size = wx.Size(int(width),int(height))
        except ValueError:
            pass

        wx.Frame.__init__(self, parent, -1, title, pos, size, wx.SYSTEM_MENU | wx.CAPTION | wx.RESIZE_BORDER | 
                         wx.MINIMIZE_BOX | wx.MAXIMIZE_BOX | wx.CLOSE_BOX)

        if sys.platform == 'win32':
            # switch off remapping of bitmaps
            if wx.PyApp.GetComCtl32Version() >= 600 and wx.DisplayDepth() >= 32:
                wx.SystemOptions.SetOption("msw.remap", "0");
                
        ib = wx.IconBundle()
        ib.AddIcon(images.getAppIconIcon())
        ib.AddIcon(images.getAppIcon32Icon())
        self.SetIcons(ib)

        atexit.register(shutdownFrame, self)

        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)
        self.Bind(wx.EVT_SPLITTER_DOUBLECLICKED, self.OnSplitterDClick, id=self.splitterId)
        self.Bind(wx.EVT_MENU, self.OnMenuDump, id=self.fileMenuDumpId)
        self.Bind(wx.EVT_MENU, self.OnMenuExit, id=self.fileMenuExitId)
        self.Bind(wx.EVT_MENU, self.OnOptions, id=self.fileMenuOptionsId)
        self.Bind(wx.EVT_MENU, self.OnAbout, id=self.helpMenuAboutId)
        self.Bind(wx.EVT_MENU, self.OnHelp, id=self.helpMenuHelpId)
        self.Bind(wx.EVT_MENU, self.OnQuickStart, id=self.helpMenuQuickStartId)
        self.Bind(wx.EVT_MENU, self.OnAddFile, id=self.fileMenuAddFileId)
        self.Bind(wx.EVT_MENU, self.OnAddDir, id=self.fileMenuAddDirId)
        self.Bind(wx.EVT_MENU, self.OnSaveButton, id=self.fileMenuSaveId)
        self.Bind(wx.EVT_TIMER, self.OnTimer, id=self.timerId)
        self.Bind(events.EVT_COVERART_LOAD, self.OnCoverArtLoad)
        self.Bind(events.EVT_COVERART_SHOW, self.OnCoverArtShow)
        self.Bind(events.EVT_BROWSER_MSG, self.OnBrowserMessage)
        self.Bind(events.EVT_ALBUM_UPDATE, self.OnAlbumUpdate)
        self.Bind(events.EVT_ALBUM_TRACK_COUNTS_CHANGED, self.OnAlbumTrackCountsChanged)
        self.Bind(events.EVT_ALBUM_FILES_ADDED, self.OnAlbumFilesAdded)
        self.Bind(events.EVT_ALBUM_FILES_REMOVED, self.OnAlbumFilesRemoved)
        self.Bind(events.EVT_ALBUM_REMOVE_UNMATCHED_FILES, self.OnAlbumRemoveUnmatched)
        self.Bind(events.EVT_MOVE_ALBUM_UNMATCHED_TO_UNMATCHED, self.OnAlbumMoveUnmatched)
        self.Bind(events.EVT_MOVE_FILES_TO_UNMATCHED, self.OnMoveFilesToUnmatched)
        self.Bind(events.EVT_CLUSTER_ADDED, self.OnClusterAdded)
        self.Bind(events.EVT_CLUSTER_REMOVED, self.OnClusterRemoved)
        self.Bind(events.EVT_CLUSTER_CHANGED, self.OnClusterChanged)
        self.Bind(events.EVT_CLUSTER_FILE_ADDED, self.OnClusterFileAdded)
        self.Bind(events.EVT_CLUSTER_FILE_REMOVED, self.OnClusterFileRemoved)
        self.Bind(events.EVT_ALBUM_ADD, self.OnAlbumAdd)
        self.Bind(events.EVT_ALBUM_REMOVE, self.OnAlbumRemove)
        self.Bind(events.EVT_ALBUM_REMOVED, self.OnAlbumRemoved)
        self.Bind(events.EVT_FILE_ADD, self.OnFileAdd)
        self.Bind(events.EVT_DIR_ADD, self.OnDirAdd)
        self.Bind(events.EVT_SHOW_ITEM_DETAILS, self.OnShowItemDetails)
        self.Bind(events.EVT_LINK_FILE, self.OnLinkFile)
        self.Bind(events.EVT_UPDATE_FILE, self.OnUpdateFile)
        self.Bind(events.EVT_ANALYZE_FILE, self.OnAnalyzeFile)
        self.Bind(events.EVT_ANALYZE_FILE2, self.OnAnalyzeFile2)
        self.Bind(events.EVT_SAVE_FILE, self.OnSaveFile)
        self.Bind(events.EVT_MATCH_DIR_TO_ALBUM, self.OnMatchDirToAlbum)
        self.Bind(events.EVT_ADD_FILE_TO_CLUSTER, self.OnAddFileToCluster)
        self.Bind(events.EVT_REMOVE_CLUSTER_ALBUM, self.OnRemoveClusterAlbum)
        self.Bind(events.EVT_CLUSTER_UNMATCHED_FILES, self.OnClusterFiles)
        self.Bind(events.EVT_MOVE_CLUSTERS_TO_UNMATCHED, self.OnMoveClustersToUnmatched)
        self.Bind(events.EVT_SHOW_DIRECTORY, self.OnShowDirectory)
        self.Bind(events.EVT_SETTINGS_CHANGED, self.OnSettingsChanged)
        self.Bind(events.EVT_CLEAR_ERROR, self.OnClearError)
        self.Bind(events.EVT_FORCE_SAVE, self.OnForceSave)
        self.Bind(events.EVT_GENERATE_CUESHEET, self.OnGenerateCuesheet)
        self.Bind(events.EVT_GENERATE_PLAYLIST, self.OnGeneratePlaylist)
        self.Bind(events.EVT_RELOAD_ALBUM, self.OnReloadAlbum)
        self.Bind(events.EVT_REMOVE_TPFILE_COMMAND, self.OnRemoveTPFileCommand)
        self.Bind(events.EVT_MATCH_FILES_TO_ALBUM, self.OnMatchFilesToAlbum)
        self.Bind(events.EVT_MATCH_FILEIDS_TO_ALBUM, self.OnMatchFileIdsToAlbum)
        self.Bind(events.EVT_SET_STATUS_TEXT, self.onSetStatusText)
        self.Bind(events.EVT_SHOW_STARTUP_DIALOGS, self.onShowStartupDialogs)

        self.refreshDirDict = {}
        self.dropTargetTrackIdCache = {}
        self.timer = wx.Timer(self, self.timerId)
        self.tunePimp = tunepimp.tunepimp(APPNAME, APPVERSION, tunepimp.tpThreadWrite | tunepimp.tpThreadRead | tunepimp.tpThreadAnalyzer, self.config.pluginsDir)
        self.tpmanager = tpmanager.TPManager(self.tunePimp)
        self.config.setTunePimp(self.tunePimp)
        if hasattr(self.tunePimp, 'setMusicDNSClientId'):
            self.config.disableFingerprinting = False
        else:
            self.config.disableFingerprinting = True
        self.setTunePimpSettings()

        extList = self.tunePimp.getSupportedExtensions()
        self.supportedExtensions = extList
        if sys.platform == 'win32':
            extList.append(".*")
        else:
            extList.append("*")

        # Reorder the list to have mp3 appear first
        try:
            extList.index(u".mp3")
            extList.remove(u".mp3")
            extList.insert(0, u".mp3")
        except:
            pass
        extList = [ (x[1:] + u" files (*" + x + u")|*" + x) for x in extList]
        self.fileFilter = "|".join(extList)

        self.coverArtCache = coverartcache.CoverArtCache(self, self.config)
        self.coverArtCache.start()
        self.currentAsin = u""

        self.browserIntegration = browser.BrowserIntegration(self.config)
        self.browserIntegration.start()


        # Setup the toolbar
        tb = self.CreateToolBar(wx.HORIZONTAL | wx.TB_FLAT)
        self.toolbar = toolbar.Toolbar(self, config, tb)

        # Setup the file browser and album panel
        self.splitter = wx.SplitterWindow(self, self.splitterId, wx.DefaultPosition, 
                                    wx.DefaultSize, wx.SP_PERMIT_UNSPLIT | wx.SP_LIVE_UPDATE)
        self.albumPanel = albumpanel.AlbumPanel(self.splitter, self.config, self.albumPanelId,
                                         self.tunePimp.getSupportedExtensions())
        self.fileSelectionPanel = fileselection.FileSelectionPanel(self.splitter, 
                                         self.config, self.albumPanel,
                                         self.fileSelectionPanelId, self.dirControlId,
                                         self.tunePimp.getSupportedExtensions())
        self.splitter.SetMinimumPaneSize(150)
        self.splitter.SplitVertically(self.fileSelectionPanel, self.albumPanel, self.config.persistSplitterPos);

        # Setup the toolbar panel
        statusPanel = wx.Panel(self, -1, style=wx.SIMPLE_BORDER)
        statusSizer = wx.BoxSizer(wx.HORIZONTAL)
        self.fileNameText = wx.StaticText(statusPanel, wx.NewId(), u"")
        statusSizer.Add(self.fileNameText, 0, wx.EXPAND | wx.ALL, 3)
        statusPanel.SetSizer(statusSizer)

        # Setup the info panel, metadata panel and album coverart panel
        self.infoPanel = infopanel.InfoPanel(self, self.config)
        self.metadataPanel = mdatapanel.MetadataPanel(self, self.config, self.albumPanel, 
                                                      self.infoPanel)
        self.coverArtPanel = coverartpanel.CoverArtPanel(self)

        # Add the info panel and cover art onto a sizer
        infoArtSizer = wx.BoxSizer(wx.HORIZONTAL)
        infoArtSizer.Add(self.metadataPanel, 30)
        infoArtSizer.Add(self.infoPanel, 11, wx.EXPAND)
        infoArtSizer.Add(self.coverArtPanel, 0, wx.EXPAND)

        # Construct the layout of the window finally
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(self.splitter, 1, wx.EXPAND)
        sizer.Add(infoArtSizer, 0, wx.EXPAND)
        sizer.Add(statusPanel, 0, wx.EXPAND)
        self.SetSizer(sizer)

        self.createMenu()

        if len(sys.argv) > 1 and sys.argv[1] == 'lookup':
            self.OnLookupCD(None)

        c = TaggerContext()
        self.context = c

        c.albummanager = albummanager.AlbumManager(c, self, self.config, self.tpmanager)
        c.clustermanager = cluster.ClusterManager(c)
        c.config = self.config
        c.frame = self
        c.tpmanager = self.tpmanager

        self.albumManager = c.albummanager
        self.tpmanager.setAlbumManager(c.albummanager)
        c.albummanager.start()

        c.puidmanager = puidmanager.PUIDManager(c)
        c.puidmanager.start()
        
        self.albumPanel.setAlbumManager(self.albumManager)
        self.albumPanel.setTPManager(self.tpmanager)

        self.albumPanel.setCurrentObject(None)

        userCheck = UserCheckThread(self, self.config)
        userCheck.start()
        
        # start the callback timer and get things moving
        self.timer.Start(self.timerDelay)

    def shutdown(self):

        unsaved = len(self.context.clustermanager.clusters)
        for album in self.context.albummanager.albumIndex.values():
            unsaved += len(album.unmatchedFiles)
            unsaved += len(album.unsavedFiles)

        if unsaved:
            dlg = wx.MessageDialog(self, _("Are you sure you want to exit the application?"), _("Are You Sure?"), wx.YES_NO | wx.NO_DEFAULT)
            if dlg.ShowModal() != wx.ID_YES: 
                return
            
        self.timer.Stop()

        self.config.persistOpenAlbums = u','.join([al for al in self.albumManager.getAlbumList() if not al.startswith('[')])
        self.config.persistSplitterPos = self.splitter.GetSashPosition()
        self.config.persistFileSplitterPos = self.fileSelectionPanel.getSashPosition()
        self.config.persistWindowGeometry = unicode(self.GetPositionTuple()[0]) + u"," + \
                                            unicode(self.GetPositionTuple()[1]) + u"," + \
                                            unicode(self.GetSizeTuple()[0]) + u"," + \
                                            unicode(self.GetSizeTuple()[1])

        # Bug the form/post launcher to clean up any files left behind
        l = launch.Launch(None)
        l.cleanup()

        c = self.context
        del c.albummanager
        del c.clustermanager
        del c.config
        del c.frame
        del c.tpmanager

        c.puidmanager.stop()
        del c.puidmanager
        
        del self.tunePimp
        self.tunePimp = None
        self.coverArtCache.stop()
        self.albumManager.stop()
        self.browserIntegration.stop()
        self.Destroy()

    def OnSettingsChanged(self, event):
        self.albumPanel.settingsChanged()
        self.setTunePimpSettings()

    def OnLookupCD(self, event):
        busyCursor = wx.BusyCursor()
        try:
            disc = readDisc(self.config.settingCDLookupDevice.encode(self.config.settingIOEncoding, 'replace'))
            lookupURL = getSubmissionUrl(disc=disc,
                                         host=self.config.settingServer.encode('UTF-8', 'replace'),
                                         port=self.config.settingServerPort)
        except:
            del busyCursor
            dlg = wx.MessageDialog(self, _("CD Lookup error -- is there a CD in the "
                 "CD-ROM drive?"), _("CD Lookup Failed"), wx.OK)
            dlg.ShowModal()
            return
            
        del busyCursor
        if lookupURL:
            lookup = filelookup.FileLookup(self, self.config.settingServer, 
                                           self.config.settingServerPort, 
                                           self.getTaggerPort())
            lookup.discLookup(lookupURL)

    def setTunePimpSettings(self):
        self.tunePimp.setMoveFiles(self.config.settingMoveFiles)
        self.tunePimp.setRenameFiles(self.config.settingRenameFiles)
        self.tunePimp.setFileMask(self.config.settingFileMask)
        self.tunePimp.setVariousFileMask(self.config.settingVAFileMask)
        self.tunePimp.setDestDir(self.config.settingDestDir)
        self.tunePimp.setTopSrcDir(u"")
        self.tunePimp.setAllowedFileCharacters(self.config.settingAllowedChars)
        self.tunePimp.setWriteID3v1(self.config.settingWriteID3v1)
        self.tunePimp.setClearTags(self.config.settingClearTags)
        self.tunePimp.setWriteID3v2_3(self.config.settingWriteID3v23)
        self.tunePimp.setID3Encoding(self.config.settingID3Encoding)
        self.tunePimp.setFileNameEncoding(self.config.settingIOEncoding)
        self.tunePimp.setAutoSaveThreshold(-1)
        if hasattr(self.tunePimp, 'setWinSafeFileNames'):
            self.tunePimp.setWinSafeFileNames(self.config.settingWinSafeFileNames)
        if self.config.disableFingerprinting:
            self.config.settingAutoAnalyze = False
        else:
            self.tunePimp.setMusicDNSClientId("0736ac2cd889ef77f26f6b5e3fb8a09c ")

    def getPUIDManager(self):
        return self.context.puidmanager        
        
    def createMenu(self):

        if sys.platform == "darwin":
            self.createMacMenu()
        else:
            self.createGenericMenu()

    def createGenericMenu(self):
        '''Creates the menubar suited for Linux/Unix/Windows'''

        self.fileMenu = wx.Menu() 
        self.fileMenu.Append(self.fileMenuAddFileId, _('Add &Files...\tCtrl+O')) 
        self.fileMenu.Append(self.fileMenuAddDirId, _('Add &Directory...\tCtrl+D')) 
        self.fileMenu.AppendSeparator() 
        self.fileMenu.Append(self.fileMenuSaveId, _('&Save Files\tCtrl+S')) 
        self.fileMenu.AppendSeparator() 
        self.fileMenu.Append(self.fileMenuOptionsId, _('&Options...')) 
        self.fileMenu.AppendSeparator() 
        if debug.isDebugClassEnabled("dump"):
            self.fileMenu.Append(self.fileMenuDumpId, u'Dump State') 
        self.fileMenu.Append(self.fileMenuExitId, _('E&xit')) 
             
        editMenu = wx.Menu() 
        editMenu.Append(self.editMenuCopyId, _('C&opy\tCtrl+C')) 
        editMenu.Append(self.editMenuPasteId, _('&Paste\tCtrl+V'))
        editMenu.Enable(self.editMenuCopyId, False)
        editMenu.Enable(self.editMenuPasteId, False)

        helpMenu = wx.Menu() 
        helpMenu.Append(self.helpMenuAboutId, _('&About...')) 
        helpMenu.Append(self.helpMenuQuickStartId, _('&Quick Start Guide')) 
        helpMenu.Append(self.helpMenuHelpId, _('&Help')) 
             
        menuBar = wx.MenuBar()
        menuBar.Append(self.fileMenu, _('&File'))
        menuBar.Append(editMenu, _('&Edit')) 
        menuBar.Append(helpMenu, _('&Help')) 
                     
        self.SetMenuBar(menuBar) 

    def createMacMenu(self):
        # FIXME: needs updating?
        '''Creates the menubar suited for Mac OS X'''

        # Override these values for the mac
        self.fileMenuOptionsId = wx.ID_PREFERENCES
        self.helpMenuAboutId = wx.ID_ABOUT

        self.fileMenu = wx.Menu() 
        self.fileMenu.Append(self.fileMenuAddFileId, _('Add File...')) 
        self.fileMenu.Append(self.fileMenuAddDirId, _('Add Directory...')) 
        #fileMenu.AppendSeparator() 
             
        editMenu = wx.Menu() 
        editMenu.Append(self.editMenuCopyId, _('Copy')) 
        editMenu.Append(self.editMenuPasteId, _('Paste')) 

        helpMenu = wx.Menu() 
        helpMenu.Append(self.helpMenuAboutId, _('About Tagger')) 
        helpMenu.Append(self.helpMenuHelpId, _('Help')) 
        helpMenu.Append(self.helpMenuQuickStartId, _('Quick Start Guide')) 
        helpMenu.Append(self.fileMenuOptionsId, _('Preferences')) 
             
        menuBar = wx.MenuBar() 
        menuBar.Append(self.fileMenu, _('File')) 
        menuBar.Append(editMenu, _('Edit')) 
        menuBar.Append(helpMenu, _('&Help')) 

        self.SetMenuBar(menuBar) 

        wx.App_SetMacPreferencesMenuItemId(self.fileMenuOptionsId)
        wx.App_SetMacAboutMenuItemId(self.helpMenuAboutId)
        wx.App_SetMacHelpMenuTitleName(_("&Help"))
        wx.App_SetMacExitMenuItemId(self.fileMenuExitId)

    def setStatusText(self, text):
        self.fileNameText.SetLabel(text.replace("&", "&&"))        
        
    def onSetStatusText(self, event):
        self.setStatusText(event.text)

    def onShowStartupDialogs(self, event):
        if event.dialogs & event.ACCOUNT:
            dlg = wx.MessageDialog(self, _("You have not specified an username and password. In order to contribute data to MusicBrainz, you must have a MusicBrainz account.\n"
                                           "Do you want to create a new account now? If you already have a MusicBrainz account, please enter your user information in the options dialog. "),
                                         _("MusicBrainz account"), wx.YES_NO | wx.NO_DEFAULT | wx.ICON_EXCLAMATION)
            if dlg.ShowModal() == wx.ID_YES:
                l = launch.Launch(None)
                l.launch('http://musicbrainz.org/newlogin.html')
                dlg.Destroy()
            
        if event.dialogs & event.DONATION:
            dlg = nagdialog.NagDialog(self)
            dlg.ShowModal()
            dlg.Destroy()

        if event.dialogs & event.CONNECTION_ERROR:
            dlg = wx.MessageDialog(self, _("Couldn't connect to the MusicBrainz server."),
                                         _("Connection error"), wx.OK | wx.ICON_EXCLAMATION)
            dlg.ShowModal()
            dlg.Destroy()
            
    def OnTimer(self, event):

        if not wx.USE_UNICODE:
            self.timer.Stop()
            dlg = wx.MessageDialog(self, _("MusicBrainz Picard requires wxPython with "
                "UNICODE support.\nPlease download the appropriate toolkit from "
                "http://www.wxpython.org/"), _("Unicode Required"), wx.OK)
            dlg.ShowModal()
            self.shutdown()

        if self.config.persistOpenAlbums:
            if self.config.settingReloadAlbums:
                for al in self.config.persistOpenAlbums.split(u','):
                    self.albumManager.add(al)
            self.config.persistOpenAlbums = u''

        
        if self.tunePimp:
         
            notificationCount = 0
            while notificationCount < self.maxNotificationsPerPass:
                notificationCount = notificationCount + 1
                ret, type, fileId, status = self.tunePimp.getNotification()
                if not ret:
                    break

                # Begin DEBUG BLOCK
                # We can get rid of this later, but for debugging we need the filename
                tr = self.tunePimp.getTrack(fileId)
                tr.lock()
                fileName = tr.getFileName()
                tr.unlock()
                self.tunePimp.releaseTrack(tr)
                # End DEBUG BLOCK

                debug.debug("TP callback: fileId=%d type=%s status=%s filename=%s"
                    % (fileId, tpmanager.labelCallbackEnum(type), tpmanager.labelFileStatusEnum(status), fileName),
                    "tunepimp.event")
                
                if type == tunepimp.eFileAdded:
                    self.tpmanager.fileAdded(fileId)

                elif type == tunepimp.eFileRemoved:
                    self.tpmanager.fileRemoved(fileId)

                elif type == tunepimp.eWriteTagsComplete:
                    try:
                        del self.busyCursor
                    except:
                        pass
                    self.writeTagsComplete()

                elif type == tunepimp.eFileChanged:

                    tpfile = self.tpmanager.findFile(fileId)
                    
                    try:
                        albumAndSeq = self.dropTargetTrackIdCache[fileId]
                        del self.dropTargetTrackIdCache[fileId]

                        al = self.albumManager.get(albumAndSeq[0])
                        if al:
                            # Check to see if this track is already assigned to an album. If so, remove the
                            # existing ids loaded from disk so the track goes to the album it was dropped on
                            tr = tpfile.getTrack()
                            ldata = tr.getLocalMetadata()
                            ldata.albumId = ""
                            ldata.artistId = ""
                            ldata.trackId = ""
                            tr.setLocalMetadata(ldata)
                            tpfile.releaseTrack(tr)

                            if albumAndSeq[1] == -1:
                                matchDict = self.albumManager.matchFileIdsToAlbum([fileId], al.getId())
                                for match in matchDict:
                                    albumAndSeq[1] = al.findSeqOfTrack(matchDict[match])

                            if albumAndSeq[1] != -1:
                                tpfile.moveToAlbumAsLinked(al, albumAndSeq[1])

                    except KeyError:
                        pass

                    if status == tunepimp.eUnrecognized:
                        album = tpfile.getAlbum()
                        if album.getId() == albummanager.pendingFiles:
                            if not tpfile.getStatus() and self.config.settingAutoAnalyze:
                                tr = self.tunePimp.getTrack(fileId)
                                tr.lock()
                                tr.setStatus(tunepimp.ePending)
                                tr.unlock()
                                self.tunePimp.wake(tr)
                                self.tunePimp.releaseTrack(tr)
                            else:
                                self.setStatusText("")
                                if tpfile.getStatus() == tunepimp.ePending and tpfile.getPUID() == None:
                                    tpfile.setPUID("")
                                unmatched = self.albumManager.get(albummanager.unmatchedFiles)
                                tpfile.moveToAlbumAsUnlinked(unmatched)

                    elif status == tunepimp.eRecognized:
                        self.setStatusText(_("File %s identified!") % (fileName,))

                        tr = tpfile.getTrack()
                        ldata = tr.getLocalMetadata()
                        sdata = tr.getServerMetadata()
                        tpfile.releaseTrack(tr)

                        # Check for local (from tags) and/or server metadata (from PUID lookup)
                        if sdata.albumId and sdata.trackId:
                            mdataList = (sdata, ldata)
                        else:
                            mdataList = (ldata,)
                        
                        for mdata in mdataList:
                            if mdata.albumId:
                                al = self.albumManager.add(mdata.albumId)
                                if al.isLoaded():
                                    # Try to use trackId
                                    seq = None
                                    if mdata.trackId:
                                        seq = al.findSeqOfTrack(mdata.trackId)
                                        if seq is not None:
                                            tpfile.moveToAlbumAsLinked(al, seq)
                                    # If we have no trackId, then try to match by metadata
                                    if seq is None:
                                        matchDict = self.albumManager.matchFileIdsToAlbum([fileId], mdata.albumId)
                                        for trackId in matchDict.values():
                                            seq = al.findSeqOfTrack(trackId)
                                            if seq is not None:
                                                tpfile.moveToAlbumAsLinked(al, seq)
                                elif al.hasError():
                                    unmatched = al.getUnmatchedFiles()
                                    unmatched.append(fileId)
                                    self.albumPanel.moveFilesToUnmatched(unmatched)
                                else:
                                    tpfile.moveToAlbumAsUnlinked(al)
                                break
                        
                        # Move the file out of "New Files" if it wasn't matched to an album
                        if tpfile.getAlbum().getId() == albummanager.pendingFiles:
                            unmatched = self.albumManager.get(albummanager.unmatchedFiles)
                            tpfile.moveToAlbumAsUnlinked(unmatched) 

                    elif status == tunepimp.ePending:
                        if tpfile.getStatus() == tunepimp.ePending:
                            tr = self.tunePimp.getTrack(fileId)
                            tr.lock()
                            tr.setStatus(tunepimp.eUnrecognized)
                            tr.unlock()
                            self.tunePimp.wake(tr)
                            self.tunePimp.releaseTrack(tr)
                        else:
                            #self.setStatusText(_("Analyzing file %s ...") % (fileName,))
                            pass
                        
                    elif status == tunepimp.ePUIDLookup:
                        tr = self.tunePimp.getTrack(fileId)
                        tr.lock()
                        puid = tr.getPUID()
                        tr.unlock()
                        self.tunePimp.releaseTrack(tr)
                        tpfile.setPUID(puid)
                        self.context.puidmanager.lookup(fileId, puid)
                        
                    elif status == tunepimp.eFileLookup:
                        self.setStatusText(_("Looking up a metadata on file %s ...") % (fileName,))
                        
                    elif status == tunepimp.ePUIDCollision:
                        self.setStatusText(_("PUID collision on file %s!") % (fileName,))
                        unmatched = self.albumManager.get(albummanager.unmatchedFiles)
                        tpfile.moveToAlbumAsUnlinked(unmatched)
                        
                    elif status == tunepimp.eUserSelection:
                        self.setStatusText("")
                        unmatched = self.albumManager.get(albummanager.unmatchedFiles)
                        tpfile.moveToAlbumAsUnlinked(unmatched)
                        
                    elif status == tunepimp.eVerified:
                        pass

                    elif status == tunepimp.eSaved:
                        album = tpfile.getAlbum()
                        if album:
                            album.checkTrackUnsaved(tpfile)
                        self.albumPanel.updateFile(fileId)

                    elif status == tunepimp.eDeleted:
                        pass

                    elif status == tunepimp.eError:
                        self.setStatusText(_("Error while processing file %s.") % (fileName,))
                        self.albumPanel.moveFilesToError([fileId])

                    tpfile.setStatus(status)
                    
                else:        
                    debug.debug("Unhandled callback event. fileId: %d type: %s status: %s"
                                % (fileId, tpmanager.labelCallbackEnum(type),
                                   tpmanager.labelFileStatusEnum(status)),
                                "tunepimp.event.unhandled")

    def OnSubmit(self, event):
        self.context.puidmanager.submit()
                    
    def OnFileAdd(self, event):
        for filename in event.getFileNames():
            assert filename.__class__.__name__ == "unicode"
            if self.isSupported(filename):
                fileId = self.tunePimp.addFile(filename, 0)
                if event.getAlbumId:
                    self.dropTargetTrackIdCache[fileId] = [ event.getAlbumId(), event.getAlbumSeq() ]
                debug.debug("Added %s as file #%d" % (filename, fileId), "tunepimp.add")

    def OnDirAdd(self, event):
        for dirname in event.getDirNames():
            assert dirname.__class__.__name__ == "unicode"
            self.tunePimp.addDir(dirname)

    def OnShowItemDetails(self, event):
        self.metadataPanel.showItemDetails(event.getItem())

    def OnShowDirectory(self, event):
        self.fileSelectionPanel.showDirectory(event.getDir())

    def OnUpdateFile(self, event):
        self.albumPanel.updateFile(event.getFileId())

    def OnAnalyzeFile(self, event):
        for fileId in event.getFileIdList():
            tr = self.tunePimp.getTrack(fileId)
            tr.lock()
            tr.setStatus(tunepimp.eMetadataRead)
            tr.unlock()
            self.tunePimp.wake(tr)
            self.tunePimp.releaseTrack(tr)
        self.albumPanel.moveFilesToPending(event.getFileIdList())

    def OnAnalyzeFile2(self, event):
        for fileId in event.getFileIdList():
            tr = self.tunePimp.getTrack(fileId)
            tr.lock()
            tr.setStatus(tunepimp.ePending)
            tr.unlock()
            self.tunePimp.wake(tr)
            self.tunePimp.releaseTrack(tr)
        self.albumPanel.moveFilesToPending(event.getFileIdList())

    def OnClearError(self, event):
        tr = self.tunePimp.getTrack(event.getFileId())
        tr.lock()
        tr.setStatus(tunepimp.eRecognized)
        tr.unlock()
        self.tunePimp.releaseTrack(tr)
        self.albumPanel.updateFile(event.getFileId())
        self.metadataPanel.showItemDetails(self.albumPanel.getCurrentSelection())

    def OnForceSave(self, event):
        tr = self.tunePimp.getTrack(event.getFileId())
        tr.lock()
        tr.setChanged()
        if tr.getStatus() == tunepimp.eSaved:
            tr.setStatus(tunepimp.eRecognized)
        tr.unlock()
        self.tunePimp.releaseTrack(tr)
        self.albumPanel.updateFile(event.getFileId())
        self.metadataPanel.showItemDetails(self.albumPanel.getCurrentSelection())

    def OnReloadAlbum(self, event):
        '''
        This causes the album to be reloaded from the main server.
        '''
        albumId = event.getId()
        self.albumManager.updateFromMainServer(albumId)

    def OnGeneratePlaylist(self, event):
        
        album = self.albumManager.get(event.getId())
        pls = playlist.Playlist(self, album)
        
        wildcard = []
        for fmt in pls.formats:
          wildcard.append("%s|%s" % (fmt[1], fmt[0]))      
        
        defaultDir = self.config.persistDirControlPath
        if not defaultDir:
            defaultDir = os.getcwd()
        
        dlg = wx.FileDialog(self, message=_("Select file") + ":", 
                           defaultDir=defaultDir, defaultFile="", wildcard="|".join(wildcard),
                           style=wx.SAVE | wx.CHANGE_DIR)
        if dlg.ShowModal() == wx.ID_OK:
            fileName = dlg.GetPath() 
            pls.save(fileName, dlg.GetFilterIndex())
            self.setStatusText(_('Playlist %s saved.') % fileName)
            self.fileSelectionPanel.refreshPaths([os.path.dirname(fileName)])
        
    def OnGenerateCuesheet(self, event):
        
        album = self.albumManager.get(event.getId())
        
        fileId = album.getTrack(0).getFileId()
        if fileId >= 0:
            tr = self.tunePimp.getTrack(fileId)
            tr.lock()
            fileName = tr.getFileName()
            tr.unlock()
            self.tunePimp.releaseTrack(tr)
        else:
            return

        period = fileName.rfind('.')
        cueFileName = u'%s.cue' % (fileName[:period])
        defaultDir = os.path.dirname(fileName)

        dlg = wx.FileDialog(self, message=_("Select file") + ":", 
                           defaultDir=defaultDir, defaultFile=cueFileName, wildcard="CUE (*.cue)|*.cue|All Files (*.*)|*.*",
                           style=wx.SAVE | wx.CHANGE_DIR)
        if dlg.ShowModal() == wx.ID_OK:
            cueFileName = dlg.GetPath() 
            try:
                cue = cuesheet.Cuesheet(album)
                cue.save(cueFileName, fileName)
            except:
                traceback.print_exc()
                dlg = wx.MessageDialog(self, _("Error while saving cuesheet %s.") % cueFileName, _("Save error"), wx.OK)
                dlg.ShowModal()
            else:
                self.setStatusText(_('Cuesheet %s saved.') % cueFileName)
                self.fileSelectionPanel.refreshPaths([os.path.dirname(cueFileName)])
        
    def OnRemoveTPFileCommand(self, event):
        try:
            tpfile = self.tpmanager.findFile(event.getFileId())
        except LookupError:
            return
        else:
            tpfile.removeFromSystem()

    def OnLinkFile(self, event):
        self.albumPanel.linkFile(event.getFileId(), event.getTrackId())

    def OnAnalyzeButton(self, event):
        self.albumPanel.analyze()

    def OnRemoveButton(self, event):
        if len(self.refreshDirDict): return
        self.albumPanel.remove()

    def OnSaveButton(self, event):
        if len(self.refreshDirDict): return

        # If we have a track that is open for editing, save the details to tunepimp
        self.metadataPanel.saveTrackInfo()

        # And now save them!
        self.albumPanel.save()

    def OnSaveFile(self, event):
        assert len(self.refreshDirDict) == 0 # why?

        if self.config.settingMoveFiles and \
           (self.config.settingDestDir == '' or \
            not self.wpath.isdir(self.config.settingDestDir)):
                dlg = wx.MessageDialog(self, _("The move files option is enabled, but the " 
                    "destination directory is not specified or invalid.\nPlease fix " 
                    "the move option or the destination directory option and try again."), 
                    _("Save error"), wx.OK)
                dlg.ShowModal()
                return

        fileList = event.getFileIdList()
        for fileId in fileList:
            tr = self.tunePimp.getTrack(fileId)
            tr.lock()
            fileName = tr.getFileName()
            tr.unlock()
            self.tunePimp.releaseTrack(tr)
            self.refreshDirDict[self.wpath.dirname(fileName)] = fileId 

        if not len(fileList):
            self.refreshDirDict = {}
            return

        ret = self.tunePimp.writeTags(fileList)
        if ret:
            self.busyCursor = wx.BusyCursor()
            self.enableSave(False)
            self.enableRemove(False)
            self.enableCluster(False)
        else:
            self.writeTagsComplete()

    def writeTagsComplete(self):
       
        if self.config.settingDestDir:
            self.refreshDirDict[self.config.settingDestDir] = -1
        self.fileSelectionPanel.refreshPaths(self.refreshDirDict.keys())

        foundErrors = False
        for fileId in self.refreshDirDict.values():    
            if fileId >= 0:
                tr = self.tunePimp.getTrack(fileId)
                tr.lock()
                status = tr.getStatus()
                tr.unlock()
                self.tunePimp.releaseTrack(tr)
                if status == tunepimp.eError: foundErrors = True

        if foundErrors:
            dlg = wx.MessageDialog(self, _("Not all files could be saved. Please check for and examine "
                                           "tracks that have an error icon in front of them. To retry "
                                           "saving the track, right click the track and select 'Clear Error'"), 
                                         _("Save error"), wx.OK)
            dlg.ShowModal()

        sel = self.albumPanel.getCurrentSelection()
        if len(sel) == 1:
            data = sel[0]
        else:
            data = None
        
        self.metadataPanel.showItemDetails(data)
        self.refreshDirDict = {}
        assert len(self.refreshDirDict) == 0

    def OnClusterFiles(self, event):
        if len(self.refreshDirDict): return

        debug.debug("OnClusterFiles triggered")
        # Is the try/finally block necessary here?
        busyCursor = wx.BusyCursor()
        try:
            self.albumManager.clusterUnmatchedFiles(event.getThreshold())
        finally:
            del busyCursor

    def OnMoveClustersToUnmatched(self, event):
        self.albumPanel.moveClustersToUnmatched()

    def OnMatchFilesToAlbum(self, event):

        fileList = event.getFileList()
        matchDict = self.albumManager.matchFilesToAlbum(fileList, event.getAlbumId())
        if matchDict:
            for match in matchDict:
                self.albumPanel.linkFile(match, matchDict[match])

    def OnMatchFileIdsToAlbum(self, event):

        album = self.albumManager.get(event.getAlbumId())

        matchDict = self.albumManager.matchFileIdsToAlbum(copy.copy(event.getFileIdList()), event.getAlbumId())
        if matchDict:
            for match in matchDict:
                tpfile = self.tpmanager.findFile(match)
                seq = album.findSeqOfTrack(matchDict[match])
                tpfile.moveToAlbumAsLinked(album, seq)

    def OnMatchDirToAlbum(self, event):

        fileList = []
        dirName = event.getDirName()
        for f in self.wpath.listdir(dirName):
            path = self.wpath.join(dirName, f)
            if self.wpath.isfile(path):
                fileList.append(unicode(path))

        wx.PostEvent(self, events.FileAddEvent(fileList, event.getAlbumId(), -1))

    def OnAddFileToCluster(self, event):
        tpfile = self.tpmanager.findFile(event.getFileId())
        clusterAlbum = self.context.clustermanager.findCluster(event.getClusterId())
        tpfile.moveToCluster(clusterAlbum)

    def OnRemoveClusterAlbum(self, event):
        self.albumPanel.removeClusterAlbum(event.getClusterId())

    def getAlbumManager(self):
        return self.albumManager

    def OnCoverArtLoad(self, event):
        asin = event.getAsin()
        if asin == self.currentAsin:
           image = self.coverArtCache.getImage(asin)
           if image:
               self.coverArtPanel.showImage(image, asin)

    def OnCoverArtShow(self, event):
        self.showCoverArt(event.getAsin())

    def OnAlbumUpdate(self, event):
        self.albumPanel.updateAlbum(event.getId())

        # If the album failed to load, move its files back to "unmatched" and
        # remove this album
        try:
            al = self.albumManager.get(event.getId())
        except LookupError:
            pass
        else:
            if al.hasError():
                unmatched = self.albumManager.get(albummanager.unmatchedFiles)
                debug.debug("album has error, moving %s back to unmatched" %
                        repr(al.getUnmatchedFiles()) )
                for fileId in al.getUnmatchedFiles():
                    tpfile = self.tpmanager.findFile(fileId)
                    tpfile.moveToAlbumAsUnlinked(unmatched)

    def OnAlbumTrackCountsChanged(self, event):
        if event.getId() in (albummanager.pendingFiles, albummanager.unmatchedFiles):
            self.enableCluster(self.context.clustermanager.CanCluster())

        self.albumPanel.albumTrackCountsChanged(event.getId())

    def OnAlbumFilesAdded(self, event):
        self.albumPanel.albumFilesAdded(event.getId(), event.getFileList())

    def OnAlbumFilesRemoved(self, event):
        self.albumPanel.albumFilesRemoved(event.getId(), event.getFileList())

    def OnAlbumRemoveUnmatched(self, event):
        self.albumPanel.removeAlbumUnmatched(event.getId())

    def OnAlbumMoveUnmatched(self, event):
        self.albumPanel.moveAlbumUnmatchedToUnmatched(event.getId())

    def OnMoveFilesToUnmatched(self, event):
        self.albumPanel.moveFilesToUnmatched(event.getFileIdList())

    def OnClusterAdded(self, event):
        self.albumPanel.clusterAdded(event.getCluster())
    def OnClusterRemoved(self, event):
        self.albumPanel.clusterRemoved(event.getCluster())
    def OnClusterChanged(self, event):
        self.albumPanel.clusterChanged(event.getCluster())
    def OnClusterFileAdded(self, event):
        self.albumPanel.clusterFileAdded(event.getCluster(), event.getFileId())
    def OnClusterFileRemoved(self, event):
        self.albumPanel.clusterFileRemoved(event.getCluster(), event.getFileId())

    def OnAlbumAdd(self, event):
        self.albumPanel.addAlbum(event.getId())

    def OnAlbumRemoved(self, event):
        self.albumPanel.albumRemoved(event.getId())

    def OnAlbumRemove(self, event):
        self.albumManager.remove(event.getId())
       
    def OnBrowserMessage(self, event):
        action = event.getAction()
        args = event.getArgs()
        if action == u'init' and len(args) > 0:
            self.taggerPort = args[u'port']
            return

        self.Raise()
        if action == u'openalbum' and args[u'id']:
           self.albumManager.add(args[u'id'])

    def showCoverArt(self, asin):
        if self.currentAsin != asin:
           self.currentAsin = asin
           image = self.coverArtCache.getImage(asin)
           if image:
              self.coverArtPanel.showImage(image, asin)
        
    def OnSplitterDClick(self, event):
        self.splitter.Unsplit(self.fileSelectionPanel)    

    def OnMenuDump(self, event):
        n = debug.NestedDumper()

        n(u"picard dump at %s" % ( time.asctime(time.localtime()) ))

        self.tpmanager.dumpState(n.nest())
        self.albumManager.dumpState(n.nest())
        self.context.clustermanager.dumpState(n.nest())
        self.albumPanel.dumpState(n.nest())

        n(u"end of dump")

    def OnMenuExit(self, event):
        self.shutdown()

    def OnCloseWindow(self, event):
        self.shutdown()

    def OnOptions(self, event):
        if len(self.refreshDirDict): return
        self.metadataPanel.showOptions()

    def OnAbout(self, event):
        if len(self.refreshDirDict): return
        self.metadataPanel.showOptions(-1)

    def OnHelp(self, event):
        page = launch.Launch(self)
        page.launch("http://musicbrainz.org/doc/PicardDocumentation")

    def OnQuickStart(self, event):
        page = launch.Launch(self)
        page.launch("http://musicbrainz.org/doc/HowToTagFilesWithPicard")

    def OnAddFile(self, event):
        if len(self.refreshDirDict): return

        defaultDir = self.config.persistAddFilePath
        if not defaultDir:
            defaultDir = os.getcwd()

        dlg = wx.FileDialog(self, message=_("Select files to tag") + ":", 
                           defaultDir=defaultDir, defaultFile="", wildcard=self.fileFilter,
                           style=wx.OPEN | wx.MULTIPLE | wx.CHANGE_DIR)
                
        if dlg.ShowModal() == wx.ID_OK:
            self.config.persistAddFilePath = self.wpath.dirname(dlg.GetPath())
            paths = dlg.GetPaths()
            for path in paths:
                if self.isSupported(path):
                    id = self.tunePimp.addFile(path, 0)
                    debug.debug("Added file #%d" % id, "tunepimp.add")

    def OnAddDir(self, event):
        if len(self.refreshDirDict): return

        defaultDir = self.config.persistAddDirPath
        if not defaultDir:
            defaultDir = os.getcwd()
        dlg = wx.DirDialog(self, message=_("Select directory that contains music files"), 
                           defaultPath=defaultDir)
                
        if dlg.ShowModal() == wx.ID_OK:
            self.config.persistAddDirPath = dlg.GetPath()
            self.tunePimp.addDir(dlg.GetPath())
     
    def getTaggerPort(self):
        return self.taggerPort

    def isSupported(self, filename):
        filename = filename.lower()
        for ext in self.supportedExtensions:
            if filename.endswith(ext):
                return True
        return False

    def enableSave(self, state):
        self.toolbar.enableSave(state)
        self.fileMenu.Enable(self.fileMenuSaveId, state)
        
    def enableCluster(self, state):
        self.toolbar.enableCluster(state)
        
    def enableRemove(self, state):
        self.toolbar.enableRemove(state)
        
    def enableAnalyze(self, state):
        self.toolbar.enableAnalyze(state)
        
        
#---------------------------------------------------------------------------

class TaggerApp(wx.App):
  
    def __init__(self, localeDir=None, pluginsDir=None):

        configObj = None
        if sys.platform == 'win32':
            configObj = wx.Config(APPNAME, VENDORNAME)
        else:
            configObj = wx.Config(APPNAME, VENDORNAME, ".mbtaggerconf")

        self.config = util.ConfigSettings(configObj, APPVERSION)
        self.config.readSettings()

        # Check to make sure that the loaded encoding is valid:
        try:
            "foobar!".encode(self.config.settingIOEncoding);
        except:
            self.config.settingIOEncoding = 'utf-8';

        self.config.pluginsDir = pluginsDir

        if self.config.settingLanguage:
            os.environ['LANGUAGE'] = ''
            os.environ['LANG'] = self.config.settingLanguage

        if sys.platform == 'win32':
            try:
                locale.setlocale(locale.LC_ALL, os.environ['LANG'])
            except KeyError:    
                os.environ['LANG'] = locale.getdefaultlocale()[0]
                locale.setlocale(locale.LC_ALL, '')
            except:
                pass
        else:
            try:
                locale.setlocale(locale.LC_ALL, '')
            except:
                pass

        try:
            self.translation = gettext.translation('picard', localeDir)
            self.translation.install(True)
        except IOError:
            pass

        wx.App.__init__(self, 0)

    def OnInit(self):

        frame = TaggerFrame(None, -1, APPNAME, self.config)
        
        frame.Show(True)
        self.SetTopWindow(frame)

        # Return a success flag
        return True

    def OnExit(self):
        self.config.writeSettings()

#---------------------------------------------------------------------------

def main(localeDir=None, pluginsDir=None):

    try:
        foo = wx.CLOSE_BOX
    except:
        print "This application requires wxPython 2.5.x.x or later to run"
        sys.exit(-1)

    app = TaggerApp(localeDir, pluginsDir)
    app.MainLoop()


