# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2006 Lukas Lalinsky. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Lukas Lalinsky
#
#
# ***** END LICENSE BLOCK *****

import os.path
from musicbrainz2.utils import getCountryName

class CuesheetExcetion(Exception):
    pass

class Cuesheet:
    """Cuesheet writer. 
       
    ftp://64.251.26.164/pub/goldenhawk/download/cdrwin.pdf
    Appendix A - Cuesheet Commands
    """
    
    def __init__(self, album):
        self.album = album
       
    def _escape(self, text):
        # anyone know how to escape quotation marks in .cue files?
        return text.replace('"', '\'')
        
    def load(self, fileName):
        f = file(fileName, 'rt')
        self.tracks = {}
        trackNum = 0
        self.tracks[trackNum] = {}
        for line in f.readlines():
            line = line.strip().decode('utf-8', 'replace') 
            if line.startswith(u'\ufeff'):
                line = line[1:]
            cmd = line.split(' ', 1)
            if cmd[0] == 'TRACK':
                trackNum = int(cmd[1].split(' ', 1)[0])
                self.tracks[trackNum] = {}
            else:
                if cmd[0] == 'REM':
                    cmd2 = cmd[1].split(' ', 1)
                    if len(cmd2) == 2:
                        cmd[0] = '%s %s' % (cmd[0], cmd2[0])
                        cmd[1] = cmd2[1]
                if not self.tracks[trackNum].has_key(cmd[0]):
                    self.tracks[trackNum][cmd[0]] = []
                self.tracks[trackNum][cmd[0]].append(cmd[1])
        f.close()
        
    def save(self, cueFileName, fileName):
        album = self.album
        
        if os.path.dirname(cueFileName) == os.path.dirname(fileName):
            fileName = os.path.basename(fileName)
        
        if fileName[-3:].lower() == 'mp3':
            format = 'MP3'
        else:
            format = 'WAVE'
            
        self.tracks = {}
        if os.path.isfile(cueFileName):
            self.load(cueFileName)
            
        for i in range(album.getNumTracks() + 1):
            if not self.tracks.has_key(i):
                self.tracks[i] = {}
        
        lines = []
        w = lines.append

        w(u'TITLE "%s"' % self._escape(album.getName()))
        w(u'PERFORMER "%s"' % self._escape(album.getArtist().getName()))
        for key in self.tracks[0].keys():
            if key not in ['TITLE', 'PERFORMER', 'FILE', 'REM DATE', 
                           'REM MUSICBRAINZ_ALBUM_ID', 'REM RELEASE_TYPE',
                           'REM RELEASE_STATUS', 'REM COUNTRY',
                           'REM MUSICBRAINZ_ALBUM_ARTIST_ID']:
                for value in self.tracks[0][key]:
                    w(u'%s %s' % (key, value))
        w(u'REM DATE %04d' % album.getReleaseYear())
        country = album.getReleaseCountry()
        if country != '':
            country = getCountryName(country) or country 
        w(u'REM COUNTRY "%s"' % self._escape(country))
        w(u'REM RELEASE_TYPE "%s"' % self._escape(album.getReleaseType()))
        w(u'REM RELEASE_STATUS "%s"' % self._escape(album.getReleaseStatus()))
        w(u'REM MUSICBRAINZ_ALBUM_ID %s' % album.getId())
        w(u'REM MUSICBRAINZ_ALBUM_ARTIST_ID %s' % album.getArtist().getId())
        w(u'FILE "%s" %s' % (self._escape(fileName), format))
            
        index = 0
        for i in range(album.getNumTracks()):
            track = album.getTrack(i)
            mm = index / 60.0
            ss = (mm - int(mm)) * 60.0
            ff = (ss - int(ss)) * 75.0
            index += track.getDuration() / 1000.0
            w(u'  TRACK %02d AUDIO' % (i+1))
            w(u'    TITLE "%s"' % self._escape(track.getName()))
            w(u'    PERFORMER "%s"' % self._escape(track.getArtist().getName()))
            if not self.tracks[i+1].has_key('INDEX'):
                w(u'    INDEX 01 %02d:%02d:%02d' % (int(mm), int(ss), int(ff)))
            for key in self.tracks[i+1].keys():
                if key not in ['TITLE', 'PERFORMER']:
                    for value in self.tracks[i+1][key]:
                        w(u'    %s %s' % (key, value))

        f = file(cueFileName, 'wt')
        f.write(u'\ufeff'.encode('utf-8'))
        f.write(u'\n'.join(lines).encode('utf-8'))
        f.close()
            

