<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.12 $
 *
 *  ABOUT
 *  -----
 *
 *  Main Kolab admin object.
 *
 */

/** This is based on smarty. */
require_once('smarty/Smarty.class.php');

/** Handles the generation of the menu. */
require_once('Kolab/Webadmin/menu.php');

/** Classes for authentication. */
require_once('Kolab/Webadmin/auth.class.php');
require_once('Kolab/Webadmin/ldap.class.php');

/** Provides functionality for locale handling. */
require_once('Kolab/Webadmin/locale.php');

/** We need some sieve tools. */
require_once 'Kolab/Webadmin/sieveutils.class.php';

/** We need some form tools. */
require_once 'Kolab/Webadmin/form.class.php';

/** Provides handling for the Kolab objects. */
require_once 'Kolab/Webadmin/object.php';

/**
 * This class combines methods common to most pages of the Kolab web
 * admin frontend.
 *
 * $Header: /home/kroupware/jail/kolabrepository/server/php-kolab/Kolab_Webadmin/Webadmin/webadmin.class.php,v 1.12 2007/09/18 06:08:48 gunnar Exp $
 *
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Webadmin
 */
class KolabWebadmin extends Smarty {

    /**
     * A link to the object that handles our ldap connection.
     *
     * @var KolabLDAP
     */
    var $_ldap;

    /**
     * The handler for the Kolab objects
     *
     * @var KolabObjectDb
     */
    var $_db;

    /**
     * A link to the object that handles our authentication
     * information.
     *
     * @var KolabAuth
     */
    var $_auth;

    /**
     * A class that handles access to the Kolab user db.
     *
     * @var KolabUsers
     */
    var $_users;

    /**
     * A sieve connection.
     *
     * @var KolabSieve
     */
    var $_sieve;

    /**
     * The configuration for the webadmin.
     *
     * @var array
     */
    var $_config;

    /**
     * User currently logged in.
     *
     * @var KolabObject
     */
    var $_user;

    /**
     * An array of section accessible to the current user.
     *
     * @var array
     */
    var $_menuitems;

    /**
     * The current section selected by the user
     *
     * @var string
     */
    var $_section;

    /**
     * Initialize the web admin class
     *
     * @param array $config An array of basic configuration variables.
     */
    function KolabWebadmin($config) 
    {
        // Start the session
        session_start();

        /* 
         * Prepare authentication.
         */

        $this->_ldap =& new KolabLDAP($config['ldap_master_uri'],
                                      $config['base_dn'],
                                      $config['php_dn'],
                                      $config['php_pw']);
        $this->_db   =& new KolabObjectDb($this->_ldap);
        $this->_auth =& new KolabAuth($this->_ldap, $config);

        /* 
         * Initialize Smarty.
         */

        // Call the parent initialization
        $this->Smarty();

        $basedir = $config['smarty_basedir'];
        $this->template_dir = $basedir . 'templates/';
        $this->compile_dir = $basedir . 'templates_c/';
        $this->config_dir = $basedir . 'configs/';
        $this->cache_dir = $basedir . 'cache/';

        $this->assign( 'topdir', $config['topdir'] );
        $this->assign( 'prefix', $config['webserver_web_prefix'] );
        $this->assign( 'self_url', $_SERVER['REQUEST_URI'] );

        /* 
         * Handle locale support.
         */

        // Check if the language should be changed
        if(!empty($_REQUEST["lang"])) {
            setLanguage($_REQUEST["lang"]);
        }

        // I18N support information
        $language = getLanguage();
        putenv("LANG=$language"); 
        putenv("LANGUAGE=$language"); 
        setlocale(LC_ALL, $language);

        $domain = "messages";

        bindtextdomain($domain, $config['locale_dir']); 
        bind_textdomain_codeset($domain, "UTF-8");
        textdomain($domain);

        // Added for i18n management (Romain 05-03-03)
        $this->register_function("tr", "translate");

        $this->assign( 'currentlang', $language );
        $this->assign( 'languages', offered_languages());

        $cleanurl = preg_replace('/(\?|&)lang=(.*)(&|$)/', 
                                 '', 
                                 $_SERVER['REQUEST_URI']);
        $this->assign( 'lang_url', 
                       strpos($cleanurl, '?') === false ?
                       ($cleanurl . '?lang='):
                       ($cleanurl . '&lang=') 
        );

        /* 
         * Time to check our authentication status.
         */
        $authenticated = $this->_auth->authenticate();

        // The user is not authenticated
        if (!$authenticated || is_a($authenticated, 'PEAR_Error')) {
            $this->assign( 'uid', '' );
            $this->assign( 'group', '' );
            $this->assign( 'menuitems', array() );
            // Show the login page
            $this->assign( 'maincontent', 'login.tpl' );
            if (is_a($authenticated, 'PEAR_Error')) { 
                $this->assign( 'errors', array($authenticated->getMessage()));
                $this->assign( 'page_title', _('Error') );
            }
            if (!$authenticated) {
                $this->assign( 'page_title', _('Login') );
            }
            // Display and exit
            $this->render();
            exit();
        }

        // User is authenticated
        $this->_user = $authenticated;
        $this->assign( 'uid', $this->_user->getUid() );
        $this->assign( 'group', $this->_user->group() );

        // Get the menu the user is able to see
        $this->_menuitems = generate_menu($this->_user, 
                                          $config['topdir']);

        $this->_config = $config;
    }

    /**
     * Check if all domains have been correctly initialized.
     *
     * @return boolean true if all domains have the necessary
     * distribution lists
     */
    function domains_valid() 
    {
        $domains = $this->_ldap->domains();
        foreach( $domains as $domain ) {
            if( !$this->_ldap->exists_group( 'hostmaster@' . $domain ) ||
                !$this->_ldap->exists_group( 'postmaster@' . $domain ) ||
                !$this->_ldap->exists_group( 'abuse@' . $domain ) ||
                !$this->_ldap->exists_group( 'virusalert@' . $domain ) ||
                !$this->_ldap->exists_group( 'MAILER-DAEMON@' . $domain ) ) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Produce the page headers.
     */
    function output_headers()
    {
        header("Content-Type: text/html; charset=UTF-8");
        header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT");
        header("Cache-Control: no-cache, must-revalidate");
        header("Pragma: no-cache");
    }

    /**
     * Set the page content template.
     *
     * @param string $content The content of the page
     */
    function setContent($contenttemplate)
    {
        $this->assign( 'maincontent', $contenttemplate );
    }

    /**
     * Set the page title.
     *
     * @param string $title The title of the page
     */
    function setTitle($title)
    {
        $this->assign( 'page_title', $title );
    }

    /**
     * Set the web admin section
     *
     * @param string $section The section of the web admin
     */
    function setSection($section = '')
    {
        $this->_section = $section;

        $title = 'Kolab';
        $submenu = array();

        if (isset($this->_menuitems[$section])) {
            $this->_menuitems[$section]['selected'] = 'selected';
            if (isset($this->_menuitems[$section]['title'])) {
                $title = $this->_menuitems[$section]['title'];
            }
            if (isset($this->_menuitems[$section]['submenu'])) {
                $submenu = $this->_menuitems[$section]['submenu'];
            }
        }

        $this->assign( 'page_title', $title);
        $this->assign( 'submenuitems', $submenu);
    }

    /**
     * Render a page.
     *
     * @param string $template Name of the template to display
     */
    function render($template = 'page.tpl')
    {
        $this->output_headers();

        // Set the visible menu
        $this->assign( 'menuitems', $this->_menuitems );

        // Render the page
        $this->display($template);
        
    }
    
    /**
     * Get the DN of the current user
     *
     * @return string The distinguished name of the current user.
     */
    function getCurrentDn()
    {
        return $this->_user->getDn();
    }

    
    /**
     * Check if the current user is member of one of these groups
     *
     * @param array $groups The possible groups
     *
     * @return boolean Is the current user member in one of these
     * groups?
     */
    function hasGroup($groups)
    {
        return (in_array($this->_user->group(), $groups));
    }

    /**
     * Return the sieve connection.
     */
    function getSieve()
    {
        if (empty($this->_sieve)) {
            $obj = $this->_ldap->read( $this->_dn );
            $this->_sieve =& new KolabSieve( $this->_auth->uid(), 
                                             $this->_auth->password(), 
                                             $obj['kolabHomeServer'][0] );
        }
        return $this->_sieve;
    }

    /**
     * Build a LDAP filter for leading characters based on an array.
     *
     * @param string $attr  The attribute to filter on
     * @param array  $chars An array of acceptable leading characters.
     *
     * @return string A LDAP search filter.
     */
    function alphaFromArray($attr, $chars) 
    {
        $filter = '(|';
        foreach ($chars as $char) {
            $filter .= '(' . $attr . '=' . $char . '*)';
        }
        return $filter . ')';
    }

    /**
     * Build an alphabetic LDAP filter.
     *
     * @param string $attr  The attribute to filter on
     *
     * @return string A LDAP search filter.
     */
    function buildAlphaFilter($attr = 'sn')
    {
        if ($this->_config['alpha_filter_type'] == 'none') {
            return '';
        }

        $alphagroup = KolabForm::getRequestVar('alphagroup');


        if ($this->_config['alpha_filter_type'] == 'group') {
            switch( $alphagroup ) {
            case 'a': 
                return $this->alphaFromArray($attr, array('a', 'b', 'c', 'd', 'e', 'f'));
            case 'g': 
                return $this->alphaFromArray($attr, array('g', 'h', 'i', 'j', 'k', 'l'));
            case 'm': 
                return $this->alphaFromArray($attr, array('m', 'n', 'o', 'p', 'q', 'r'));
            case 's': 
                return $this->alphaFromArray($attr, array('s', 't', 'u', 'v', 'w', 'x', 'y', 'z'));
            case 'other': 
                return $this->alphaFromArray($attr, array('æ', 'ø', 'å', 'ä', 'ö', 'ü', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'));
            case 'all': 
                return '';
            case 'none':
                return "(&($attr=a*)($attr=b*))";
            default: 
                return '';
            }
        } else {
            if (strlen($alphagroup) == 1 && 
                ord($alphagroup) > 96 && 
                ord($alphagroup) < 123) {
                return $this->alphaFromArray($attr, array($alphagroup));
            }
            switch( $alphagroup ) {
            case 'other': 
                return $this->alphaFromArray($attr, array('æ', 'ø', 'å', 'ä', 'ö', 'ü', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'));
            case 'all': 
                return '';
            case 'none':
                return "(&($attr=a*)($attr=b*))";
            default: 
                return '';
            }
        }
        
        return '';
    }

    /**
     * Build a LDAP filter based on user settings.
     *
     * @return string A LDAP search filter.
     */
    function buildUserFilter()
    {
        if (!$this->_config['allow_user_filter']) {
            return '';
        }

        $filterattr = KolabForm::getRequestVar('filterattr');
        $filtertype = KolabForm::getRequestVar('filtertype');
        $filtervalue = KolabForm::getRequestVar('filtervalue');

        if( !in_array( $filterattr, array( 'cn', 'uid', 'mail' ) ) ) {
            $filterattr = 'cn';
        }
        
        if( !empty( $filtervalue ) ) {
            switch( $filtertype ) {
            case 'contains': // contains
                return '(' . $filterattr . '=*' .
                    $this->_ldap->escape($filtervalue) . '*)';
            case 'is': // is
                return '(' . $filterattr . '=' .
                    $this->_ldap->escape($filtervalue) . ')';
            case 'begins': // begins with
                return '(' . $filterattr . '=' .
                    $this->_ldap->escape($filtervalue) . '*)';
            case 'ends': // ends with
                return '(' . $filterattr . '=*' .
                    $this->_ldap->escape($filtervalue) . ')';
            default:
                return '(' . $filterattr . '=*' .
                    $this->_ldap->escape($filtervalue) . '*)';
            }
        }
        return '';
    }

    /**
     * Build a LDAP filter to select specifc domains for a domain
     * maintainer.
     *
     * @return string A LDAP search filter.
     */
    function buildDomainFilter()
    {
        if (!$this->hasGroup(array(KOLAB_OBJECT_DOMAINMAINTAINER))) {
            return '';
        }

        $dm =& new KolabDomainMaintainer($this->_ldap, $this->_dn);
        $domains = $dm->domains();
        if( !is_array($domains) ) {
            return '';
        }
        
        $domainfilter='(|';
        foreach( $domains as $domain ) {
            $domainfilter .= '(mail=*@' . $this->_ldap->escape($domain) . ')';
        }
        return $domainfilter . ')';
    }
    
    function listVisible($type)
    {
        $userfilter   = $this->buildUserFilter();
        $alphafilter  = $this->buildAlphaFilter();
        if ($type == KOLAB_OBJECT_USER ||
            $type == KOLAB_OBJECT_GROUP ||
            $type == KOLAB_OBJECT_SHAREDFOLDER) {
            $domainfilter = $this->buildDomainFilter();
        } else {
            $domainfilter = '';
        }
        $addfilter = $userfilter . $alphafilter . $domainfilter;
	
        if (!empty($addfilter)) {
            $addfilter = '(&' . $addfilter . ')';
        }
        return $this->_db->listObjects($type, null, $addfilter, 
                                       $this->_config['entries_per_page'],
                                       KolabForm::getRequestVar('page'));
    }

    /**
     * Run a text through htmlentities.
     *
     * @param string $text Text to transform
     *
     * @return string The transformed text.
     */
    function htmlentitites($text)
    {
        return KolabForm::htmlentities($text);
    }

    /**
     * Check if the given dn is maintainable by the current user
     *
     * @param string $dn DN of the user to be modified
     *
     * @return boolean True if the current user may modify the user
     * specified by the given dn.
     */
    function inMaintainerDomain($dn) {

        // both groups have full access
        if ($this->hasGroup(array(KOLAB_OBJECT_ADMIN,
                                  KOLAB_OBJECT_MAINTAINER))) {
            return true;
        }

        // user may not maintain anything
        if ($this->hasGroup(array(KOLAB_OBJECT_USER))) {
            return false;
        }
  
        // we have a domain maintainer. Get his domains
        $domains = $this->_user->domains();

        // retrieve the mail for the current dn
        $user = $this->_db->fetch($dn);
        $mail = $user->getMail();

        $ok = false;

        // Check if the mail is within that domain
        foreach( $domains as $domain ) {
            if( endsWith( $mail, '@'.$domain ) ) {
                $ok = true;
            }
        }
        return $ok;
    }

    /**
     * Get the LDAP object classes for the given DN
     *
     * @param string $dn DN of the object
     *
     * @return array An array of object classes
     */
    function storeUser($dn, $create = false) {

        // Handle the users object classes
        $oc = array('top', 'inetOrgPerson', 'kolabInetOrgPerson');
        if( $dn ) {
            $oc = $this->getObjectClasses($dn);
            if (is_a($oc, 'PEAR_Error')) {
                return $oc;
            }
        }

        // Get the interface to the user db

        if ($create) {
            $ldap_object['uid'] = ($uid == '') ? $ldap_object['mail'] : $userdata['uid'];
            $ldap_object['kolabHomeServer'] = $userdata['kolabhomeserver'];
        }

        if( !$ldap_object['alias'] && $action == 'firstsave' ) unset($ldap_object['alias']);
    
    }

    /**
     * Grab data from a posted form
     *
     * @return boolean True on success, a PEAR error otherwise
     */
    function parseUserFromPost()
    {

        $userdata = array();

        $userdata['sn'] = trim($_POST['sn']);
        $userdata['givenname'] = trim($_POST['givenname']);
        $userdata['password_0'] = $_POST['password_0'];
        $userdata['usermail'] = trim( strtolower( $_POST['user_mail'] ) );
        $userdata['domainmail'] = trim( strtolower( $_POST['domain_mail'] ) );
        $userdata['uid'] = trim( strtolower( $_POST['uid'] ) );
        $userdata['kolabhomeserver'] = trim($_POST['kolabhomeserver']);
        $userdata['accttype'] = (int) $_POST['accttype'];
        
        // Multiline data
        foreach( array( 'kolabDelegate', 'alias') as $attr ) {
            $userdata[$attr] = array_unique(
                array_filter(
                    array_map(
                        'trim',
                        preg_split('/\n/',
                                   $_POST[$attr]
                        )
                    ),
                    'strlen'
                )
            );
        }
        

        // Multiple entry data
        foreach( 
            array( 
                'title', 'o', 'ou', 'roomNumber', 'street', 'postOfficeBox',
                'postalCode', 'l', 'c', 'telephoneNumber',
                'facsimileTelephoneNumber', 'kolabFreeBusyFuture',
                'user_kolabinvitationpolicy', 'policy_kolabinvitationpolicy'
            ) as $attr ) {
            $count = 0;
            $key = $attr."_0";
            $args = array();
            while (isset($_POST[$key])) {
                $args[$count] = trim($_POST[$key]);
                $count++;
                $key = $attr."_".$count;
            }
            if ($count > 0) {
                $userdata[$attr] = $args;
            } elseif (!empty($_POST[$key])) {
                $userdata[$attr] = $_POST[$key];
            } else {
                $userdata[$attr] = array();
            }
        }
        
        // userquota
        if( isset( $_POST['cyrus-userquota'] ) ) {
            $userdata['cyrus-userquota'] = trim($_POST['cyrus-userquota']);
            if( empty( $userdata['cyrus-userquota'] ) ) {
                $userdata['cyrus-userquota'] = array();
            }
        }

    }
    


    /**
     * Apply attribute access settings to an array of entries.
     *
     * @param array $entries An array of form entries
     *
     * @return array The modified array of entries
     */
    function apply_attributeaccess( &$entries ) {

        $attributeaccess =& $this->_config['attribute_access'];
        foreach( $entries as $key=>$value ) {
            if( ereg( '(.*)_[0-9]', $key, $regs ) ) {
                $akey = $regs[1];
            } else {
                $akey = $key;
            }
            if( isset($attributeaccess[$akey] ) ) {
                if( $attributeaccess[$akey] == 'ro' ) {
                    $entries[$key]['attrs'] = 'readonly';
                } else if( $attributeaccess[$akey] == 'hidden' ) {
                    //$entries[$key]['attrs'] = 'hidden';
                    unset($entries[$key]);
                } else if( $attributeaccess[$akey] == 'mandatory' ) {
                    if( isset( $entries[$key]['validation'] ) ) {
                        if( is_array( $entries[$key]['validation'] ) ) {
                            $entries[$key]['validation'][] = 'notempty';
                        } else {
                            $entries[$key]['validation'][] = array( $entries[$key]['validation'], 
                                                                    'notempty' );
                        }
                    } else {
                        $entries[$key]['validation'] = 'notempty';          
                    }
                }
            }
        }
        return $entries;
    }

    /**
     * Update the password of the current user
     *
     * @param array $pass The new password
     */
    function updatePassword($pass) {
        $this->_auth->setPassword($pass);
    }
    

};

/*
  Local variables:
  mode: php
  indent-tabs-mode: f
  tab-width: 4
  buffer-file-coding-system: utf-8
  End:
  vim:encoding=utf-8:
 */
?>
