<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.10 $
 *
 *  ABOUT
 *  -----
 *
 *  A class for Kolab user management.
 *
 */

/** The basic Kolab object definition */
require_once('Kolab/Webadmin/object.php');

/**
 * This class combines methods common to most pages of the Kolab web
 * admin frontend.
 *
 * $Header: /home/kroupware/jail/kolabrepository/server/php-kolab/Kolab_Webadmin/Webadmin/user.class.php,v 1.10 2007/09/18 10:39:18 gunnar Exp $
 *
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Webadmin
 */
class KolabUser extends KolabObject {

    /**
     * The LDAP filter to retrieve this object type
     *
     * @var string
     */
    var $filter = '(&(objectclass=kolabInetOrgPerson)(uid=*)(mail=*)(sn=*))';

    /**
     * Sort using this attribute by default
     *
     * @var string
     */
    var $sort = KOLAB_ATTR_SN;

    /**
     * May this object log in to the Kolab system?
     *
     * @var boolean
     */
    var $login_allowed = true;

    /**
     * The possible selections for invitation policies
     *
     * @var array
     */
    var $invpol = array(
        'ACT_ALWAYS_ACCEPT', 
        'ACT_ALWAYS_REJECT', 
        'ACT_REJECT_IF_CONFLICTS', 
        'ACT_MANUAL_IF_CONFLICTS', 
        'ACT_MANUAL' 
    );

    /**
     * The possible account types
     *
     * @var array
     */
    var $acctyp = array(
        '', 
        'cn=internal,', 
        'cn=groups,', 
        'cn=resources,'
    );


    /**
     * The LDAP attributes fetched for listing
     *
     * @var array
     */
    var $list_attributes = array(
        KOLAB_ATTR_SN,
        KOLAB_ATTR_CN,
        KOLAB_ATTR_UID,
        KOLAB_ATTR_MAIL,
        KOLAB_ATTR_DELETED,
    );

    /**
     * The attributes supported by this class
     *
     * @var array
     */
    var $_supported_attributes = array(
        KOLAB_ATTR_SN,
        KOLAB_ATTR_FN,
        KOLAB_ATTR_UID,
        KOLAB_ATTR_MAIL,
        KOLAB_ATTR_DELETED,
    );

    /**
     * Get the user type
     *
     * @return string The type of this user
     */
    function getType()
    {
        $dncomp = split(',', $this->_dn);
        if( in_array('cn=groups',$dncomp) ) {
            return 'G';
        } else if(in_array('cn=resources',$dncomp)) {
            return 'R';
        } else if(in_array('cn=internal',$dncomp)) {
            return 'I';
        } else {
            return 'U';
        }
    }

    /**
     * Create a user
     *
     * @param string $dn DN
     *
     * @return boolean True on success, a PEAR error otherwise
     */
    function createUser($userdata)
    {
        $ldap_object = array();
        $ldap_object['objectClass'] = $userdata['objectclasses'];
        $ldap_object['sn'] = $userdata['sn'];
        $ldap_object['cn'] = $userdata['givenname'] . ' ' . $ldap_object['sn'];
        $ldap_object['givenName'] = $userdata['givenname'];
        $ldap_object['userPassword'] = ssha( $userdata['password_0'], gensalt());
        $ldap_object['mail'] = $userdata['usermail'] . '@' . $userdata['domainmail'];
        if ($create) {
            $ldap_object['uid'] = ($uid == '') ? $ldap_object['mail'] : $userdata['uid'];
            $ldap_object['kolabHomeServer'] = $userdata['kolabhomeserver'];
        }
        foreach( array( 'title', 'o', 'ou', 'roomNumber', 'street',
                        'postOfficeBox', 'postalCode', 'l', 'c',
                        'telephoneNumber', 'facsimileTelephoneNumber', 
                        'kolabFreeBusyFuture', 'kolabDelegate', 'cyrus-userquota' ) as $attr ) {
            $ldap_object[$attr] = $userdata[$attr];
        }
        
        // Handle group/resource policies
        $i = 0;
        $ldap_object['kolabInvitationPolicy'] = array();
        while( isset( $userdata['user_kolabinvitationpolicy_' . $i] ) ) {
            $user = $userdata['user_kolabinvitationpolicy_' . $i];
            $pol  = (int)$userdata['policy_kolabinvitationpolicy_' . $i];
            $i++;
            if( !empty($user) && 0 <= $pol && $pol < 5  ) {
                if( $this->invpol[$pol] ) {
                    $ldap_object['kolabInvitationPolicy'][] = ($user=='anyone' ? '' : "$user:") . $this->invpol[$pol];
                }
            }
        }         

        $type = $userdata['accttype'];
        if( $type < 0 || $type > 3 ) {
            $type = 3;
        }
        $domain_dn = $this->acctyp[$type] . $this->_params['base_dn'];


    }

    /**
     * Delete a user
     *
     * @param string $dn DN of the user that should be deleted
     *
     * @return boolean True on success, a PEAR error otherwise
     */
    function deleteUser($dn)
    {
   // Check for distribution lists with only this user as member
   $smarty->ldap->search( $_SESSION['base_dn'], 
                  '(&(objectClass=kolabGroupOfNames)(member='.$smarty->ldap->escape($dn).'))',
                  array( 'dn', 'cn', 'mail', 'member' ) );
   $distlists = $smarty->ldap->getEntries();
   unset($distlists['count']);
   foreach( $distlists as $distlist ) {
     $dlmail = $distlist['mail'][0];
     if( !$dlmail ) $dlmail = $distlist['cn'][0]; # Compatibility with old stuff
     if( $distlist['member']['count'] == 1 ) {
       $errors[] = sprintf(_("Account could not be deleted, distribution list '%s' depends on it."), $dlmail);
     }
   }
   if( !$errors ) foreach( $distlists as $distlist ) {
     $dlmail = $distlist['mail'][0];
     if( !$dlmail ) $dlmail = $distlist['cn'][0]; # Compatibility with old stuff
     if( ldap_mod_del( $smarty->ldap->connection, $distlist['dn'], array('member' => $dn ) ) ) {
       $messages[] = sprintf(_("Account removed from distribution list '%s'."), $dlmail);
     } else {
       $errors[] = sprintf(_("Failure to remove account from distribution list '%s', account will not be deleted."),
                           $dlmail);
       break;
     }
   }
        
    }

    function getName() 
    {
        return $this->getSn() . ', ' . $this->getFn();
    }
    
};

class KolabUserListView extends KolabObjectListView {

    /**
     * Return a list title for this type of object
     *
     * @return 
     */
    function getTitle()
    {
        return _("Email Users");
    }

    /**
     * Return an array of table header entries for this type of object
     *
     * @return array The header entries
     */
    function _getHeader()
    {
        return array( _("Type"), _("Name"), _("E-mail"), _("UID"));
    }
    
    /**
     * Return a list entry for this object
     *
     * @return 
     */
    function getRow()
    {
		$row = '';
        if ($this->_object->getType() == 'U') {
            $row .= '<td class="contentcell" title="' . _("User Account") . '" align="center">U</td>';
        } elseif ($this->_object->getType() == 'G') {
            $row .= '<td class="contentcell" title="' . _("Group Account") . '" align="center">G</td>';
        } elseif ($this->_object->getType() == 'R') {
            $row .= '<td class="contentcell" title="' . _("Resource Account") . '" align="center">R</td>';
        } else {
            $row .= '<td class="contentcell" title="' . _("Unknown Account Type") . '" align="center">?</td>';
        }
        $row .= '<td class="contentcell">' . htmlspecialchars($this->_object->getName()) . '</td>';
        $mail = htmlspecialchars($this->_object->getMail());
        $row .= '<td class="contentcell"><a href="mailto:' . $email . '">' . $mail . '</a></td>';
        $row .= '<td class="contentcell">' . htmlspecialchars($this->_object->getUid()) . '</td>';
		$row .= $this->getAction();
        return $row;
    }
   
};

class KolabUserListView extends KolabObjectListView {

    /**
     * Return a list title for this type of object
     *
     * @return 
     */
    function getTitle()
    {
        return _("Email Users");
    }

    /**
     * Return an array of table header entries for this type of object
     *
     * @return array The header entries
     */
    function _getHeader()
    {
        return array( _("Type"), _("Name"), _("E-mail"), _("UID"));
    }
    
    /**
     * Return a list entry for this object
     *
     * @return 
     */
    function getRow()
    {
		$row = '';
        if ($this->_object->getType() == 'U') {
            $row .= '<td class="contentcell" title="' . _("User Account") . '" align="center">U</td>';
        } elseif ($this->_object->getType() == 'G') {
            $row .= '<td class="contentcell" title="' . _("Group Account") . '" align="center">G</td>';
        } elseif ($this->_object->getType() == 'R') {
            $row .= '<td class="contentcell" title="' . _("Resource Account") . '" align="center">R</td>';
        } else {
            $row .= '<td class="contentcell" title="' . _("Unknown Account Type") . '" align="center">?</td>';
        }
        $row .= '<td class="contentcell">' . htmlspecialchars($this->_object->getName()) . '</td>';
        $mail = htmlspecialchars($this->_object->getMail());
        $row .= '<td class="contentcell"><a href="mailto:' . $email . '">' . $mail . '</a></td>';
        $row .= '<td class="contentcell">' . htmlspecialchars($this->_object->getUid()) . '</td>';
		$row .= $this->getAction();
        return $row;
    }
    

}

/*
  Local variables:
  mode: php
  indent-tabs-mode: f
  tab-width: 4
  buffer-file-coding-system: utf-8
  End:
  vim:encoding=utf-8:
 */
?>
