<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.6 $
 *
 *  ABOUT
 *  -----
 *
 *  A class for Kolab domain maintainer management.
 *
 */

/** The basic Kolab object definition */
require_once('Kolab/Webadmin/object.php');

/**
 * This class provides methods associated to Kolab domain maintainers.
 *
 * $Header: /home/kroupware/jail/kolabrepository/server/php-kolab/Kolab_Webadmin/Webadmin/domainmaintainer.class.php,v 1.6 2007/09/18 06:08:48 gunnar Exp $
 *
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Webadmin
 */
class KolabDomainMaintainer extends KolabObject {

    /**
     * The LDAP filter to retrieve this object type
     *
     * @var string
     */
    var $filter = '(&(cn=*)(objectclass=kolabInetOrgPerson)(!(uid=manager))(sn=*))';

    /**
     * Sort using this attribute by default
     *
     * @var string
     */
    var $sort = KOLAB_ATTR_SN;

    /**
     * May this object log in to the Kolab system?
     *
     * @var boolean
     */
    var $login_allowed = true;

    /**
     * The LDAP attributes fetched for listing
     *
     * @var array
     */
    var $list_attributes = array(
        KOLAB_ATTR_SN,
        KOLAB_ATTR_CN,
        KOLAB_ATTR_UID,
        KOLAB_ATTR_DELETED,
    );

    /**
     * The attributes supported by this class
     *
     * @var array
     */
    var $_supported_attributes = array(
        KOLAB_ATTR_SN,
        KOLAB_ATTR_FN,
        KOLAB_ATTR_UID,
        KOLAB_ATTR_DELETED,
    );

    /**
     * Return the domains that this maintainer maintains
     *
     * @return array The domains belonging to this domain maintainer.
     */
    function getDomains() {
        $domains = array();
        $filter = '(member=' . $this->escape($this->_dn) . ')';
        $result = $this->_ldap->search('cn=domains,cn=internal,' . 
                                       $this->_ldap->base_dn, $filter, 
                                       array('cn'));
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        $entries = $this->_ldap->getEntries($result);
        unset($entries['count']);
        if (count($entries) > 0) {
            foreach($entries as $val) {
                $domains[] = $val['cn'][0];
            }
        }
        return $domains;
    }

    function domainsForMaintainerUid($uid) {
        debug("domainsForMaintainer($uid):");
        $dn = $this->dnForUid($uid);
        if ($dn) {
            return $this->domainsForMaintainerDn($dn);
        }
        return false;
    }

    function addToDomainGroups($member, $domains) {
        if (empty($domains)) {
            return true;
        }
        foreach($domains as $domain) {
            $domgrpdn = 'cn='.$this->dn_escape($domain).',cn=domains,cn=internal,'.$this->_base_dn;
            $dom_obj = $this->read($domgrpdn);      
            if (!$dom_obj) {
                debug("+Adding group $domgrpdn with member $member");
                if (!ldap_add($this->_ldap, $domgrpdn, 
                              array('objectClass' => array('top', 'kolabGroupOfNames'),
                                    'cn' => $domain,
                                    'member' => $member))) {
                    debug("Error adding domain group: ".ldap_error($this->_ldap));
                    return false;
                }
            } else {
                if (!in_array($member, $dom_obj['member'])) {
                    debug("+Adding member $member to $domgrpdn");
                    if (!ldap_mod_add($this->_ldap, $domgrpdn, array('member' => $member))) {
                        debug("Error adding $member to domain $domgrpdn: ".ldap_error($this->_ldap));
                        return false;
                    }
                }
            }
        }
        return true;
    }

    function removeFromDomainGroups($member, $domains) {
        if (empty($domains)) {
            return true;
        }
        foreach($domains as $domain) {
            $domgrpdn = 'cn='.$this->dn_escape($domain).',cn=domains,cn=internal,'.$this->_base_dn;
            $dom_obj = $this->read($domgrpdn);
            if ($dom_obj) {
                if (count($dom_obj['member'] == 1)) {
                    debug("-Removing group $domgrpdn");
                    if (!ldap_delete($this->_ldap, $domgrpdn)) {
                        debug("Error deleting domain group $domgrpdn: ".ldap_error($this->_ldap));
                        return false;            
                    }
                } else {
                    debug("-Removing member $member from group $domgrpdn");
                    if (!ldap_mod_del($this->_ldap, $domgrpdn, array('member' => $member))) {
                        debug("Error deleting $member from domain $domgrpdn: ".ldap_error($this->_ldap));
                        return false;
                    }  
                }
            }
        }    
    }
}

class KolabDomainMaintainerListView extends KolabObjectListView {

    /**
     * Return a list title for this type of object
     *
     * @return 
     */
    function getTitle()
    {
        return _("Domain maintainers");
    }

    /**
     * Return an array of table header entries for this type of object
     *
     * @return array The header entries
     */
    function _getHeader()
    {
        return array(_("Name"), _("UID"), _("Domains"));
    }
    
    /**
     * Return an array of table row entries for this type of object
     *
     * @return array The row entries
     */
    function _getRow()
    {
        return array($this->_object->getName(),
					 $this->_object->getUid(),
					 join(' ', $this->_object->getDomains()));
    }
}
/*
  Local variables:
  mode: php
  indent-tabs-mode: f
  tab-width: 4
  buffer-file-coding-system: utf-8
  End:
  vim:encoding=utf-8:
 */
?>
