<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.10 $
 *
 *  ABOUT
 *  -----
 *
 *  This provides error pages for the Kolab free/busy system.
 *
 */

/* Load the required PEAR libraries */ 
require_once 'PEAR.php';

/* Load the required Horde libraries */ 
require_once 'Horde.php';
require_once "Horde/String.php";
require_once "Horde/Util.php";
require_once "Horde/Kolab/LDAP.php";

/**
 * The Kolab_Freebusy_Error:: class provides error pages for the 
 * Kolab free/busy system.
 *
 * @author  Gunnar Wrobel <p@rdus.de>
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @package Kolab_Freebusy
 */
class Kolab_Freebusy_Error {

    /**
     * Deliver a "Not Found" page
     *
     * @param PEAR_Error $error    The error.
     */
    function notFound($error)
    {
        $headers = array('HTTP/1.0 404 Not Found');
        $url = htmlentities($_SERVER['REQUEST_URI']);
        $message = sprintf(_("The requested URL %s was not found on this server."), $url);
        
        Kolab_Freebusy_Error::_errorPage($error, $headers, _("404 Not Found"), _("Not found"), $message);
    }
    
    /**
     * Deliver a "Unauthorized" page
     *
     * @param PEAR_Error $error    The error.
     */
    function unauthorized($error) {
        global $conf;
        
        if (!empty($conf['fb']['email_domain'])) {
            $email_domain = $conf['fb']['email_domain'];
        } else {
            $email_domain = 'localhost';
        }

        $headers = array('WWW-Authenticate: Basic realm="freebusy-' . $email_domain . '"',
                         'HTTP/1.0 401 Unauthorized');
        
        Kolab_Freebusy_Error::_errorPage($error, $headers, _("401 Unauthorized"), _("Unauthorized"),
                  _("You are not authorized to access the requested URL."));
    }
    
    /**
     * Deliver a "Server Error" page
     *
     * @param PEAR_Error $error    The error.
     */
    function serverError($error) {
        $headers = array('HTTP/1.0 500 Server Error');
        Kolab_Freebusy_Error::_errorPage($error, $headers, _("500 Server Error"), _("Error"),
                  htmlentities($_SERVER['REQUEST_URI']));
    }

    /**
     * Deliver an error page
     *
     * @param PEAR_Error $error    The error.
     * @param array      $headers  The HTTP headers to deliver with the response
     * @param string     $title    The page title
     * @param string     $headline The headline of the page
     * @param string     $body     The message to display on the page
     */
    function _errorPage($error, $headers, $title, $headline, $body) {

        /* Print the headers */
        foreach ($headers as $line) {
            header($line);
        }
        
        /* Print the page */
        echo "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">\n";
        echo "<html><head><title>" . $title . "</title></head>\n";
        echo "<body>\n";
        echo "<h1>" . $headline . "</h1>\n";
        echo "<p>" . $body . "</p>\n";
        if (!empty($error)) {
            echo "<hr><pre>" . $error->getMessage() . "</pre>\n";
            Horde::logMessage($error, __FILE__, __LINE__, PEAR_LOG_ERR);
        }
        echo "<hr>\n";
        echo $_SERVER['SERVER_SIGNATURE'] . "\n";
        echo "</body></html>\n";
        exit;
    }
}

/**
 * The Timer:: class provides a high precision timer for calculating
 * page generation time.
 *
 * @author  Gunnar Wrobel <p@rdus.de>
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @package Kolab_Freebusy
 */
class Timer
{
    var $_start;

    function Timer()
    {
        $this->_start = $this->_microtime_float();
    }
    
    function stop()
    {
        return $this->_microtime_float() - $this->_start;
    }
    
    function _microtime_float() 
    {
        list($usec, $sec) = explode(" ", microtime());
        return (float) $usec + (float) $sec;
    }
}

/**
 * The FolderAccess:: class provides functionality to check free/busy
 * access rights for the specified folder.
 *
 * @author  Gunnar Wrobel <p@rdus.de>
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @package Kolab_Freebusy
 */
class FolderAccess {

    /**
     * The user calling the script
     *
     * @var string
     */
    var $user;

    /**
     * The password of the user calling the script
     *
     * @var string
     */
    var $pass;

    /**
     * The folder we try to access
     *
     * @var string
     */
    var $folder;

    /**
     * The IMAP path of folder we try to access
     *
     * @var string
     */
    var $imap_folder;

    /**
     * The requested owner
     *
     * @var string
     */
    var $req_owner;

    /**
     * The owner of that folder
     *
     * @var string
     */
    var $owner;

    /**
     * The common name (CN) of the owner
     *
     * @var string
     */
    var $cn = '';

    /**
     * The homeserver for that folder
     *
     * @var string
     */
    var $homeserver;

    /**
     * The groups of the owner
     *
     * @var array
     */
    var $groups;

    /**
     * The groups of the user
     *
     * @var array
     */
    var $user_groups;

    /**
     * The number of days to calculate free/busy into the past
     *
     * @var int
     */
    var $fbpast;

    /**
     * The number of days to calculate free/busy into the future
     *
     * @var int
     */
    var $fbfuture = 60;

    function FolderAccess($user = null, $pass = null)
    {
        if (isset($user) && isset($pass)) {
            $this->user = $user;
            $this->pass = $pass;
        } else {
            $this->_parseUser();
        }
    }
    
    function parseFolder($req_folder = '')
    {
        /* Handle the owner/folder name and make sure the owner part is in lower case */
        $req_folder = String::convertCharset($req_folder, 'UTF-8', 'UTF7-IMAP');
        $folder = explode('/', $req_folder);
        if (count($folder) < 2) {
            return PEAR::raiseError(sprintf(_("No such folder %s"), $req_folder));
        }

        $folder[0] = strtolower($folder[0]);
        $req_folder = implode('/', $folder);
        $this->req_owner = $folder[0];
        unset($folder[0]);
        $this->folder = join('/', $folder);

        $result = $this->_process();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
    }

    function parseOwner($req_owner = '')
    {
        $this->req_owner = $req_owner;

        $result = $this->_process();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
    }
    
    function fetchRemote()
    {
        global $conf;

        if (!empty($conf['fb']['server'])) {
            $server = $conf['fb']['server'];
        } else {
            $server = 'localhost';
        }
        if (!empty($conf['fb']['redirect'])) {
            $do_redirect = $conf['fb']['redirect'];
        } else {
            $do_redirect = false;
        }

        /* Check if we are on the right server and redirect if appropriate */
        if ($this->homeserver && $this->homeserver != $server) {
            $redirect = 'https://' . $this->homeserver . $_SERVER['REQUEST_URI'];
            Horde::logMessage(sprintf(_("Found remote user, redirecting to %s"), 
                                      $this->homeserver), __FILE__, __LINE__, PEAR_LOG_ERR);
            if ($do_redirect) {
                header("Location: $redirect");
            } else {
                header("X-Redirect-To: $redirect");
                $redirect = 'https://' . urlencode($this->user) . ':' . urlencode($this->pass)
                    . '@' . $this->homeserver . $_SERVER['REQUEST_URI'];
                if (!@readfile($redirect)) {
                    $message = sprintf(_("Unable to read free/busy information from %s"), 
                                       'https://' . urlencode($this->user) . ':XXX'
                                       . '@' . $this->homeserver . $_SERVER['REQUEST_URI']);
                    return PEAR::raiseError($message);
                }
            }
            exit;
        }
    }

    function authenticate()
    {
        if (empty($this->user)) {
            return PEAR::raiseError(_("Please authenticate!"));
        }
        
        /* Load the authentication libraries */
        require_once "Horde/Auth.php";
        require_once 'Horde/Secret.php';

        $auth = &Auth::singleton('kolab');

        if (!$auth->authenticate($this->user, array('password' => $this->pass), false)) {
            return PEAR::raiseError(sprintf(_("Invalid Kolab authentication for user %s!"), 
                                            $this->user));
        }
        @session_start();
        $_SESSION['__auth'] = array(
            'authenticated' => true,
            'userId' => $this->user,
            'timestamp' => time(),
            'remote_addr' => isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : null,
        );
        Auth::setCredential('password', $this->pass);
    }

    function _parseUser()
    {
        $this->user = isset($_SERVER['PHP_AUTH_USER'])?$_SERVER['PHP_AUTH_USER']:false;
        $this->pass = isset($_SERVER['PHP_AUTH_PW'])?$_SERVER['PHP_AUTH_PW']:false;

        // This part allows you to use the PHP scripts with CGI rather than as
        // an apache module. This will of course slow down things but on the
        // other hand it allows you to reduce the memory footprint of the 
        // apache server. The default is to use PHP as a module and the CGI 
        // version requires specific Apache configuration.
        //
        // The line you need to add to your configuration of the /freebusy 
        // location of your server looks like this:
        //
        //    RewriteRule .* - [E=REMOTE_USER:%{HTTP:Authorization}]
        //
        // The complete section will probably look like this then:
        //
        //  <IfModule mod_rewrite.c>
        //    RewriteEngine On
        //    # FreeBusy list handling
        //    RewriteBase /freebusy
        //    RewriteRule .* - [E=REMOTE_USER:%{HTTP:Authorization}]
        //    RewriteRule ^([^/]+)\.ifb       freebusy.php?uid=$1		    [L]
        //    RewriteRule ^([^/]+)\.vfb       freebusy.php?uid=$1		    [L]
        //    RewriteRule ^([^/]+)\.xfb       freebusy.php?uid=$1&extended=1        [L]
        //    RewriteRule ^trigger/(.+)\.pfb  pfb.php?folder=$1&cache=0             [L]
        //    RewriteRule ^(.+)\.pfb          pfb.php?folder=$1&cache=1             [L]
        //    RewriteRule ^(.+)\.pxfb         pfb.php?folder=$1&cache=1&extended=1  [L]
        //  </IfModule>
        if (empty($this->user) && isset($_ENV['REDIRECT_REDIRECT_REMOTE_USER'])) {
            $a = base64_decode(substr($_ENV['REDIRECT_REDIRECT_REMOTE_USER'], 6)) ;
            if ((strlen($a) != 0) && (strcasecmp($a, ":" ) == 0)) {
                list($this->user, $this->pass) = explode(':', $a, 2);
            }
        }
    }

    function _process() 
    {
        /* Bind to LDAP so that we can request user data */
        $ldap = Horde_Kolab_LDAP::singleton();

        if (!$ldap->is_bound) {
            $result = $ldap->bind();
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
        }

        /* Does not really belong here but since we are connected now,
         we should fetch that config info */
        $this->fbpast = $ldap->freeBusyPast();

        /* Fetch the user info of the calling user */
        $userinfo = $ldap->userInfo($this->user);
        if (is_a($userinfo, 'PEAR_Error')) {
            Horde::logMessage($userinfo, __FILE__, __LINE__, PEAR_LOG_ERROR);
            $userinfo = null;
        }

        /* Possibly rewrite the calling UID into the primary mail address */
        if ($userinfo) {
            if (isset($userinfo['MAIL']) && !empty($userinfo['MAIL'])) {
                $this->user = $userinfo['MAIL'];
            }
            if (!empty($userinfo['GROUPS'])) {
                $this->user_groups = $userinfo['GROUPS'];  
            }
        }

        if (empty($this->user_groups)) {
            $this->user_groups = array($this->user);
        } else {
            $this->user_groups[] = $this->user;
        }

        /* Fetch the user info of the requested folder owner */
        $uinfo = $ldap->userInfo($this->req_owner);
        if (is_a($uinfo, 'PEAR_Error')) {
            Horde::logMessage($uinfo, __FILE__, __LINE__, PEAR_LOG_ERROR);
            $uinfo = null;
        }

        /* Possibly rewrite the owner UID into the primary mail address */
        if (!empty($uinfo) && isset($uinfo['MAIL']) && !empty($uinfo['MAIL'])) {
            $this->owner = $uinfo['MAIL'];
        }

        /* If we were unable to determine the owner we will finally
         * try to append the domain name of the calling user. */
        if (empty($uinfo) || empty($this->owner) ||
            strpos($this->owner, '@') === false) {
            /* try guessing the domain */
            $idx = strpos($this->user, '@');
            if($idx !== false) {
                $domain = substr($this->user, $idx+1);
                Horde::logMessage(sprintf(_("Trying to append %s to %s"), 
                                          $domain, $this->req_owner),
                                  __FILE__, __LINE__, PEAR_LOG_DEBUG);
                $uinfo = $ldap->userInfo($this->req_owner . '@' . $domain);
                if (is_a($uinfo, 'PEAR_Error')) {
                    Horde::logMessage($uinfo, __FILE__, __LINE__, PEAR_LOG_ERROR);
                    $uinfo = null;
                }
            }
        }

        /* Get the owner mail address and the home server */
        if ($uinfo) {
            if (!empty($uinfo['MAIL'])) {
                $this->owner = $uinfo['MAIL'];
            }
            if (!empty($uinfo['CN'])) {
                $this->cn = $uinfo['CN'];  
            }
            if (!empty($uinfo['HOMESERVER'])) {
                $this->homeserver = $uinfo['HOMESERVER'];  
            }
            if (!empty($uinfo['FBFUTURE'])) {
                $this->fbfuture = $uinfo['FBFUTURE'];  
            }
            if (!empty($uinfo['GROUPS'])) {
                $this->groups = $uinfo['GROUPS'];  
            }
        }
        if (empty($this->groups)) {
            $this->groups = array($this->owner);
        } else {
            $this->groups[] = $this->owner;
        }
        $this->imap_folder = $this->_getImapFolder();
    }

    function _getImapFolder() 
    {
        $userdom = false;
        $ownerdom = false;
        if (ereg( '(.*)@(.*)', $this->user, $regs)) {
            // Regular user
            $user = $regs[1];
            $userdom  = $regs[2];
        } else {
            $user = $this->user;
        }
        
        if(ereg( '(.*)@(.*)', $this->owner, $regs)) {      
            // Regular owner
            $owner = $regs[1];
            $ownerdom = $regs[2];
        } else {
            $owner = $this->owner;
        }

        $fldrcomp = array();
        if ($user == $owner) {
            $fldrcomp[] = 'INBOX';
        } else {
            $fldrcomp[] = 'user';
            $fldrcomp[] = $owner;
        }
        
        if (!empty($this->folder)) {
            $fldrcomp[] = $this->folder;
        }
    
        $folder = join('/', $fldrcomp);
        if ($ownerdom && !$userdom) {
            $folder .= '@' . $ownerdom;
        }
        return $folder;
    }

}

class FreeBusyView {

    var $_vfb;

    function FreeBusyView($vfb)
    {
        $this->_vfb = $vfb->exportvCalendar();

        $ts = time();
        
        $components = &$vfb->getComponents();
        foreach ($components as $component) {
            if ($component->getType() == 'vFreebusy') {
                $attr = $component->getAttribute('DTSTAMP');
                if (!empty($attr) && !is_a($attr, 'PEAR_Error')) {
                    $ts = $attr;
                    break;
                }
            }
        }

        $this->_ts  = $ts;
    }
    
    function render($content = '')
    {
        global $conf;

        if (!empty($conf['fb']['send_content_type'])) {
            $send_content_type = $conf['fb']['send_content_type'];
        } else {
            $send_content_type = false;
        }
        
        if (!empty($conf['fb']['send_content_length'])) {
            $send_content_length = $conf['fb']['send_content_length'];
        } else {
            $send_content_length = false;
        }
        
        if (!empty($conf['fb']['send_content_disposition'])) {
            $send_content_disposition = $conf['fb']['send_content_disposition'];
        } else {
            $send_content_disposition = false;
        }
        
        /* Ensure that the data doesn't get cached along the way */
        header('Cache-Control: no-store, no-cache, must-revalidate');
        header('Cache-Control: post-check=0, pre-check=0', false);
        header('Pragma: no-cache');
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
        header('Last-Modified: ' . gmdate("D, d M Y H:i:s", $this->_ts) . ' GMT');
        header('Pragma: public');
        header('Content-Transfer-Encoding: none');
        if ($send_content_type) {
            header('Content-Type: text/calendar');
        }
        if ($send_content_length) {
            header('Content-Length: ' . strlen($this->_vfb));
        }
        if ($send_content_disposition) {
            header('Content-Disposition: attachment; filename="' . $content . '"');
        }

        echo $this->_vfb;
    }
    
}
