<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.12 $
 *
 *  ABOUT
 *  -----
 *
 *  FIXME
 *
 */

require_once 'Net/IMAP.php';
require_once 'Horde/iCalendar.php';
require_once 'Horde/MIME.php';
require_once 'Horde/MIME/Message.php';
require_once 'Horde/MIME/Headers.php';
require_once 'Horde/MIME/Part.php';
require_once 'Horde/MIME/Structure.php';

require_once 'Kolab/Filter/recurrence.class.php';

require_once 'Horde/String.php';
String::setDefaultCharset('utf-8');

// Required to be able to use the old Horde framework
function wrap($text) 
{
    return String::wrap($text, 76, "\n", 'utf-8');
}

// What actions we can take when receiving an event request
define('RM_ACT_ALWAYS_ACCEPT',              1);
define('RM_ACT_REJECT_IF_CONFLICTS',        2);
define('RM_ACT_MANUAL_IF_CONFLICTS',        3);
define('RM_ACT_MANUAL',                     4);
define('RM_ACT_ALWAYS_REJECT',              5);

// What possible ITIP notification we can send
define('RM_ITIP_DECLINE',                   1);
define('RM_ITIP_ACCEPT',                    2);
define('RM_ITIP_TENTATIVE',                 3);

// Globals
$imap = NULL;
$server = '';
$mailbox = '';
$calmbox = '';
$prefix = '';
$suffix = '';
$connected = false;

/* Recurrence implementation for looking for
   conflicts between an event and a freebusy list
*/
 class ResmgrRecurrence extends Recurrence {
     function ResmgrRecurrence() { 
         $this->conflict = false;
     }

     function setBusy( $start, $end, $duration ) {
         if( $this->conflict ) return;
         if( is_null($end) ) $end = $start + $duration;
         foreach ($this->busyperiods as $busyfrom => $busyto) {
             if ( in_array(base64_decode($this->extraparams[$busyfrom]['X-UID']), $this->ignore) ||
                  in_array(base64_decode($this->extraparams[$busyfrom]['X-SID']), $this->ignore) ) {
                 // Ignore
                 continue;
             }
             if (($busyfrom >= $start && $busyfrom < $end) || ($start >= $busyfrom && $start < $busyto)) {
                 Horde::logMessage('Request overlaps', __FILE__, __LINE__, PEAR_LOG_DEBUG);
                 $this->conflict = true;
                 break;
             }
         }
     }

     function hasConflict() { return $this->conflict; }

     var $busyperiods;
     var $extraparams;
    var $ignore;
     var $conflict;
 };

function imapClose()
{
    global $imap, $connected;

    if (defined($imap) && $imap !== false) {
        $imap->disconnect();      
    }

    $connected = false;
}

$is_shutting_down = false;

function shutdown($return = 1, $errormsg = "", $updatefb = true )
{
    global $is_shutting_down;

    //Guard against recursion(!)
    if( $is_shutting_down ) return;
    $is_shutting_down = true;
    Horde::logMessage("Shutting down ($return)", __FILE__, __LINE__, PEAR_LOG_DEBUG);

    imapClose();
    logClose();
    print $errormsg;
    exit($return);
}

function parseactionstring( $action ) {
    switch (trim($action)) {
    case 'ACT_ALWAYS_ACCEPT': return RM_ACT_ALWAYS_ACCEPT;
    case 'ACT_ALWAYS_REJECT': return RM_ACT_ALWAYS_REJECT;
    case 'ACT_REJECT_IF_CONFLICTS': return RM_ACT_REJECT_IF_CONFLICTS;
    case 'ACT_MANUAL_IF_CONFLICTS': return RM_ACT_MANUAL_IF_CONFLICTS;
    case 'ACT_MANUAL': return RM_ACT_MANUAL;
    default:  return false;
    }
}

function actiontostring( $action ) {
    switch ($action) {
    case RM_ACT_ALWAYS_ACCEPT: return 'ACT_ALWAYS_ACCEPT';
    case RM_ACT_ALWAYS_REJECT: return 'ACT_ALWAYS_REJECT';
    case RM_ACT_REJECT_IF_CONFLICTS: return 'ACT_REJECT_IF_CONFLICTS';
    case RM_ACT_MANUAL_IF_CONFLICTS: return 'ACT_MANUAL_IF_CONFLICTS';
    case RM_ACT_MANUAL: return 'ACT_MANUAL';
    default:  return false;
    }  
};


function getDN( $ldap, $mail ) {
    global $conf;
    $filter = "(&(objectClass=kolabInetOrgPerson)(|(mail=$mail)(alias=$mail)))";
    $result = ldap_search($ldap, $conf['filter']['base_dn'],
                          $filter,
                          array('dn'));
    if (!$result) {
        Horde::logMessage(sprintf(_("Unable to perform LDAP search: %s"),
                                  ldap_error($ldap)),
                          __FILE__, __LINE__, PEAR_LOG_WARNING);
        return false;
    }
    $dn = false;
    if( ldap_count_entries( $ldap, $result ) > 0 ) {
        $entries = ldap_get_entries($ldap, $result);
        $dn = $entries[0]['dn'];
    }
    ldap_free_result( $result );
    return $dn;
}

/**
 * Look up action and encrypted password from LDAP and decrypt it
 */
function getLDAPData($sender,$resource)
{
    global $conf;

    Horde::logMessage(sprintf(_("getLDAPData(%s, %s)"),
                              $sender, $resource),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    // Connect to the LDAP server and retrieve the users' password
    $ldap = ldap_connect($conf['filter']['ldap_uri']);
    if (!ldap_bind($ldap, $conf['filter']['bind_dn'], $conf['filter']['bind_pw'])) {
        return PEAR::raiseError(sprintf(_("Unable to contact LDAP server: %s"),
                                        ldap_error($ldap)),
                                OUT_LOG | EX_TEMPFAIL);
    }
    
    $result = ldap_search($ldap, $conf['filter']['base_dn'],
                          "(&(objectClass=kolabInetOrgPerson)(mail=$resource))",
                          array("cn", "kolabHomeServer", /*"kolabEncryptedPassword",*/
                                "kolabInvitationPolicy" ));
    if (!$result) {
        return PEAR::raiseError(sprintf(_("Unable to perform LDAP search: %s"),
                                        ldap_error($ldap)),
                                OUT_LOG | EX_TEMPFAIL);
    }
    
    $entries = ldap_get_entries($ldap, $result);
    if ($entries['count'] != 1) {
        Horde::logMessage(sprintf(_("%s objects returned for %s"),
                                  $entries['count'], $resource),
                          __FILE__, __LINE__, PEAR_LOG_WARNING);
        return false;
    }
    
    $cn = $entries[0]['cn'][0];
    $hs = $entries[0]['kolabhomeserver'][0];
    $actions = $entries[0]['kolabinvitationpolicy'];

    $policies = array();
    $defaultpolicy = false;
    foreach( $actions as $action ) {
        if( ereg( '(.*):(.*)', $action, $regs ) ) {
            Horde::logMessage(sprintf(_("Found policy %s: %s"),
                                      $regs[1], $regs[2]), 
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
            $policies[strtolower($regs[1])] = parseactionstring($regs[2]);
        } else {
            $defaultpolicy = parseactionstring($action);
        }
    }
    // Find sender's policy
    if( array_key_exists( $sender, $policies ) ) {
        // We have an exact match, stop processing
        $action = $policies[$sender];
        Horde::logMessage(sprintf(_("Found exact policy match %s for %s"),
                                  actiontostring($action), $sender), 
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
    } else {
        $action = false;
        $dn = getDN( $ldap, $sender );
        if( $dn ) {
            // Sender is local, check for groups
            foreach( $policies as $gid => $policy ) {
                $result = ldap_search($ldap, $conf['filter']['base_dn'],
                                      "(&(objectClass=kolabGroupOfNames)(mail=$gid)(member=$dn))",
                                      array('dn'));
                if (!$result) {
                    Horde::logMessage(sprintf(_("Unable to perform LDAP search: %s"),
                                              ldap_error($ldap)),
                                      __FILE__, __LINE__, PEAR_LOG_WARNING);
                    return false;
                }
                if(ldap_count_entries($ldap, $result) > 0) {
                    // User is member of group
                    if( !$action ) $action = $policy;
                    else $action = min( $action, $policy );
                }
            }
        }
        if( !$action && $defaultpolicy ) $action = $defaultpolicy;
    }


    ldap_close($ldap);
    return array( 'cn' => $cn, 'homeserver' => $hs, 'action' => $action );
}


function getRequest($filename)
{
    global $requestText;

    $requestText = '';
    $handle = fopen( $filename, "r" );
    while (!feof($handle)) {
        $requestText .= fread($handle, 8192);
    }
}

function checkSMTPResponse(&$smtp, $code)
{
    $resp = $smtp->getResponse();
    if ($resp[0] != $code) {
        Horde::logMessage($resp[1], __FILE__, __LINE__, PEAR_LOG_ERR);
        shutdown(1, $resp[1]);
    }

    return true;
}

function sendSMTP($sender, $recip, &$data)
{
    global $conf;
    static $smtp;
    if (!isset($smtp)) {
        require_once 'Net/SMTP.php';

        $smtp = &new Net_SMTP($conf['filter']['local_addr'] ,10026);
        if (!$smtp) {
            $msg = 'Could not create SMTP object';
            Horde::logMessage($msg, __FILE__, __LINE__, PEAR_LOG_ERR);
            shutdown(1, $msg);
        }

        if (PEAR::isError($error = $smtp->connect())) {
            $msg = 'Failed to connect to SMTP: ' . $error->getMessage();
            Horde::logMessage($msg, __FILE__, __LINE__, PEAR_LOG_ERR);
            shutdown(1, $msg);
        }
        checkSMTPResponse($smtp, 250);

        if (PEAR::isError($error = $smtp->mailFrom($sender))) {
            $msg = "Failed to set sender \"$sender\": " . $error->getMessage();
            Horde::logMessage($msg, __FILE__, __LINE__, PEAR_LOG_ERR);
            shutdown(1, $msg);
        }
        checkSMTPResponse($smtp, 250);

        if (PEAR::isError($error = $smtp->rcptTo($recip))) {
            $msg = "Failed to set recipient \"$recip\": " . $error->getMessage();
            Horde::logMessage($msg, __FILE__, __LINE__, PEAR_LOG_ERR);
            shutdown(1, $msg);
        }
        checkSMTPResponse($smtp, 250);

        if (PEAR::isError($error = $smtp->data($data))) {
            $msg = 'Failed to send data: ' . $error->getMessage();
            Horde::logMessage($msg, __FILE__, __LINE__, PEAR_LOG_ERR);
            shutdown(1, $msg);
        }
        checkSMTPResponse($smtp, 250);

        $smtp->disconnect();
    }
}

function &getICal($sender,$resource)
{
    global $requestText, $conf;

    $mime = &MIME_Structure::parseTextMIMEMessage($requestText);

    $parts = $mime->contentTypeMap();
    foreach ($parts as $mimeid => $conttype) {
        if ($conttype == 'text/calendar') {
            $part = $mime->getPart($mimeid);

            $iCalendar = &new Horde_iCalendar();
            $iCalendar->parsevCalendar($part->transferDecode());

            return $iCalendar;
        }
    }
    
    // No iCal found
    $ical = false;
    return $ical;
}

/** Helper function */
function assembleUri($parsed)
{
    if (!is_array($parsed)) return false;

    $uri = empty($parsed['scheme']) ? '' :
        $parsed['scheme'] . ':' . ((strtolower($parsed['scheme']) == 'mailto') ? '' : '//');

    $uri .= empty($parsed['user']) ? '' :
        ($parsed['user']) .
        (empty($parsed['pass']) ? '' : ':'.($parsed['pass']))
        . '@';

    $uri .= empty($parsed['host']) ? '' :
        $parsed['host'];
    $uri .= empty($parsed['port']) ? '' :
        ':' . $parsed['port'];

    $uri .= empty($parsed['path']) ? '' :
        $parsed['path'];
    $uri .= empty($parsed['query']) ? '' :
        '?' . $parsed['query'];
    $uri .= empty($parsed['anchor']) ? '' :
        '#' . $parsed['anchor'];

    return $uri;
}

function &getFreeBusy($resource) {
    global $conf;
    return internalGetFreeBusy( $resource, $conf['filter']['freebusy_url']);
}

function &triggerFreeBusy($resource) {
    global $conf;
    return internalGetFreeBusy( $resource, $conf['filter']['pfb_trigger_url']);
}

function &internalGetFreeBusy($resource, $url)
{
    global $conf, $calmbox;

    if( ereg( 'user/[^/].*/(.*)', $calmbox, $regs ) ) {
        // Folder in INBOX
        $folder = $regs[1];
    } else {
        // Best guess, probably wont work
        $folder = $calmbox;
    }
    $url = str_replace('${USER}', urlencode($resource), $url);
    $url = str_replace('${FOLDER}', urlencode($folder), $url);

    Horde::logMessage(sprintf(_("URL = %s"), $url), __FILE__, __LINE__, PEAR_LOG_DEBUG);

    $parsed = parse_url($url);

    list($user, $domain) = split('@', $resource);
    if (empty($domain)) {
        $domain = $conf['filter']['email_domain'];
    }
    $parsed['user'] = urlencode($conf['filter']['calendar_id'] . '@' . $domain);
    $parsed['pass'] = urlencode($conf['filter']['calendar_pass']);

    $url = assembleUri($parsed);

    Horde::logMessage(sprintf(_("URL = %s"), $url), __FILE__, __LINE__, PEAR_LOG_DEBUG);

    $text = @file_get_contents($url);
    if ($text == false || empty($text)) {
        $parsed = parse_url($url);
        $parsed['pass'] = 'XXX';
        $url = assembleUri($parsed);        
        Horde::logMessage(sprintf(_("Unable to retrieve free/busy information for %s from %s"), 
                                  $resource, $url), 
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        //shutdown(1, "Unable to retrieve free/busy information for $resource");
        $result = false;
        return $result;
    }

    $iCalendar = &new Horde_iCalendar();
    $iCalendar->parsevCalendar($text);
    $vfb = &$iCalendar->findComponent('VFREEBUSY');
   
    if ($vfb === false) {
        Horde::logMessage(sprintf(_("Invalid or no free/busy information available for %s"), 
                                  $resource), 
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        //shutdown();
        $result = false;
        return $result;
    }
    $vfb->simplify();

    return $vfb;
}

function sendITipReply($cn,$resource, $itip, $type = RM_ITIP_ACCEPT)
{
    global $organiser, $uid, $sid, $is_update;

    Horde::logMessage(sprintf(_("sendITipReply(%s, %s, %s, %s)"), 
                              $cn, $resource, get_class($itip), $type), 
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    // Build the reply.
    $vCal = &new Horde_iCalendar();
    $vCal->setAttribute('PRODID', '-//proko2//resmgr 1.0//EN');
    $vCal->setAttribute('METHOD', 'REPLY');

    $summary = _('No summary available');

    $itip_reply =& Horde_iCalendar::newComponent('VEVENT', $vCal);
    $itip_reply->setAttribute('UID', $sid);
    if (!is_a($itip->getAttribute('SUMMARY'), 'PEAR_error')) {
        $itip_reply->setAttribute('SUMMARY', $itip->getAttribute('SUMMARY'));
        $summary = $itip->getAttribute('SUMMARY');
    }
    if (!is_a($itip->getAttribute('DESCRIPTION'), 'PEAR_error')) {
        $itip_reply->setAttribute('DESCRIPTION', $itip->getAttribute('DESCRIPTION'));
    }
    if (!is_a($itip->getAttribute('LOCATION'), 'PEAR_error')) {
        $itip_reply->setAttribute('LOCATION', $itip->getAttribute('LOCATION'));
    }
    $itip_reply->setAttribute('DTSTART', $itip->getAttribute('DTSTART'), array_pop($itip->getAttribute('DTSTART', true)));
    if (!is_a($itip->getAttribute('DTEND'), 'PEAR_error')) {
        $itip_reply->setAttribute('DTEND', $itip->getAttribute('DTEND'), array_pop($itip->getAttribute('DTEND', true)));
    } else {
        $itip_reply->setAttribute('DURATION', $itip->getAttribute('DURATION'), array_pop($itip->getAttribute('DURATION', true)));
    }
    if (!is_a($itip->getAttribute('SEQUENCE'), 'PEAR_error')) {
        $itip_reply->setAttribute('SEQUENCE', $itip->getAttribute('SEQUENCE'));
    } else {
        $itip_reply->setAttribute('SEQUENCE', 0);
    }
    $itip_reply->setAttribute('ORGANIZER', $itip->getAttribute('ORGANIZER'), array_pop($itip->getAttribute('ORGANIZER', true)));

    // Let's try and remove this code and just create
    // the ATTENDEE stuff in the reply from scratch   
    //     $attendees = $itip->getAttribute( 'ATTENDEE' );
    //     if( !is_array( $attendees ) ) {
    //       $attendees = array( $attendees );
    //     }
    //     $params = $itip->getAttribute( 'ATTENDEE', true );
    //     for( $i = 0; $i < count($attendees); $i++ ) {
    //       $attendee = preg_replace('/^mailto:\s*/i', '', $attendees[$i]);
    //       if ($attendee != $resource) {
    // 	continue;
    //       }
    //       $params = $params[$i];
    //       break;
    //     }

    $params = array();
    $params['CN'] = $cn;
    switch ($type) {
    case RM_ITIP_DECLINE:
        Horde::logMessage(sprintf(_("Sending DECLINE iTip reply to %s"), 
                                  $organiser), 
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
        $message = $is_update ? sprintf(_("%s has declined the update to the following event:\r\n\r\n%s"), $resource, $summary) :
            sprintf(_("%s has declined the invitation to the following event:\r\n\r\n%s"), $resource, $summary);
        $subject = 'Declined: ' . $summary;
        $params['PARTSTAT'] = 'DECLINED';
        break;

    case RM_ITIP_ACCEPT:
        Horde::logMessage(sprintf(_("Sending ACCEPT iTip reply to %s"), 
                                  $organiser), 
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
        $message = $is_update ? sprintf(_("%s has accepted the update to the following event:\r\n\r\n%s"), $resource, $summary) :
            sprintf(_("%s has accepted the invitation to the following event:\r\n\r\n%s"), $resource, $summary);
        $subject = 'Accepted: ' . $summary;
        $params['PARTSTAT'] = 'ACCEPTED';
        break;

    case RM_ITIP_TENTATIVE:
        Horde::logMessage(sprintf(_("Sending TENTATIVE iTip reply to %s"), 
                                  $organiser), 
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
        $message = $is_update ? sprintf(_("%s has tentatively accepted the update to the following event:\r\n\r\n%s"), $resource, $summary) :
            sprintf(_("%s has tentatively accepted the invitation to the following event:\r\n\r\n%s"), $resource, $summary);
        $subject = 'Tentative: ' . $summary;
        $params['PARTSTAT'] = 'TENTATIVE';
        break;

    default:
        Horde::logMessage(sprintf(_("Unknown iTip method (%s) passed to sendITipReply()"), 
                                  $type), 
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        shutdown(1, "Unknown iTip method ($type) passed to sendITipReply()");
    }

    $itip_reply->setAttribute('ATTENDEE', 'MAILTO:' . $resource, $params);
    $vCal->addComponent($itip_reply);

    $ics = &new MIME_Part('text/calendar', $vCal->exportvCalendar(), 'UTF-8' );
    //$ics->setName('event-reply.ics');
    $ics->setContentTypeParameter('method', 'REPLY');

    //$mime->addPart($body);
    //$mime->addPart($ics);
    // The following was ::convertMimePart($mime). This was removed so that we
    // send out single-part MIME replies that have the iTip file as the body,
    // with the correct mime-type header set, etc. The reason we want to do this
    // is so that Outlook interprets the messages as it does Outlook-generated
    // responses, i.e. double-clicking a reply will automatically update your
    // meetings, showing different status icons in the UI, etc.
    $mime = &MIME_Message::convertMimePart($ics);
    $mime->setCharset('UTF-8');
    $mime->setTransferEncoding('quoted-printable');
    $mime->transferEncodeContents();

    // Build the reply headers.
    $msg_headers = &new MIME_Headers();
    $msg_headers->addReceivedHeader();
    $msg_headers->addMessageIdHeader();
    $msg_headers->addHeader('Date', date('r'));
    $msg_headers->addHeader('From', "$cn <$resource>");
    $msg_headers->addHeader('To', $organiser);
    $msg_headers->addHeader('Subject', $subject);
    $msg_headers->addMIMEHeaders($mime);

    // Send the reply.
    static $mailer;

    if (!isset($mailer)) {
        require_once 'Mail.php';
        $mailer = &Mail::factory('SMTP', array('auth' => false));
    }

    $msg = $mime->toString();
    if (is_object($msg_headers)) {
        $headerArray = $mime->encode($msg_headers->toArray(), $mime->getCharset());
    } else {
        $headerArray = $mime->encode($msg_headers, $mime->getCharset());
    }

    /* Make sure the message has a trailing newline. */
    if (substr($msg, -1) != "\n") {
        $msg .= "\n";
    }

    $status = $mailer->send(MIME::encodeAddress($organiser), $headerArray, $msg);

    //$status = $mime->send($organiser, $msg_headers);
    if (is_a($status, 'PEAR_Error')) {
        Horde::logMessage(sprintf(_("Unable to send iTip reply: %s"), 
                                  $status->getMessage()), 
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        shutdown(1, "Unable to send iTip reply: " . $status->getMessage());
    } else {
        Horde::logMessage(_("Successfully sent iTip reply"), 
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
    }
}

function imapConnect($resource, $inbox = false)
{
    global $conf, $imap, $server, $mailbox, $calmbox, $prefix, $suffix;

    // Handle virtual domains
    list($user, $domain) = split('@', $resource);
    if (empty($domain)) {
        $domain = $conf['filter']['email_domain'];
    }
    $calendar_user = $conf['filter']['calendar_id'] . '@' . $domain;

    // Handle virtual domains
    $prefix = $resource;
    $suffix = '';
    if ($conf['filter']['virtual_domains']) {
        list($prefix, $suffix) = split('@', $resource);
        if ($conf['filter']['append_domains'] && !empty($suffix)) {
            $suffix = '@' . $suffix;
        } else {
            $suffix = '';
        }
    }

    // Get our mailbox strings for use in the imap_X functions
    $server = '{' . $conf['filter']['imap_server'] . '/imap/notls/novalidate-cert/norsh}';
    if ($inbox) {
        $mymailbox = "user/$prefix$suffix";
    } else {
        $mymailbox = "user/$prefix/" . $conf['filter']['calendar_store'] . "$suffix";
    }
    //$mailbox = "INBOX/Calendar";
    $fullmbox = $server . $mymailbox;

    Horde::logMessage(sprintf(_("Logging into %s as user %s to access %s"),
                              $conf['filter']['imap_server'], $calendar_user, $mymailbox),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    $imap = &new Net_IMAP( $conf['filter']['imap_server'] );
    if( PEAR::isError($imap) ) {
        Horde::logMessage(sprintf(_("Unable to create Net_IMAP object:  %s"),
                                  $imap->getMessage()),
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        return false;
    }
    //$imap->setDebug(true);
    $rc = $imap->login($calendar_user, $conf['filter']['calendar_pass'], true, false);
    if( PEAR::isError($rc) ) {
        Horde::logMessage(sprintf(_("Unable to authenticate:  %s"),
                                  $rc->getMessage()),
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        return false;      
    }
    $mailboxes = $imap->getMailBoxes( "user/$prefix$suffix/" );
    if( PEAR::isError( $mailboxes ) ) {
        Horde::logMessage(sprintf(_("Unable to get mailbox list: %s"),
                                  $mailboxes->getMessage()),
                          __FILE__, __LINE__, PEAR_LOG_ERR);
    }
    $calmbox = false;
    Horde::logMessage(sprintf(_("Mailboxes under account: %s"),
                              print_r($mailboxes, true)),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    foreach( $mailboxes as $mailbox ) {
        $a = $imap->getAnnotation('/vendor/kolab/folder-type',
                                  /*array('value.shared' => 'event.default')*/'value.shared',
                                  $mailbox);
        Horde::logMessage(sprintf(_("/vendor/kolab/folder-type annotation for folder %s is %s"),
                                  $mailbox, print_r($a, true)),
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
        if( $a == 'event.default' ) {
            // Found default calendar!
            $calmbox = $mailbox;
            break;
        }
    }

    if( !$calmbox ) {
        // No default calendar, try to create one
        $calmbox = "user/$prefix/" . $conf['filter']['calendar_store'] . "$suffix";
        if( !in_array( $calmbox, $mailboxes ) ) {
            // Create mailbox
            $rc = $imap->createMailBox( $calmbox );
            if( PEAR::isError($rc) && PEAR::isError( $imap->selectMailBox($calmbox) ) ) {
                /* Notice that in case of an error, we check if
                 we can select the mailbox. This is to test
                 if CREATE failed because the mailbox already
                 exists (but had the wrong annotation)
                */
                Horde::logMessage(sprintf(_("IMAP Errors from createMailBox(%s): %s"),
                                          $calmbox, $rc->getMessage()),
                                  __FILE__, __LINE__, PEAR_LOG_ERR);
                return false;
            }
        }
        $rc = $imap->setAnnotation('/vendor/kolab/folder-type',
                                   array('value.shared' => 'event.default'),
                                   $calmbox);
        if( PEAR::isError($rc)) {
            // Non fatal error
            Horde::logMessage(sprintf(_("Unable to set the folder-type annotation on mailbox %s: %s"),
                                      $mymailbox, $rc->getMessage()),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
        }
    }

    Horde::logMessage(sprintf(_("Selecting  %s for %s"),
                              $calmbox, $calendar_user),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    // Open an IMAP connection to the requested users' calendar
    $rc = $imap->selectMailBox( $calmbox );
    if( PEAR::isError($rc)) {
        Horde::logMessage(sprintf(_("Error selecting %s: %s"),
                                  $calmbox, $rc->getMessage()),
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        return false;
    }

    Horde::logMessage(sprintf(_("Successfully connected to %s."),
                              $calmbox),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    return $imap;
}

function expire_events( $imap, $when )
{
    // PENDING(steffen): Think about how to add event expiry
    // without serious overhead and without deleting events from 
    // normal user accounts
}

// resource error when date is used and no timestring

function cleanArray($ical_date)
{
	if ( array_key_exists('hour',$ical_date) ) {
		$temp['hour']   = array_key_exists('hour',$ical_date) ? $ical_date['hour'] :  '00';
		$temp['minute']   = array_key_exists('minute',$ical_date) ? $ical_date['minute'] :  '00';
		$temp['second']   = array_key_exists('second',$ical_date) ? $ical_date['second'] :  '00';
		$temp['zone']   = array_key_exists('zone',$ical_date) ? $ical_date['zone'] :  'UTC';
	} else { 
		$temp['DATE'] = '1';
	}
	$temp['year']   = array_key_exists('year',$ical_date) ? $ical_date['year'] :  '0000';
	$temp['month']   = array_key_exists('month',$ical_date) ? $ical_date['month'] :  '00';
	$temp['mday']   = array_key_exists('mday',$ical_date) ? $ical_date['mday'] :  '00';

	return $temp;
}



/*

An all day event must have a dd--mm-yyyy notation and not a yyyy-dd-mmT00:00:00z notation
Otherwise the event is shown as a 2-day event
--> do not try to convert everything to epoch first !!
*/

function iCalDate2Kolab($ical_date,$type= ' ')
{
    Horde::logMessage(sprintf(_("Converting to kolab format %s"),
                              print_r($ical_date, true)),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    // $ical_date should be a timestamp
    if (is_array($ical_date)) {
# going to create date again
        $temp=cleanArray($ical_date);
    	if (array_key_exists('DATE',$temp)) {
            if ($type == 'ENDDATE') {
# substract a day (86400 seconds) using epochs to take number of days per month into account
                $epoch=convert2epoch($temp)-86400;
        		$date = gmstrftime('%Y-%m-%d', $epoch);
            } else {
		    	$date= sprintf('%04d-%02d-%02d',$temp['year'], $temp['month'], $temp['mday']);
            }
        } else {
            $time = sprintf('%02d:%02d:%02d', $temp['hour'], $temp['minute'], $temp['second']);
	        if ($temp['zone'] == 'UTC') {
        	    $time .= 'Z';
        	}
        	$date= sprintf('%04d-%02d-%02d',$temp['year'], $temp['month'], $temp['mday']). "T".$time;
        }
    }  else {
        $date = gmstrftime('%Y-%m-%dT%H:%M:%SZ', $ical_date);
    }
    Horde::logMessage(sprintf(_("To <%s>"), $date),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    return $date;
}


function convert2epoch($values)
{
    Horde::logMessage(sprintf(_("Converting to epoch %s"),
                              print_r($values, true)),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    if (is_array($values)) {
        $temp=cleanArray($values);
        $epoch=gmmktime($temp['hour'],$temp['minute'],$temp['second'],$temp['month'],$temp['mday'],$temp['year']);
    } else { $epoch=$values;} 
    
    Horde::logMessage(sprintf(_("Converted <%s>"), $epoch),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    return $epoch;
}

function &buildKolabEvent(&$itip)
{
    global $organiser, $resource, $uid, $sid;

    require_once('Horde/DOM.php');

    $recurrence = false;

    $kolab_xml = Horde_DOM_Document::factory();
    if (is_a($kolab_xml, 'PEAR_Error')) {
        // There were errors building the xml document
        Horde::logMessage(sprintf(_("Error building xml document: %s"), 
                                  $result->getMessage()),
                          __FILE__, __LINE__, PEAR_LOG_ERR);
        return false;
    }
    
    $kolab_event = $kolab_xml->append_child($kolab_xml->create_element('event'));
    $kolab_event->set_attribute('version', '1.0');

    // We're not interested in too much information about the event; the only
    // critical things are the UID and the time spans. We include the
    // summary, body and organizer for display purposes (e.g. in the free/busy viewer).
    //
    // Actually, we need everything! /steffen
    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('uid'));
    $kolab_node->append_child($kolab_xml->create_text_node($uid));

    // No SID anymore
    //$kolab_node = $kolab_event->append_child($kolab_xml->create_element('scheduling-id'));
    //$kolab_node->append_child($kolab_xml->create_text_node($sid));

    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('organizer'));
    $org_params = $itip->getAttribute('ORGANIZER', true);
    if( !is_a( $org_params, 'PEAR_Error' ) ) {
        $orgcn = $org_params[0]['CN'];
        $orgnzr_node = $kolab_node->append_child($kolab_xml->create_element('display-name'));
        $orgnzr_node->append_child($kolab_xml->create_text_node($orgcn));
    }
    $orgnzr_node = $kolab_node->append_child($kolab_xml->create_element('smtp-address'));
    $orgemail = $itip->getAttributeDefault('ORGANIZER', '');
    if( eregi('mailto:(.*)', $orgemail, $regs ) ) $orgemail = $regs[1];
    $orgnzr_node->append_child($kolab_xml->create_text_node( $orgemail ));

    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('summary'));
    $kolab_node->append_child($kolab_xml->create_text_node(
                                  $itip->getAttributeDefault('SUMMARY', '')
                              ));

    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('location'));
    $kolab_node->append_child($kolab_xml->create_text_node(
                                  $itip->getAttributeDefault('LOCATION', '')
                              ));

    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('body'));
    $kolab_node->append_child($kolab_xml->create_text_node(
                                  $itip->getAttributeDefault('DESCRIPTION', '')
                              ));

    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('start-date'));
    $kolab_node->append_child($kolab_xml->create_text_node(
                                  iCalDate2Kolab($itip->getAttributeDefault('DTSTART', 0))
                              ));

    $kolab_node = $kolab_event->append_child($kolab_xml->create_element('end-date'));
    $kolab_node->append_child($kolab_xml->create_text_node(
                                  iCalDate2Kolab($itip->getAttributeDefault('DTEND', 0),'ENDDATE')
                              ));

# default is busy, so set free if info is available
    if ($itip->getAttributeDefault('TRANSP', 'OPAQUE') == 'TRANSPARENT') {
        $kolab_node = $kolab_event->append_child($kolab_xml->create_element('show-time-as'));
        $kolab_node->append_child($kolab_xml->create_text_node('free' ));
    }
    
    // Attendees
    $attendees = $itip->getAttribute('ATTENDEE');
    if( !is_a( $attendees, 'PEAR_Error' ) ) {
        $attendees_params = $itip->getAttribute('ATTENDEE', true);
        if( !is_array( $attendees ) ) $attendees = array( $attendees );
        if( !is_array( $attendees_params ) ) $attendees_params = array( $attendees_params );
        for( $i = 0; $i < count($attendees); $i++ ) {
            $attendee_node = $kolab_event->append_child($kolab_xml->create_element('attendee'));	
            $dispname_node = $attendee_node->append_child($kolab_xml->create_element('display-name'));	
            $dispname_node->append_child($kolab_xml->create_text_node($attendees_params[$i]['CN']));
            $kolab_node = $attendee_node->append_child($kolab_xml->create_element('smtp-address'));
            $attendeeemail = $attendees[$i];
            if( eregi('mailto:(.*)',$attendeeemail,$regs) ) {
                $attendeeemail = $regs[1];
            }
            $kolab_node->append_child($kolab_xml->create_text_node($attendeeemail));
            $kolab_node = $attendee_node->append_child($kolab_xml->create_element('status'));
            $kolab_node->append_child($kolab_xml->create_text_node(strtolower($attendees_params[$i]['PARTSTAT'])));
            $kolab_node = $attendee_node->append_child($kolab_xml->create_element('request-response'));
            if( $attendees_params[$i]['RSVP'] == 'FALSE' ) {
                $kolab_node->append_child($kolab_xml->create_text_node('false'));
            } else {
                $kolab_node->append_child($kolab_xml->create_text_node('true'));	  
            }
            $kolab_node = $attendee_node->append_child($kolab_xml->create_element('role'));
            $kolab_node->append_child($kolab_xml->create_text_node(strtolower($attendees_params[$i]['ROLE'])));
        }
    }

    // Alarm
    $valarm = $itip->findComponent('VALARM');
    if( $valarm ) {
        $trigger = $valarm->getAttribute('TRIGGER');
        if( !PEAR::isError($trigger) ) {
            $p = $valarm->getAttribute('TRIGGER',true);
            if( $trigger < 0 ) {
                // All OK, enter the alarm into the XML
                // NOTE: The Kolab XML format seems underspecified
                // wrt. alarms currently...
                $kolab_node = $kolab_event->append_child($kolab_xml->create_element('alarm'));
                $kolab_node->append_child($kolab_xml->create_text_node((int)(-$trigger/60)));
            }
        } else {
            Horde::logMessage(sprintf(_("No TRIGGER in VALARM"), 
                                      $trigger->getMessage()),
                __FILE__, __LINE__, PEAR_LOG_ERR);
        }
    }

    // Recurrence
    $rrule_str = $itip->getAttribute('RRULE');
    if( !is_a( $rrule_str, 'PEAR_Error' ) ) {
        $kolab_days = array( 'MO' => 'monday',
                             'TU' => 'tuesday',
                             'WE' => 'wednesday',
                             'TH' => 'thursday',
                             'FR' => 'friday',
                             'SA' => 'saturday',
                             'SU' => 'sunday');
        $kolab_months = array( 1  => 'january',
                               2  => 'february',
                               3  => 'march',
                               4  => 'april',
                               5  => 'may',
                               6  => 'june',
                               7  => 'july',
                               8  => 'august',
                               9  => 'september',
                               10 => 'october',
                               11 => 'november',
                               12 => 'december' );
        $rrule_list = split(';', $rrule_str);
        $rrule = array();
        foreach( $rrule_list as $r ) {
            list( $k, $v ) = split( '=', $r );
            $rrule[$k]=$v;
            Horde::logMessage(sprintf(_("RRULE[%s]=%s"), 
                                      $k, $v),
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
        }
        $recur_node = $kolab_event->append_child($kolab_xml->create_element('recurrence'));
        $recurrence =& new ResmgrRecurrence();
        $freq = strtolower($rrule['FREQ']);
        $recur_node->append_child( $kolab_xml->create_attribute( 'cycle', $freq ));
        $recurrence->setCycletype( $freq );
        $recurrence->setStartdate( convert2epoch($itip->getAttributeDefault('DTSTART', 0)) );
        $recurrence->setEnddate( convert2epoch($itip->getAttributeDefault('DTEND', 0) ));
        switch( $freq ) {
        case 'daily':
            break;
        case 'weekly':
            $days = split(',', $rrule['BYDAY']);
            $kdays = array();
            foreach( $days as $day ) {
                $day_node = $recur_node->append_child( $kolab_xml->create_element( 'day'));	  
                $day_node->append_child($kolab_xml->create_text_node($kolab_days[$day]));
                $kdays[] = $kolab_days[$day];
            }
            $recurrence->setDay( $kdays );
            break;
        case 'monthly':
            if( $rrule['BYDAY'] ) {
                $recur_node->append_child( $kolab_xml->create_attribute( 'type', 'weekday' ));
                $recurrence->setType( 'weekday' );
                $wdays = split(',', $rrule['BYDAY']);
                $kdays = array();
                $kdaynumbers = array();
                foreach( $wdays as $wday ) {
                    if( ereg('([+-]?[0-9])(.*)', $wday, $regs ) ) {
                        $daynumber_node = $recur_node->append_child( $kolab_xml->create_element( 'daynumber'));
                        $daynumber_node->append_child($kolab_xml->create_text_node( $regs[1] ));	      
                        $day_node = $recur_node->append_child( $kolab_xml->create_element( 'day'));
                        $day_node->append_child($kolab_xml->create_text_node( $kolab_days[$regs[2]] ));	      
                        $kdaynumbers[] = $regs[1];
                        $kdays[] = $kolab_days[$regs[2]];
                    } else {
                        $day_node = $recur_node->append_child( $kolab_xml->create_element( 'day'));
                        $day_node->append_child($kolab_xml->create_text_node( $wday ));	      
                        $kdays[] = $wday;
                    }
                }
                if( !empty($kdaynumbers) ) $recurrence->setDaynumber($kdaynumbers);
                $recurrence->setDay( $kday );
            } else if( $rrule['BYMONTHDAY'] ) {
                $recur_node->append_child( $kolab_xml->create_attribute( 'type', 'daynumber' ));
                $recurrence->setType( 'daynumber' );
                $daynumbers = split(',', $rrule['BYMONTHDAY']);
                $kdaynumbers = array();
                foreach( $daynumbers as $daynumber ) {
                    $daynumber_node = $recur_node->append_child( $kolab_xml->create_element( 'daynumber'));
                    $daynumber_node->append_child($kolab_xml->create_text_node( $daynumber ));
                    $kdaynumbers[] = $daynumber;
                }
                $recurrence->setDaynumber($kdaynumbers);
            }
            break;
        case 'yearly':
            if( $rrule['BYDAY'] ) {
                $recur_node->append_child( $kolab_xml->create_attribute( 'type', 'weekday' ));
                $recurrence->setType( 'weekday' );
                $wdays = split(',', $rrule['BYDAY']);
                $kdays = array();
                $kdaynumbers = array();
                foreach( $wdays as $wday ) {
                    if( ereg('([+-]?[0-9])(.*)', $wday, $regs ) ) {
                        $daynumber_node = $recur_node->append_child( $kolab_xml->create_element( 'daynumber'));
                        $daynumber_node->append_child($kolab_xml->create_text_node( $regs[1] ));	      
                        $day_node = $recur_node->append_child( $kolab_xml->create_element( 'day'));
                        $day_node->append_child($kolab_xml->create_text_node( $kolab_days[$regs[2]] ));	      
                        $kdaynumbers[] = $regs[1];
                        $kdays[] = $kolab_days[$regs[2]];
                    } else {
                        $day_node = $recur_node->append_child( $kolab_xml->create_element( 'day'));
                        $day_node->append_child($kolab_xml->create_text_node( $wday ));	      
                        $kdays[] = $wday;
                    }
                }
                if( !empty($kdaynumbers) ) $recurrence->setDaynumber($kdaynumbers);
                $recurrence->setDay( $kday );
                $monthnumbers = split(',', $rrule['BYMONTH']);
                $kmonth = array();
                foreach( $monthnumbers as $monthnumber ) {
                    $month = $kolab_months[$monthnumber];
                    $month_node = $recur_node->append_child( $kolab_xml->create_element( 'month'));
                    $month_node->append_child($kolab_xml->create_text_node( $month ));
                    $kmonth[] = $month;
                }
                $recurrence->setMonth($kmonth);
            } else if( $rrule['BYMONTHDAY'] ) {
                $recur_node->append_child( $kolab_xml->create_attribute( 'type', 'monthday' ));
                $recurrence->setType( 'monthday' );
                $daynumbers = split(',', $rrule['BYMONTHDAY']);
                $kdaynumbers = array();
                foreach( $daynumbers as $daynumber ) {
                    $daynumber_node = $recur_node->append_child( $kolab_xml->create_element( 'daynumber'));
                    $daynumber_node->append_child($kolab_xml->create_text_node( $daynumber ));
                    $kdaynumbers[] = $daynumber;
                }
                $recurrence->setDaynumber($kdaynumbers);
                $monthnumbers = split(',', $rrule['BYMONTH']);
                $kmonth = array();
                foreach( $monthnumbers as $monthnumber ) {
                    $month = $kolab_months[$monthnumber];
                    $month_node = $recur_node->append_child( $kolab_xml->create_element( 'month'));
                    $month_node->append_child($kolab_xml->create_text_node( $month ));
                    $kmonth[] = $month;
                }
                $recurrence->setMonth($kmonth);
            } else if( $rrule['BYYEARDAY'] ) {
                $recur_node->append_child( $kolab_xml->create_attribute( 'type', 'yearday' ));
                $recurrence->setType( 'yearday' );
                $daynumbers = split(',', $rrule['BYMONTHDAY']);
                $kdaynumbers = array();
                foreach( $daynumbers as $daynumber ) {
                    $daynumber_node = $recur_node->append_child( $kolab_xml->create_element( 'daynumber'));
                    $daynumber_node->append_child($kolab_xml->create_text_node( $daynumber ));
                    $kdaynumbers[] = $daynumber;
                }
                $recurrence->setDaynumber($kdaynumbers);	  
            }
            break;
        default:
            // Not supported
        }
        $interval_node = $recur_node->append_child( $kolab_xml->create_element( 'interval' ));
        $interval_node->append_child($kolab_xml->create_text_node($rrule['INTERVAL']));
        $recurrence->setInterval( $rrule['INTERVAL']);

        $range_node = $recur_node->append_child( $kolab_xml->create_element( 'range' ));
        if( $rrule['COUNT'] ) {
            $range_node->append_child( $kolab_xml->create_attribute( 'type', 'number' ));      
            $range_node->append_child($kolab_xml->create_text_node($rrule['COUNT']));
            $recurrence->setRangetype('number');
            $recurrence->setRange( $rrule['COUNT']);
        } else if( $rrule['UNTIL'] ) {
            $range_node->append_child( $kolab_xml->create_attribute( 'type', 'date' ));      
            $range_node->append_child($kolab_xml->create_text_node(gmstrftime('%Y-%m-%d', $itip->_parseDateTime($rrule['UNTIL']))));
            $recurrence->setRangetype('date');
            $recurrence->setRange( $rrule['UNTIL']);
        } else {
            $range_node->append_child( $kolab_xml->create_attribute( 'type', 'none' ));      
            $recurrence->setRangetype('none');
        }
        $exclusions = $itip->getAttribute('EXDATE');
        if( !is_a( $exclusions, 'PEAR_Error' ) ) {
            if( !is_array( $exclusions ) ) $exclusions = array( $exclusions );
            foreach( $exclusions as $ex ) {
                Horde::logMessage(sprintf(_("ex=%s, ical=%s"), 
                                          print_r($ex[0], true), iCalDate2Kolab($ex[0])),
                                  __FILE__, __LINE__, PEAR_LOG_DEBUG);
                $ex_node = $recur_node->append_child( $kolab_xml->create_element( 'exclusion' ) );
                $kolab_date = $ex[0]['year'].'-'.$ex[0]['month'].'-'.$ex[0]['mday'];
                $ex_node->append_child($kolab_xml->create_text_node( $kolab_date ) );
            }
        }
    }

    $result = array($kolab_xml,$recurrence);

    return $result;
}


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

function resmgr_filter( $fqhostname, $sender, $resource, $tmpfname ) {
    global $conf, $ldapdata,$calmbox,$organiser,$uid,$sid,$requestText;

    // Make Horde happy
    $conf['server']['name'] = $conf['filter']['email_domain'];

    // Set some parameters
    $ldapdata = &getLDAPData($sender,$resource);
    if( PEAR::isError($ldapdata) ) return $ldapdata;
    else if( $ldapdata === false ) {
        // No data, probably not a local user
        return true;
    } else if( (boolean)strtolower($ldapdata['homeserver']) && strtolower($ldapdata['homeserver']) != $fqhostname ) {
        // Not the users homeserver, ignore
        return true;
    }

    $cn = $ldapdata['cn'];
    $conf['filter']['action'] = $ldapdata['action'];
    Horde::logMessage(sprintf(_("Action for %s is %s"), 
                              $sender, actiontostring($conf['filter']['action'])),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
    if( !$conf['filter']['action'] ) {
        // Manual is the only safe default!
        $conf['filter']['action'] = RM_ACT_MANUAL;
    }

    // Get out as early as possible if manual
    if( $conf['filter']['action'] == RM_ACT_MANUAL ) {
        Horde::logMessage(sprintf(_("Passing through message to  %s"), 
                                  $resource),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
        return true;
    }

    $requestText = '';

    // Get our request from tmp file
    getRequest($tmpfname);

    // Get the iCalendar data (i.e. the iTip request)
    $iCalendar = &getICal($sender,$resource);
    if( $iCalendar === false ) {
        // No iCal in mail
        Horde::logMessage(sprintf(_("Could not parse iCalendar data, passing through to %s"), 
                                  $resource),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
        return true;
    }
    // Get the event details out of the iTip request
    $itip = &$iCalendar->findComponent('VEVENT');
    if ($itip === false) {
        Horde::logMessage(sprintf(_("No VEVENT found in iCalendar data, passing through to %s"), 
                                  $resource),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
        return true;
    }

    // What is the request's method? i.e. should we create a new event/cancel an
    // existing event, etc.
    $method = strtoupper($iCalendar->getAttributeDefault('METHOD',
                                                         $itip->getAttributeDefault('METHOD', 'REQUEST')));

    // What resource are we managing?
    Horde::logMessage(sprintf(_("Processing %s method for %s"), 
                              $method, $resource),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    // This is assumed to be constant across event creation/modification/deletipn
    $uid = $itip->getAttributeDefault('UID', '');
    Horde::logMessage(sprintf(_("Event has UID %s"), 
                              $sid),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    $sid = $uid;

    // Who is the organiser?
    $organiser = preg_replace('/^mailto:\s*/i', '', $itip->getAttributeDefault('ORGANIZER', ''));
    Horde::logMessage(sprintf(_("Request made by %s"), 
                              $organiser),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    // What is the events summary?
    $summary = $itip->getAttributeDefault('SUMMARY', '');

    $dtstart = convert2epoch ($itip->getAttributeDefault('DTSTART', 0));
    $dtend = convert2epoch ($itip->getAttributeDefault('DTEND', 0));

    Horde::logMessage(sprintf(_("Event starts on <%s> %s and ends on <%s> %s."), 
                              $dtstart, iCalDate2Kolab($dtstart), $dtend, iCalDate2Kolab($dtend)),
                      __FILE__, __LINE__, PEAR_LOG_DEBUG);

    if ($conf['filter']['action'] == RM_ACT_ALWAYS_REJECT) {
        if ($method == 'REQUEST') {
            Horde::logMessage(sprintf(_("Rejecting %s method"), 
                                      $method),
                              __FILE__, __LINE__, PEAR_LOG_INFO);
            sendITipReply($cn,$resource,$itip,RM_ITIP_DECLINE);
            return false;//shutdown(0);
        } else {
            Horde::logMessage(sprintf(_("Passing through %s method for ACT_ALWAYS_REJECT policy"), 
                                      $method),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
            return true;
        }
    }

    $is_update = false;
    $ignore = array();
    $imap = imapConnect($resource);
    $connected = ($imap !== false);
    if( !$connected ) {
        return PEAR::raiseError(_("Error, could not open calendar folder!"),
                                OUT_LOG | EX_TEMPFAIL);
    }
    switch ($method) {
    case 'REQUEST':
        if ($conf['filter']['action'] == RM_ACT_MANUAL) {
            Horde::logMessage(sprintf(_("Passing through %s method"), 
                                  $method),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
            break;
        }

        // Check if the request is actually an update of an existing event
        $updated_messages = $imap->search('SUBJECT "' . $sid . '"');
        if( PEAR::isError( $updated_messages ) ) {
            Horde::logMessage(sprintf(_("Error searching mailbox: %s"), 
                                      $updated_messages->getMessage()),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            $updated_messages = array();
        }
        if (!empty($updated_messages)) {
            Horde::logMessage(sprintf(_("Updating message %s"), 
                                  $sid),
                              __FILE__, __LINE__, PEAR_LOG_INFO);

            $ignore[] = $sid;
            $is_update = true;

            if (!is_array($updated_messages)) {
                $updated_messages = array($updated_messages);
            }
        }
        list($kolab_xml,$recurrence) = buildKolabEvent($itip);

        $outofperiod=0;

        // Don't even bother checking free/busy info if RM_ACT_ALWAYS_ACCEPT
        // is specified
        if ($conf['filter']['action'] != RM_ACT_ALWAYS_ACCEPT) {
            // Get the resource's free/busy list
            triggerFreeBusy($resource);
            $vfb = &getFreeBusy($resource);

            if (!$vfb) {
                return new PEAR_Error("Error building free/busy list");
            }

            $vfbstart = $vfb->getAttributeDefault('DTSTART', 0);
            $vfbend = $vfb->getAttributeDefault('DTEND', 0);
            Horde::logMessage(sprintf(_("Free/busy info starts on <%s> %s and ends on <%s> %s"), 
                                      $vfbstart, iCalDate2Kolab($vfbstart), $vfbend, iCalDate2Kolab($vfbend)),
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
	
            if ($vfbstart && $dtstart > convert2epoch ($vfbend)) {
                $outofperiod=1;
            } else {
                // Check whether we are busy or not
                $busyperiods = $vfb->getBusyPeriods();
                Horde::logMessage(sprintf(_("Busyperiods: %s"), 
                                          print_r($busyperiods, true)),
                                  __FILE__, __LINE__, PEAR_LOG_DEBUG);
                $extraparams = $vfb->getExtraParams();
                Horde::logMessage(sprintf(_("Extraparams: %s"), 
                                          print_r($extraparams, true)),
                                  __FILE__, __LINE__, PEAR_LOG_DEBUG);
                $conflict = false;	    
                if( $recurrence !== false ) {
                    $recurrence->busyperiods =& $busyperiods;
                    $recurrence->extraparams =& $extraparams;
                    $recurrence->ignore = $ignore;
                    Horde::logMessage(sprintf(_("Recurrence is %s"), 
                                              print_r($recurrence, true)),
                                      __FILE__, __LINE__, PEAR_LOG_DEBUG);
                    $recurrence->expand( $vfbstart, $vfbend );
                    $conflict = $recurrence->hasConflict();
                } else {	    
                    foreach ($busyperiods as $busyfrom => $busyto) {
                        Horde::logMessage(sprintf(_("Busy period from %s to %s"), 
                                                  strftime('%a, %d %b %Y %H:%M:%S %z', $busyfrom),
                                                  strftime('%a, %d %b %Y %H:%M:%S %z', $busyto)
                                          ),
                                          __FILE__, __LINE__, PEAR_LOG_DEBUG);
                        if ( in_array(base64_decode($extraparams[$busyfrom]['X-UID']), $ignore) ||
                             in_array(base64_decode($extraparams[$busyfrom]['X-SID']), $ignore) ) {
                            // Ignore
                            continue;
                        }
                        if (($busyfrom >= $dtstart && $busyfrom < $dtend) || ($dtstart >= $busyfrom && $dtstart < $busyto)) {
                            Horde::logMessage(_("Request overlaps"), 
                                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
                            $conflict = true;
                            break;
                        }
                    }
                }
            }

            if ($conflict) {
                if ($conf['filter']['action'] == RM_ACT_MANUAL_IF_CONFLICTS) {
                    //sendITipReply(RM_ITIP_TENTATIVE);
                    Horde::logMessage(_("Conflict detected; Passing mail through"), 
                                      __FILE__, __LINE__, PEAR_LOG_INFO);
                    return true;
                } else if ($conf['filter']['action'] == RM_ACT_REJECT_IF_CONFLICTS) {
                    Horde::logMessage(_("Conflict detected; rejecting"), 
                                      __FILE__, __LINE__, PEAR_LOG_INFO);
                    sendITipReply($cn,$resource,$itip,RM_ITIP_DECLINE);
                    return false;//shutdown(0);
                }
            }
        }

        // At this point there was either no conflict or RM_ACT_ALWAYS_ACCEPT
        // was specified; either way we add the new event & send an 'ACCEPT'
        // iTip reply

        Horde::logMessage(sprintf(_("Adding event %s"), $sid),
                          __FILE__, __LINE__, PEAR_LOG_INFO);

        $message = &new MIME_Message();

        $kolabinfo = 'This is a Kolab Groupware object. To view this object you will need a client that understands the Kolab Groupware format. For a list of such clients please visit http://www.kolab.org/kolab2-clients.html';

        $body = &new MIME_Part('text/plain', wrap($kolabinfo));

        $part = &new MIME_Part('application/x-vnd.kolab.event', $kolab_xml->dump_mem(true));
        $part->setName('kolab.event');
        $part->setDisposition('attachment');

        $message->addPart($body);
        $message->addPart($part);

        $headers = &new MIME_Headers();
        $headers->addHeader("From", $resource);
        $headers->addHeader("To", $resource);
        $headers->addHeader("Subject", $sid);
        $headers->addHeader("User-Agent", "proko2/resmgr");
        $headers->addHeader("Reply-To", "");
        $headers->addHeader("Date", date("r"));
        $headers->addHeader("X-Kolab-Type", "application/x-vnd.kolab.event");
        $headers->addMIMEHeaders($message);

        $message = preg_replace("/\r\n|\n|\r/s", "\r\n",
                                $headers->toString() .
                                $message->toString(false)
        );

        Horde::logMessage(sprintf(_("Appending message to %s"), $calmbox),
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);

        $rc = $imap->appendMessage($message);
        if( PEAR::isError($rc) ) {
            Horde::logMessage(sprintf(_("Error appending message: %s"), 
                                      $rc->getMessage()),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
        }

        // Update our status within the iTip request and send the reply
        $itip->setAttribute('STATUS', 'CONFIRMED', array(), false);
        $attendees = $itip->getAttribute('ATTENDEE');
        if (!is_array($attendees)) {
            $attendees = array($attendees);
        }
        $attparams = $itip->getAttribute('ATTENDEE', true);
        foreach ($attendees as $i => $attendee) {
            $attendee = preg_replace('/^mailto:\s*/i', '', $attendee);
            if ($attendee != $resource) {
                continue;
            }

            $attparams[$i]['PARTSTAT'] = 'ACCEPTED';
            if (array_key_exists('RSVP', $attparams[$i])) {
                unset($attparams[$i]['RSVP']);
            }
        }

        // Re-add all the attendees to the event, using our updates status info
        $firstatt = array_pop($attendees);
        $firstattparams = array_pop($attparams);
        $itip->setAttribute('ATTENDEE', $firstatt, $firstattparams, false);
        foreach ($attendees as $i => $attendee) {
            $itip->setAttribute('ATTENDEE', $attendee, $attparams[$i]);
        }

        if ($outofperiod) {
            sendITipReply($cn,$resource,$itip,RM_ITIP_TENTATIVE);
            Horde::logMessage(_("No freebusy information available"), 
                              __FILE__, __LINE__, PEAR_LOG_NOTICE);
        } else {
            sendITipReply($cn,$resource,$itip,RM_ITIP_ACCEPT);
        }

        // Delete any old events that we updated
        if( !empty( $updated_messages ) ) {
            Horde::logMessage(sprintf(_("Deleting %s because of update."), 
                                      join(', ',$deleted_messages)),
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
            $imap->deleteMessages( $updated_messages );
            $imap->expunge();
        }
        // Get the resource's free/busy list
        // once more so it is up to date
        $imap->disconnect();
        unset($imap);
        if( !triggerFreeBusy($resource,false) ) {
            Horde::logMessage(_("Error updating fblist."),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
        }
        return false;//shutdown(0);

    case 'CANCEL':
        Horde::logMessage(sprintf(_("Removing event %s"),
                                  $sid),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
	
        // Try to delete the event
        $deleted_messages = $imap->search('SUBJECT "' . $sid . '"');
        if( PEAR::isError($deleted_messages) ) {
            Horde::logMessage(sprintf(_("Error searching mailbox: %s"),
                                      $deleted_messages->getMessage()),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            $deleted_messages = array();
        }

        /* Ensure that $summary is single byte here, else we get
         * problems with sprintf()
         */
        $summary = String::convertCharset($summary, 'utf-8', 'iso-8859-1');
        if (empty($deleted_messages)) {
            Horde::logMessage(sprintf(_("Canceled event %s is not present in %s's calendar"),
                                      $sid, $resource),
                              __FILE__, __LINE__, PEAR_LOG_WARNING);
            $body = sprintf(_("The following event that was canceled is not present in %s's calendar:\r\n\r\n%s"), $resource, $summary);
            $subject = sprintf(_("Error processing '%s'"), $summary);
        } else {
            $body = sprintf(_("The following event has been successfully removed:\r\n\r\n%s"), $summary);
            $subject = sprintf(_("%s has been cancelled"), $summary);
        }
        
        /* Switch back to utf-8 */
        $body = String::convertCharset($body, 'iso-8859-1');
        $subject = String::convertCharset($subject, 'iso-8859-1');

        if (!is_array($deleted_messages)) {
            $deleted_messages = array($deleted_messages);
        }

        Horde::logMessage(sprintf(_("Sending confirmation of cancelation to %s"),
                                  $organiser),
                          __FILE__, __LINE__, PEAR_LOG_WARNING);

        $body = &new MIME_Part('text/plain', wrap($body), 'utf-8');
        $mime = &MIME_Message::convertMimePart($body);
        $mime->setTransferEncoding('quoted-printable');
        $mime->transferEncodeContents();

        // Build the reply headers.
        $msg_headers = &new MIME_Headers();
        $msg_headers->addReceivedHeader();
        $msg_headers->addMessageIdHeader();
        $msg_headers->addHeader('Date', date('r'));
        $msg_headers->addHeader('From', $resource);
        $msg_headers->addHeader('To', $organiser);
        $msg_headers->addHeader('Subject', $subject);
        $msg_headers->addMIMEHeaders($mime);

        // Send the reply.
        static $mailer;

        if (!isset($mailer)) {
            require_once 'Mail.php';
            $mailer = &Mail::factory('SMTP', array('auth' => false));
        }

        $msg = $mime->toString();
        if (is_object($msg_headers)) {
            $headerArray = $mime->encode($msg_headers->toArray(), 'UTF-8');
        } else {
            $headerArray = $mime->encode($msg_headers, 'UTF-8');
        }

        /* Make sure the message has a trailing newline. */
        if (substr($msg, -1) != "\n") {
            $msg .= "\n";
        }

        $status = $mailer->send(MIME::encodeAddress($organiser), $headerArray, $msg);

        //$status = $mime->send($organiser, $msg_headers);
        if (is_a($status, 'PEAR_Error')) {
            return PEAR::raiseError(sprintf(_("Unable to send cancellation reply:  %s"),
                                            $status->getMessage()),
                                    OUT_LOG | EX_TEMPFAIL);
        } else {
            Horde::logMessage(_("Successfully sent cancellation reply"),
                              __FILE__, __LINE__, PEAR_LOG_INFO);
        }

        // Delete the messages from IMAP
        // Delete any old events that we updated
        if( !empty( $deleted_messages ) ) {
            Horde::logMessage(sprintf(_("Deleting %s because of cancel"),
                                      join(', ',$deleted_messages)),
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
            $imap->deleteMessages( $deleted_messages );
            $imap->expunge();
        }
        $imap->disconnect();
        unset($imap);
        // Get the resource's free/busy list
        // once more so it is up to date
        if( !triggerFreeBusy($resource,false) ) {
            Horde::logMessage(_("Error updating fblist"),
                              __FILE__, __LINE__, PEAR_LOG_WARNING);
        }
        return false;;

    default:
        // We either don't currently handle these iTip methods, or they do not
        // apply to what we're trying to accomplish here
        Horde::logMessage(sprintf(_("Ignoring %s method and passing message through to %s"),
                                  $method, $resource),
                          __FILE__, __LINE__, PEAR_LOG_INFO);
        return true;
    }

    // Pass the message through to the group's mailbox
    Horde::logMessage(sprintf(_("Passing through %s method to %s"),
                              $method, $resource),
                      __FILE__, __LINE__, PEAR_LOG_INFO);
    return true;
}
?>
