<?php
/**
 * Common code shared among IMP's various compose UI views.
 *
 * Copyright 2006-2013 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.horde.org/licenses/gpl.
 *
 * @author   Michael Slusarz <slusarz@horde.org>
 * @category Horde
 * @license  http://www.horde.org/licenses/gpl GPL
 * @package  IMP
 */
class IMP_Ui_Compose
{
    /**
     * Attach the auto-completer to the current compose form.
     *
     * @param array $fields  The list of DOM IDs to attach the autocompleter
     *                       to.
     */
    public function attachAutoCompleter($fields)
    {
        /* Attach autocompleters to the compose form elements. */
        foreach ($fields as $val) {
            $GLOBALS['injector']->getInstance('Horde_Core_Factory_Imple')->create('IMP_Ajax_Imple_ContactAutoCompleter', array('id' => $val));
        }
    }

    /**
     * Attach the spellchecker to the current compose form.
     */
    public function attachSpellChecker()
    {
        global $injector, $prefs, $registry;

        if ($registry->getView() == Horde_Registry::VIEW_BASIC) {
            $spell_img = '<span class="iconImg spellcheckImg"></span>';
            $br = '<br />';
        } else {
            $spell_img = $br = '';
        }
        $menu_view = $prefs->getValue('menu_view');

        $injector->getInstance('Horde_Core_Factory_Imple')->create('SpellChecker', array(
            'id' => 'spellcheck',
            'states' => array(
                'CheckSpelling' => $spell_img . (in_array($menu_view, array('both', 'text')) ? $br . _("Check Spelling") : ''),
                'Checking' => $spell_img . $br . _("Checking..."),
                'Error' => $spell_img . $br . _("Spell Check Failed"),
                'ResumeEdit' => $spell_img . $br . _("Resume Editing")
            ),
            'targetId' => 'composeMessage'
        ));
    }

    /**
     * Create the IMP_Contents objects needed to create a message.
     *
     * @param Horde_Variables $vars  The variables object.
     *
     * @return IMP_Contents  The IMP_Contents object.
     * @throws IMP_Exception
     */
    public function getContents($vars = null)
    {
        $ob = null;

        $indices = $this->getIndices($vars);

        if (!is_null($indices)) {
            try {
                $ob = $GLOBALS['injector']->getInstance('IMP_Factory_Contents')->create($indices);
            } catch (Horde_Exception $e) {}
        }

        if (is_null($ob)) {
            if (!is_null($vars)) {
                $vars->uid = null;
                $vars->type = 'new';
            }

            throw new IMP_Exception(_("Could not retrieve message data from the mail server."));
        }

        return $ob;
    }

    /**
     * Return the Indices object for the messages affected by this compose
     * action.
     *
     * @param Horde_Variables $vars  The variables object.
     *
     * @return IMP_Indices  The indices object.
     */
    public function getIndices($vars = null)
    {
        if (!is_null($vars) && isset($vars->msglist)) {
            return new IMP_Indices($vars->msglist);
        }

        return (is_null($vars) || !isset($vars->uids))
            ? IMP::mailbox(true)->getIndicesOb(IMP::uid())
            : new IMP_Indices_Form($vars->uids);
    }

    /**
     * Generate mailbox return URL.
     *
     * @param string $url  The URL to use instead of the default.
     *
     * @return string  The mailbox return URL.
     */
    public function mailboxReturnUrl($url = null)
    {
        if (!$url) {
            $url = Horde::url('mailbox.php');
        }

        $vars = $GLOBALS['injector']->getInstance('Horde_Variables');

        foreach (array('start', 'page', 'mailbox', 'thismailbox') as $key) {
            if (isset($vars->$key)) {
                $url->add($key, $vars->$key);
            }
        }

        return $url;
    }

    /**
     * Generate a compose message popup success window (compose.php).
     */
    public function popupSuccess()
    {
        global $page_output;

        $page_output->topbar = $page_output->sidebar = false;

        $page_output->addInlineScript(array(
            '$("close_success").observe("click", function() { window.close(); })'
        ), true);

        IMP::header(_("Message Successfully Sent"));

        $view = new Horde_View(array(
            'templatePath' => IMP_TEMPLATES . '/basic/compose'
        ));

        $view->close = Horde::widget(array(
            'id' => 'close_success',
            'url' => new Horde_Url(),
            'title' => _("Close this window")
        ));
        $view->new = Horde::widget(array(
            'url' => Horde::url('compose.php'),
            'title' => _("New Message")
        ));

        echo $view->render('success');

        IMP::status();
        $GLOBALS['page_output']->footer();
    }

    /**
     * Outputs the script necessary to generate the passphrase dialog box.
     *
     * @param string $type     Either 'pgp', 'pgp_symm', or 'smime'.
     * @param string $cacheid  Compose cache ID (only needed for 'pgp_symm').
     */
    public function passphraseDialog($type, $cacheid = null)
    {
        $params = array('onload' => true);

        switch ($type) {
        case 'pgp':
            $type = 'pgpPersonal';
            break;

        case 'pgp_symm':
            $params = array('symmetricid' => 'imp_compose_' . $cacheid);
            $type = 'pgpSymmetric';
            break;

        case 'smime':
            $type = 'smimePersonal';
            break;
        }

        $GLOBALS['injector']->getInstance('Horde_Core_Factory_Imple')->create('IMP_Ajax_Imple_PassphraseDialog', array(
            'onload' => true,
            'params' => $params,
            'type' => $type
        ));
    }

    /**
     */
    public function addIdentityJs()
    {
        $identities = array();
        $identity = $GLOBALS['injector']->getInstance('IMP_Identity');

        foreach (array_keys($identity->getAll('id')) as $ident) {
            $sm = $identity->getValue('sent_mail_folder', $ident);

            $identities[] = array(
                // Sent mail mailbox name
                'sm_name' => $sm ? $sm->form_to : '',
                // Save in sent mail mailbox by default?
                'sm_save' => (bool)$identity->saveSentmail($ident),
                // Sent mail display name
                'sm_display' => $sm ? $sm->display_html : '',
                // Bcc addresses to add
                'bcc' => strval($identity->getBccAddresses($ident))
            );
        }

        $GLOBALS['page_output']->addInlineJsVars(array(
            'ImpComposeBase.identities' => $identities
        ));
    }

    /**
     * Convert compose data to/from text/HTML.
     *
     * @param string $data  The message text.
     * @param string $to    Either 'text' or 'html'.
     *
     * @return string  The converted text.
     */
    public function convertComposeText($data, $to)
    {
        switch ($to) {
        case 'html':
            return IMP_Compose::text2html($data);

        case 'text':
            return $GLOBALS['injector']->getInstance('Horde_Core_Factory_TextFilter')->filter($data, 'Html2text', array(
                'wrap' => false
            ));
        }
    }

}
