#!/usr/bin/python

# Phatch - Photo Batch Processor
# Copyright (C) 2009 Nadia Alramli, www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.
#
# Follows PEP8

import glob
import os.path
import sys
import pprint
import logging
import subprocess
import time
import shutil
import optparse
import gettext
from collections import defaultdict
from PIL import Image

#FIXME: why Windows2000 fails here?
if sys.platform.startswith('win'):
    pformat = str
else:
    from pprint import pformat


DISABLE = set(['rename', 'blender', 'geotag'])
TEMPLATE ="Errors: %(errors)s\nMissing: %(missing)s\nCorrupted: %(corrupted)s\nMismatch: %(mismatch)s\nNew: %(new)s"


gettext.install('test')
def system_path(path):
    """Convert a path string into the correct form"""
    return os.path.abspath(os.path.normpath(path))

sys.path.insert(0, system_path('../phatch'))
from lib.safe import eval_safe
from phatch import init_config_paths


class PhatchTest(object):

    DEFAULT_INPUT = system_path('input')
    DEFAULT_OUTPUT = system_path('output/images')
    DEFAULT_LOG = system_path('output/logs.txt')
    DEFAULT_REPORT = system_path('output/report.txt')
    OUT_ACTIONS_PATH = system_path('output/actionlists')

    ACTIONS_PATH = system_path('../phatch/actions')
    PHATCH_PATH = system_path('../phatch/phatch.py')

    CONFIG_PATHS = init_config_paths()
    USER_LOG_PATH = CONFIG_PATHS['USER_LOG_PATH']
    PHATCH_ACTIONLISTS_PATH = CONFIG_PATHS['PHATCH_ACTIONLISTS_PATH']

    def __init__(self, input=DEFAULT_INPUT, output=DEFAULT_OUTPUT,
            verbose=False, log=DEFAULT_LOG, compare=None, clean_paths=True):
        """Build a dictionary of actions"""
        self.output = output
        self.input = input
        if not os.path.exists(self.input) or not os.listdir(self.input):
            raise Exception(
                    'WARNING: The input directory "%s" is empty or doesn\'t exist' % self.input)
        create(os.path.dirname(log))
        logging.basicConfig(
            level=logging.DEBUG,
            filename=log,
            filemode='w',
            format='%(asctime)s %(levelname)s:%(message)s'
        )
        if clean_paths:
            self.clean()

        self.compare = compare
        create(self.OUT_ACTIONS_PATH)
        create(self.output)
        self.actions = dict(
            (name, __import__('actions.%s' % name, name, fromlist=['actions']).Action())
            for name in self.get_action_names()
            if name != '__init__' and name not in DISABLE
        )
        self.by_tag = {}
        for name, action in self.actions.iteritems():
            for tag in action.tags:
                if tag in self.by_tag:
                    self.by_tag[tag][name] = action
                else:
                    self.by_tag[tag] = {name: action}
        self.file_actions = dict(
            (name, action)
            for name, action in self.actions.iteritems()
            if action.valid_last
        )
        self.library = dict(
            (file_name(actionlist_file), eval_safe(open(actionlist_file).read()))
            for actionlist_file in glob.glob(os.path.join(self.PHATCH_ACTIONLISTS_PATH, '*.phatch'))
        )
        self.verbose = verbose


    def get_action_names(self):
        return [file_name(action_file) for action_file in
            glob.glob(os.path.join(self.ACTIONS_PATH, '*.py'))]


    def clean(self):
        if os.path.exists(self.output):
            shutil.rmtree(self.output)
        if os.path.exists(self.OUT_ACTIONS_PATH):
            shutil.rmtree(self.OUT_ACTIONS_PATH)


    def build_test(self, name, actionlist):
        file_actions = [action for action in actionlist if action.valid_last]
        for index, file_action in enumerate(file_actions):
            file_action.set_field(
                'In',
                '%s' % shorten(self.output)
            )
            file_action.set_field(
                'File Name',
                self.get_save_filename(shorten(name_index(name, index)))
            )
        return dump(actionlist)


    def get_save_filename(self, file):
        return '<filename>_%s' % (file, )


    def get_output_filename(self, filename, file_action):
        try:
            ftype = file_action.get_field_string('As')
            if ftype == '<type>':
                ftype = '%(type)s'
        except KeyError:
            ftype = '%(type)s'
        return '%%(filename)s_%s.%s' % (filename, ftype)


    def generate_actionlists(self, save='save', actions=None):
        print 'Generating actionlists...'
        actionlists = {}
        if not actions:
            actions = self.actions
        for name, action in actions.iteritems():
            if name in self.file_actions:
                actionlist = [action]
                file_action = action
            else:
                file_action = self.actions[save]
                actionlist = [action, file_action]
            choices = possible_choices(action)
            for choice in choices:
                set_action_fields(action, choice)
                cname = choice_name(choice)
                if cname:
                    filename = '%s_%s' % (name, cname)
                else:
                    filename = name
                filename = shorten(filename)
                actionlist_data = self.build_test(filename, actionlist)
                path = os.path.join(self.OUT_ACTIONS_PATH, '%s.phatch' % filename)
                out_filename = self.get_output_filename(filename, file_action)
                write_file(path, actionlist_data)
                actionlists[filename] = {
                    'path': path,
                    'filename': out_filename
                }
        return actionlists


    def generate_preview_actionlists(self, name, choices):
        print 'Generating preview actionlists...'
        actionlists = {}
        action = self.actions[name]
        file_action = self.actions['save']
        actionlist = [action, file_action]
        for choice in choices:
            set_action_fields(action, choice)
            filename = shorten(self.get_preview_filename(name, choice))
            actionlist_data = self.build_test(filename, actionlist)
            path = os.path.join(self.OUT_ACTIONS_PATH, '%s.phatch' % filename)
            out_filename = self.get_output_filename(filename, file_action)
            write_file(path, actionlist_data)
            actionlists[filename] = {
                'path': path,
                'filename': out_filename
            }
        return actionlists


    def get_preview_filename(self, name, choice):
        cname = choice_name(choice)

        if cname:
            return '%s_%s' % (name, cname)

        return name


    def generate_library_actionlists(self):
        """Build library actionlists"""
        print 'Generating library actionlists...'
        actionlists = {}
        for name, actionlist in self.library.iteritems():
            file_actions = [
                action
                for action in actionlist['actions']
                if action['label'].lower() in self.file_actions
            ]
            for index, file_action in enumerate(file_actions):
                file_action['fields']['In'] = '%s' % self.output
                file_action['fields']['File Name'] = '<filename>_%s' % shorten(name_index(name, index))
            ftype = file_action['fields']['As']
            if ftype == '<type>':
                ftype = '%(type)s'
            path = os.path.join(self.OUT_ACTIONS_PATH, '%s.phatch' % name)
            out_filename = '%%(filename)s_%s.%s' % (name, ftype)
            write_file(path, actionlist)
            actionlists[name] = {
                'path': path,
                'filename': out_filename
            }
        return actionlists


    def generate_custom_actionlists(self, action1, action2):
        print 'Generating save actionlists...'
        actionlists = {}
        for choice1 in possible_choices(action1):
            set_action_fields(action1, choice1)
            cname1 = choice_name(choice1)
            for choice2 in possible_choices(action2):
                set_action_fields(action2, choice2)
                cname2 = choice_name(choice2)
                filename = shorten('_'.join([
                    action2.label.replace(' ', '-'), cname2,
                    action1.label.replace(' ', '-'), cname1,
                ]))
                actionlist_data = self.build_test(filename, [action1, action2])
                path = os.path.join(self.OUT_ACTIONS_PATH, '%s.phatch' % filename)
                out_filename = self.get_output_filename(filename, action2)
                write_file(path, actionlist_data)
                actionlists[filename] = {
                    'path': path,
                    'filename': out_filename
                }
        return actionlists


    def execute_actionlists(self, actionlists=None):
        start = time.time()
        result = {
            'errors': [],
            'mismatch': [],
            'new': [],
            'corrupted': [],
            'missing': []
        }
        try:
            if not actionlists:
                actionlists = phatch.generate_library_actionlists()
                actionlists.update(phatch.generate_actionlists())
            total = len(actionlists)
            for i, name in enumerate(sorted(actionlists)):
                if self.verbose:
                    print name.title().center(50, '-')
                else:
                    sys.stdout.write(
                        '\rRunning %s/%s %s' % (
                            i + 1,
                            total,
                            name[:50].ljust(50)
                        )
                    )
                    sys.stdout.flush()
                if not execute(self, actionlists[name]['path']):
                    result['errors'].append(name)
                if self.verbose:
                    print

            print '\nChecking all output images exist...'
            if not os.path.exists(self.input):
                images = []
            elif os.path.isfile(self.input):
                images = [tuple(os.path.basename(self.input).rsplit('.', 1))]
            else:
                images = [tuple(image.rsplit('.', 1)) for image in os.listdir(self.input)]
            filenames = []
            for name, actionlist in actionlists.iteritems():
                for image_name, image_type in images:
                    filename = actionlist['filename'] % {
                        'filename': image_name,
                        'type': image_type
                    }
                    path = os.path.join(self.output, filename)
                    if not os.path.exists(path):
                        result['missing'].append(filename)
                    else:
                        filenames.append(filename)
            total = len(filenames)
            if self.compare:
                print '\nComparing results...'
            for i, filename in enumerate(filenames):
                sys.stdout.write(
                    '\rComparing %s/%s %s' % (
                        i + 1,
                        total,
                        filename[:50].ljust(50)
                    )
                )
                sys.stdout.flush()
                try:
                    im1 = Image.open(os.path.join(self.output, filename))
                except IOError:
                    result['corrupted'].append(filename)
                    continue
                if not self.compare:
                    continue
                try:
                    im2 = Image.open(os.path.join(self.compare, filename))
                except IOError:
                    result['new'].append(filename)
                    continue

                if not compare(im1, im2):
                    result['mismatch'].append(filename)
        except KeyboardInterrupt:
            print 'Stopped'
        banner('Stats')
        final_result = {}
        for key, value in result.iteritems():
            if value and isinstance(value, list):
                value = '\n\t'.join(value)
            else:
                value = 'None'
            final_result[key] = value
        output = TEMPLATE % final_result
        output += '\n%s' % ', '.join('%s: %s' % (key, len(value)) for key, value in result.iteritems())
        f = open(self.DEFAULT_REPORT, 'w')
        f.write(output)
        f.close()
        print output
        print 'The report was saved in: %s' % self.DEFAULT_REPORT
        print 'Execution took %.2f seconds' % (time.time() - start)


def execute(obj, actionlist):
    """Execute the actionlist in phatch"""
    if obj.verbose:
        stdout = None
    else:
        stdout = subprocess.PIPE
    output, error = subprocess.Popen(
        ['python', obj.PHATCH_PATH, '-cv', actionlist, obj.input],
        stdout=stdout, stderr=subprocess.PIPE
    ).communicate()

    logs_file = open(obj.USER_LOG_PATH)
    logs = logs_file.read()
    logs_file.close()
    if logs:
        logging.error(
            'Executing actionlist: %s generated the following logs:\n%s\n'
            % (actionlist, logs)
        )
    if error:
        logging.error(
            'Executing actionlist: %s generated the following error:\n%s\n'
            % (actionlist, error)
        )
    return not(error or logs)


def shorten(x):
    return x.replace('True', '1').replace('False', '0')\
        .replace('Background', 'Bg').replace('Rotation', 'Rot')\
        .replace('Options', 'Op').replace('Position', 'Pos')\
        .replace('Orientation', 'Or').replace('Vertical', 'Ver')\
        .replace('Horizontal', 'Hor').replace('Transformation', 'Tr')\
        .replace('Advanced', 'Adv').replace('Thumbnail', 'Thumb')\
        .replace('Automatic', 'Auto').replace(' (use exif orientation)', '')\
        .replace('Rotate', 'Rot').replace('Update', 'Upd')\
        .replace('-Corner', '').replace('Transparent', 'Tra')\
        .replace('<', '(').replace('>', ')')


def possible_choices(action):
    choices = []
    possible_choices_helper(action, choices)
    return choices


def possible_choices_helper(action, choices):
    """Generate all possible action choices"""
    if hasattr(action, 'get_relevant_field_labels'):
        relevant = action.get_relevant_field_labels()
    else:
        relevant = action._fields.keys()
    choice_fields = dict(
        (name, field) for name, field in action._fields.iteritems()
        if isinstance(field, (action.BooleanField, action.ChoiceField))
        and not isinstance(field, (
            action.ImageFilterField,
            action.ImageResampleField,
            action.ImageResampleAutoField)
        )
        and name != '__enabled__'
        and name in relevant
    )
    choice = dict(
        (fname, field.get())
        for fname, field in choice_fields.iteritems()
    )
    if choice in choices:
        return
    choices.append(choice)
    for fname, field in choice_fields.iteritems():
        default = field.get()
        if isinstance(field, action.BooleanField):
            options = [True, False]
        else:
            options = field.choices
        for option in options:
            if option == default:
                continue
            action.set_field(fname, option)
            possible_choices_helper(action, choices)


def build_choice_map(fields, choices):
    choice = dict(
        (fname, value[0])
        for fname, value in fields.iteritems()
    )
    if choice in choices:
        return
    choices.append(choice)
    for fname, value in fields.iteritems():
        default = value[0]
        for option in value:
            if option == default:
                continue
            fields[fname] = value[1:] + [value[0]]
            build_choice_map(fields, choices)


def set_action_fields(action, fields):
    for field_name, field_value in fields.iteritems():
        action.set_field(field_name, field_value)


def dump(actions):
    data = [action.dump() for action in actions]
    return {'description': '', 'actions': data, 'version': '0.2.0.test'}


def write_file(path, data):
    """Write a file to disk"""
    f = open(
        path,
        'w'
    )
    f.write(pformat(data).replace('\n', os.linesep))
    f.close()


def create(path):
    """Create a path if it doesn't already exist"""
    if not os.path.exists(path):
        os.makedirs(path)

def file_name(path):
    """Get the file name from path without its extension"""
    return os.path.basename(path).partition('.')[0]

def name_index(name, index):
    if index:
        return '%s_%s' % (name, index)
    return name

def choice_name(choice):
    result = '_'.join(
        '%s=%s' % (
            field_name.strip().replace(' ', '-'),
            ('%s' % field_value).strip().replace(' ', '-')
        )
        for field_name, field_value in choice.iteritems()
    )
    return result

def compare(im1, im2):
    return list(im1.getdata()) == list(im2.getdata())

def banner(title, width=50):
    print
    print '*' * 50
    print '*%s*' % title.center(48, ' ')
    print '*' * 50


if __name__ == '__main__':
    # Option parser
    parser = optparse.OptionParser()
    parser.add_option(
        '-i', '--input',
        default=PhatchTest.DEFAULT_INPUT,
        help='Image input folder [default: %default]'
    )
    parser.add_option(
        '-o', '--output',
        default=PhatchTest.DEFAULT_OUTPUT,
        help='Image output folder [default: %default]'
    )
    parser.add_option(
        '-v', '--verbose',
        action='store_true',
        default=False,
        help='Verbose output'
    )
    parser.add_option(
        '-a', '--actionlists',
        action='store_true',
        default=False,
        help='Generate actionlists only'
    )
    parser.add_option(
        '-l', '--log',
        default=PhatchTest.DEFAULT_LOG,
        help='Log file path [default: %default]'
    )
    parser.add_option(
        '-t', '--tag',
        default=None,
        help='Test by tag'
    )
    parser.add_option(
        '-c', '--custom',
        type="string",
        nargs=2,
        default=None,
        help='Test custom action'
    )
    parser.add_option(
        '-s', '--select',
        type="string",
        default=None,
        help='select an action'
    )
    parser.add_option(
        '--compare',
        default=None,
        help='Compare results'
    )
    parser.add_option(
        '-p', '--preview',
        default=None,
        help='Generate Preview for action'
    )
    parser.add_option(
        '--choices',
        default=None,
        help='Custom choices'
    )
    options, args = parser.parse_args()
    phatch = PhatchTest(
        input=system_path(options.input),
        output=system_path(options.output),
        verbose=options.verbose,
        log=options.log,
        compare=options.compare
    )
    if options.preview:
        f = open(options.choices, 'r')
        choices = []
        build_choice_map(eval_safe(f.read()), choices)
        actionlists = phatch.generate_preview_actionlists(options.preview, choices)
        f.close()
    elif options.custom:
        actionlists = phatch.generate_custom_actionlists(
            action1=phatch.actions[options.custom[0]],
            action2=phatch.actions[options.custom[1]],
        )
    elif options.select:
        actionlists = phatch.generate_actionlists(
            actions={options.select: phatch.actions[options.select]}
        )

    elif options.tag == 'library':
        actionlists = phatch.generate_library_actionlists()
    elif options.tag == 'save':
        actionlists = phatch.generate_custom_actionlists(
            action1=phatch.actions['convert_mode'],
            action2=phatch.actions['save'],
        )
    elif options.tag:
        try:
            actions = phatch.by_tag[options.tag]
        except KeyError:
            raise Exception(
                'Tag %s is not available. Select one of:\n%s' % (
                    options.tag,
                    ', '.join(sorted(phatch.by_tag.keys() + ['library', 'save']))
                )
            )
        print 'Testing the following actions:'
        print ', '.join(actions.keys())
        actionlists = phatch.generate_actionlists(actions=actions)
    else:
        actionlists = phatch.generate_library_actionlists()
        actionlists.update(phatch.generate_actionlists())
        actionlists.update(phatch.generate_custom_actionlists(
            action1=phatch.actions['convert_mode'],
            action2=phatch.actions['save'],
        ))
    if not options.actionlists:
        phatch.execute_actionlists(actionlists)
