# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

# Embedded icon is taken from www.openclipart.org (public domain)

# Follows PEP8

from core import models
from lib.reverse_translation import _t
from lib.imtools import has_alpha, has_transparency

#---Pil

def init():
    global Image, ImageColor, ImageFilter
    import Image
    import ImageColor
    import ImageFilter
    global HTMLColorToRGBA
    from lib.colors import HTMLColorToRGBA

REFLECT_ID = 'reflect_w%s_h%s_o%s'


def gradient_vector(size, opacity, cache):
    id = REFLECT_ID%(1, size, opacity)
    try:
        return cache[id]
    except KeyError:
        pass
    opacity = float(opacity)
    grad = Image.new('L', (1, size))
    data = [int(opacity*x/size) for x in range(size, 0, -1)]
    grad.putdata(data)
    cache[id] = grad
    return grad


def gradient_mask(size, opacity, cache):
    id = REFLECT_ID%(size[0], size[1], opacity)
    try:
        return cache[id]
    except KeyError:
        pass
    #gradient vector
    vector = gradient_vector(size[1], opacity, cache)
    #scale vector
    grad = cache[id] = vector.resize(size, Image.LINEAR)
    return grad


def reflect(image, depth, opacity, background_color, background_opacity,
        scale_method, gap=0, scale_reflection=False,
        blur_reflection=False, cache=None):
    if has_transparency(image):
        image = image.convert('RGBA')
    else:
        image = image.convert('RGB')
    if cache is None:
        cache = {}
    opacity = (255 * opacity) / 100
    background_opacity = (255 * background_opacity) / 100
    scale_method = getattr(Image, scale_method)
    if background_opacity == 255:
        mode = 'RGB'
        color = background_color
    else:
        mode = 'RGBA'
        color = HTMLColorToRGBA(background_color, background_opacity)
    width, height = image.size
    depth = min(height, depth)
    #make reflection
    if has_alpha(image) and background_opacity>0:
        reflection = Image.new(mode, image.size, color)
        reflection.paste(image, (0, 0), image)
    else:
        reflection = image
    reflection = reflection.transpose(Image.FLIP_TOP_BOTTOM)
    if scale_reflection:
        reflection = reflection.resize((width, depth), scale_method)
    else:
        reflection = reflection.crop((0, 0, width, depth))
    if blur_reflection:
        reflection = reflection.filter(ImageFilter.BLUR)
    mask = gradient_mask((width, depth), opacity, cache)
    #composite
    total_size = (width, height+gap+depth)
    total = Image.new(mode, total_size, color)
    if has_alpha(image):
        total.paste(image, (0, 0), image)
    else:
        total.paste(image, (0, 0))
    total.paste(reflection, (0, height+gap), mask)
    return total

#---Phatch

class Action(models.Action):
    """Drops a reflection"""

    label = _t('Reflection')
    author = 'Stani'
    email = 'spe.stani.be@gmail.com'
    cache = True
    init = staticmethod(init)
    pil = staticmethod(reflect)
    version = '0.1'
    tags = [_t('filter')]
    __doc__ = _t('Drops a reflection')

    def interface(self, fields):
        fields[_t('Depth')] = self.PixelField('50%',
            choices=self.PIXELS[:-1])
        fields[_t('Gap')] = self.PixelField('0',
            choices=['0', '1', '2', '5'])
        fields[_t('Opacity')] = self.SliderField(60, 0, 100)
        fields[_t('Blur Reflection')] = self.BooleanField(False)
        fields[_t('Scale Reflection')] = self.BooleanField(False)
        fields[_t('Scale Method')] = self.ImageResampleField('antialias')
        fields[_t('Background Color')] = self.ColorField('#FFFFFF')
        fields[_t('Background Opacity')]= self.SliderField(100, 0, 100)

    def get_relevant_field_labels(self):
        """If this method is present, Phatch will only show relevant
        fields.

        :returns: list of the field labels which are relevant
        :rtype: list of strings

        .. note::

            It is very important that the list of labels has EXACTLY
            the same order as defined in the interface method.
        """
        relevant = ['Depth', 'Gap', 'Opacity', 'Blur Reflection',
            'Scale Reflection']
        if self.get_field_string('Scale Reflection') in ('yes', 'true'):
            relevant.append('Scale Method')
        relevant.extend(['Background Color', 'Background Opacity'])
        return relevant

    def values(self, info):
        #pixel fields
        y = info['height']
        return super(Action, self).values(info,
            pixel_fields={'Depth': y, 'Gap': y})

    icon = \
'x\xda\x01\xcb\x074\xf8\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\x00\
\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\x08\
\x08|\x08d\x88\x00\x00\x07\x82IDATh\x81\xed\x9a]\x8c]U\x15\xc7\x7f\xeb\x9c}\
\xee=g\xeeGg:3\xedtl+\x85\xa1\x80SH-6U(\x96X\x05\x11\xf9\x14M\x08\x92\xe8\
\x83\x0f\x04\x1fL\xf4Ay\x81\xc4\x07\xc5Dc\x13\xa8T|(\xc6\xf0a\x8dX\xa5T\x1a\
\x04$T\x0cZ\xf9j;\xd3\xc20\xb4\x96\xa13\xcc\xd7\xbds\xef\xdc\xf3\xb5\xf7\xf6\
a:\xb5\xd6;\xd3)v\xda\xfb\xe0Jvrr\xb2\xfek\xad\xdf\xdek\x9f\xb3\xef\xc9\x85\
\xff\xdb\xfc\xd9\xc3\x8f\xee\xb8\xfd\xf9\xbf\xee\xdf\xbf{O\xef\x0b\x9b7on\
\xa9\xe7\xa3\xcevQs\xb5\x1f?\xfc\xe4]\xdd\xdd\xab\x1f\xcc5e\xc4\x18{\xc9\xf2\
\x95kn\x076\x9f\xec\xd7\x90\x00w\xde\xf9\x9d\xdc\xf2\x15\x17\xdek@\x12mH\x12\
\x8d\xa3\xd4\xaaz\xbe\r\tp\xc9\x15\x1b\xee\x0e\x82\xa6\xc5Q\xa2\x11\x810NI\
\xb4\x14\xea\xf96\x1c@w\xf7\x973\x0bZ\xdb\xef\x8a\xe2\x14G\x04\x01J\xe5*q5\
\x1c\xab\xe7\xdfp\x00\x9f\xbc\xe1\xda\x9b\xfc \x7f^\x94\x18\x1cI\x01\x18+U\
\xd0a\xf8^=\xff\x86\x03X\xd8\xd1y{\x9ch\x1c\x11\x1c\x01m-\xe5\x89\n\xb5ji_=\
\xff\x86\x02\xe8ZwG\xd1\xcb\x04\x9f\x9b\xea}A\x04L\x98R\xa9\x86\xe6\xd5\xdd\
\x7f\xdcSO\xd3P\x00\x9d\x17\x7f\xe2\x06\xe5e\xf3Q<\x05\xe0\x08LT#\xc6\xc6\
\xc7{~\xfb\xab\x9f\xbf_O\xd3P\x00\x85\xe6\xd6\xcf\xa6\xa9\xe1\xc4\x15\x188:H\
\x98\x98?\xcf\xa4i(\x00\xc7\xcd\xac\rc\r"\x88\x08\x896\x0c\x8f\x96\xb0\xb5\
\x89\x9d3i\x1a\t\xc0u\xc4\xed\x8a\x12\x8d\xe3\x08\x8e\x08\xe5jH\xb94>x\xf0\
\xf9\x17v\xcd$j\x18\x80b\xf7\xb5+p\xb3\xd9\xe9\x97\x97\x88p\xf8\x9f\x03\x84\
\xb1\xd9\xbeo\xdf\xb6x&]\xc3\x00$nWk\xaa\xa7\xfa\xdf\x11\xa8\xd4\x12\x86G\
\xc6\xb0\xe1\xe8\xe3\xb3\xe9\x1a\x06 \x96l{\x9cj\xa28E\x80\xb7\x0f\x1f%\xaaM\
\xf6\xef\xd9v\xef\x0b\xb3\xe9\x1a\x06@\x8bt\xc6\x89&J\x0c\xa5\xca$G\x07\x06\
\x08\xb2\xee\x16\xc0\xce\xa6k\x18\x00\x9c\xfc%\xdaX\xc2(e\xdf\xc1w\xc18c\xa6\
\xbfg\xcb\xa9d\x8d\x03\xe0z\xeb\x05\xa1\xb7\x7f\x90\x91\xe1!\x9a\xfc\xa6-#\
\xafm\x1d?\x95\xac1\x00.\xbe\xfbS\xa8\xa6\xcb\x87\xc7\'9\xfc\xce\x01\xc4oI\
\x19\xff\xe0\xc1\xb9H\xcf=\xc0yW\xfb\x14:\xef\x07\x91\xde\xb7\xde\xc1\xea\
\x10k\xdc\xa8\xf6\xc6sCs\x91;\xf3]\xdf\xac\xd6u\xc7\xc7h\xdb\xf8$\xaa\xe9*\
\x00\x9b\x86S\xf7\xc5\xe6\xb8\xf43\x9f\x9eK\x08\x99\xbf\xeaf\xb0\xae\xeb\x8a\
\x14V\x7f\x0b\xb7i#*\xb8\x02q\xff\xdd\x05:\x84\xda\x10`!X\xbc\x8f\xc9\xde\
\x9b\xd8\xfbH\xdfl\xe1\xce\n\x80\xb5\xd6\x07\xd4\x17\xef\xf8\xde\xa2\xde\x11\
og)\xf2W\xc6\xda0\x99(R\xc9\xd6-#\x1fx\\\xb1\xaa}\xe2\xf5\xd7zn\x1e|\xf1\xbe\
\xe7f\x8a}\xb6\xf6@3`/\xebj\xfe\xfd\x8d\x1bZVV*\x15\xca\xe52\xe5\xf2(c\xe5\
\x1an\xb0\x80Z\xec0\x998L\xc4\x0e\x93\xb1K9\xf6\xa8\x95maa\xc1\xfb\xfa \x9c;\
\x00km\x00\xa8M\x9b6\xddz\xc1\xf2\xb6\xee0\x0cI\xd3\x948\x8e\x11\x11\x96u\
\xb6#2\xb5\x02r\xec\x14*",\\X$\n#v\xef\xe9k\xe9\x99%\xfe\xd9\xd8\xc4\x85\xfe\
\xfe~\xd7\xf3\xbco\x1acH\x92\x84(\x8c0\xc6`\x8c\xa9[\xbc\x88P\xc8\x17\xa8T+\
\x14\xb3\xfa\xc5\xd9\x82\xcf+\x80\xb5V\x01\xfe\xce\x9d;o\xf6}\xffB\xad5I\x92\
\x10\'1Zk\n\x85\x02\xd6\xda\xe3EO\x9bR\n?\xf0)\x97\xca\xf8\xbe\xbf\xf7\x9c\
\x01\x00\xc50\x0c\x11\x91oXk\x89\xe3\x988\x8e1\xc6`\xed\xd4\x11\xe7\xc4\xc2\
\xa7\xaf\x83  \x89\x13\xa28J\x8f\x1c9\xf2\xf2l\t\xe6\r\xc0Z+@~\xeb\xd6\xad\
\xd7\xfa\xbe\xdfm\x8c!MS\x92$\xf9\x8f\xf6\xa97\xdaZ\xdb\x18\x19\x19!\x8e\xe3\
\xb7v\xec\xd8Q\xf7{\xd0\xbc\x03\x00\xf9c\xf1o\x01\x88\xa2\xe8\xf8\xecO\xb7\
\xcf\x89\xfd?m\xae\xebR,\x16)\x97\xcbXk_=U\x92\xf9\x04X\xf0\xec\xb3\xcfv(\
\xa56No\xde4I\xd1Zc\x8c9\xeet\xf2&\xcex\x19\x94R\x94\xcbe\xe28~\xe5TI\xe6\
\x05\xe0\xd8\x8b+\xdb\xd7\xd7w\xb3R*\x13\xc71i\x9a\x92\xea\xf4x\xff\xcf\xd4>\
\xcd-\xcdDQDu\xb2J\xa9T\x9a\xf1\xf9?\xaf\x00@s\x14E\x16cn\xab\xd7\xfb\'\xb7\
\xcf\x89\xd7K:\x96084H\x1c\xc5\x1fl\xdf\xbe\xfd\xcds\x05`~\xf6\xc0\xa6\xcbR\
\x9d.\x9b\x9e}\xa3\xcd\xf1\xf6\xa9\xd7\xfb"\x82r\x15\xb9|\x8e\xd1\x91Q\xb4\
\xd1\xbdsI4/ob\x11\x19z\xf9\x977:\xab/h\xd6G\x8e\x84\xc9X\xc5q?\x10\x18\x11\
\xc2\tKdt\x90\x84\xa9J#\x9dQZ<7\x11\xe5\xe1f\x82\xe2\xe2\x0eO\xa7\x9a\xf2D\
\x19\x1bVN\xb9\x81\xe7\r\x00\xa0\xbd\xc5_\x97\r\xd4\xfe\xe5\xcb\x9a2KR\xad\
\x1ckU\x9a\x8a\x07V9n\xeaZ\xab=\xad\'\x95\xb5\xb8\x8e\x88+\xa0\x87\'\xfa\x19\
y\xffe\xb3\xb6-\xa9<\xd37\xf4\xc49\x03\xf8\xfeE\x0bV\xa4\x07FV\xca\xd2B\xcf\
\xc4p5o#\xed\xb5\x9e\xbf0\x16\xacBP\x80\xb2\x06\x85x\n\xac+"\xcaZQ\x85\x85V\
\xad\xb0\xa8\xae\xe5\x89\xfe\xc1\xe3/\xfem.\xb9\xce\xf8q\xfa7p\xd9E\xcay\xba\
\xbc\xa6\xa3\xbf\xf8~e\x89\xff\xde\xc4\x8arW\xcb\xde5\x8f\xdd\xfa\x14\x8eUX\
\x14X\x85\x15U*\x85~m\xb8\x96\xeb\xe8j\t\xb1\xa2\x10\xab\xde\xfd\xd3\xa1\x8e\
\x81W\xdes\xae\xdc\xd63\xa7\x1f4gt\x05\x1e\x81\x8f_\x0cO\xe7R\xd3\x91\x7fe\
\xe0#\xd3\xf7\xa5o\xbc\xf5\xcd\x1f\xfe\xa5\xc8Pe\xb1\x8c\xd4\x16\x9b\xd1\xda\
"\xcaI[v2nQ\xd6:o,-\xbe\xa3\xd6/\x7f\xc3\xee\x1d\xba\xc0\xdf;t\xa9\xf9\xe8\
\x82\xdf\xcd5\xe7\x19[\x81\x9f\xc2\xda\x0b\xe1\xa9NX\xd4\xfc!c\x88\x97!\xb3~\
=/=\xff\xdcU_\x81\x97\xe6\xa29#\x8f\xd1\x1f\xc1\x86\x00\x9eN`Q\x05\x08?T\x14\
!\xf8\xc2\xf5\x8c\r\x0c\xec\x9ek\xf1\xf0\xbf\x03\xc8\x03pO\x08\xbbj\xd0V\x01\
\xaa\xc7\xc6i\x86!\xf8\xfcu`\x0c=\x07z\xef?\x1d\xe5\x87\xdd\x03\xb2\t\xbe:\
\x08\xf7\xfd\x1d\xceofj&\xfc\x93Fn.\x81r9r\xb7|\tw\xd1"\xde\xde\xf2\xd03\xb7\
\xc1\x1fN\xa7\x909\x01\\\x03\xb9<\xac\xf6`c\x04\x1b\xaap\xf9/`A3\xd0\nx@pl\
\x9c\x0c\xe1\xceT\xb8\xf2\xc8\xae[G\xf6\xca\xf5H\x100\xfa\xd8\xa3\x13\xfb\
\xab\x95\xbbO\xa7\xf8\x19\x01~\xdd\xb9\xe2\xa25\xd7_\xb3)ni\xbe\xea\xe0\xd0\
\xa0\xdb\xff\xee\xa1\xcc\xe0\xa1CR\nC\xc6\xe3\x08\x92\x94\xf3\x8d&\xa3\r\xcd\
\xd6\x10h\x83k-XC\xac5)\x10\x015\xa6\xce\xd4\x92\xc9 ~\x80\x04>n[;\xee\xd2ed\
V\xadB\xf2\x050\x9a\xda\xae]\xec?x\xe0\x9e\xaf\xc1\xac\x9fP\xeaND\xbd\x9b;[Z\
\xbe\xdb\xeey\x97\xda\xc9Z\xce\xf5\xbc\xac\xb8N\x01\x91&7\xebg\x1c\x91\xa2\
\xe3\xfb\x81\xb8n\xc6u\x9cB.\x9b\x15q]$\xebO\x89\xb3\x19\xc4q\xc0q\x91Lf*I6\
\x0b\x8e\x03\xae\x8bx\x1e\x006\x0c\xd1G\x8fR{\xfd5\x0e\x97K?\xb9\x1a\xbe}\
\xba\xc5\xcf\x08p\x1a\xe6\xac\x83\xf6\xb5\xd0\xb4\x14\xf2M\xc7V\xd4\x87 \x07\
Y\x00\x07\x1c\x1f\xfe\xebo\x02\x1aj\xa30\xf1\x0fx\xfb!\x98\xd3g\xc4z\xf6/\
\xe9\xd5S\xc2>\xf53R\x00\x00\x00\x00IEND\xaeB`\x82\xa8[\xdd\xd0'
