<?php

abstract class DifferentialChangesetHTMLRenderer
  extends DifferentialChangesetRenderer {

  protected function renderChangeTypeHeader($force) {
    $changeset = $this->getChangeset();

    $change = $changeset->getChangeType();
    $file = $changeset->getFileType();

    $messages = array();
    switch ($change) {

      case DifferentialChangeType::TYPE_ADD:
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht('This file was added.');
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This image was added.');
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This directory was added.');
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This binary file was added.');
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This symlink was added.');
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This submodule was added.');
            break;
        }
        break;

      case DifferentialChangeType::TYPE_DELETE:
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht('This file was deleted.');
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This image was deleted.');
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This directory was deleted.');
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This binary file was deleted.');
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This symlink was deleted.');
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This submodule was deleted.');
            break;
        }
        break;

      case DifferentialChangeType::TYPE_MOVE_HERE:
        $from = phutil_tag('strong', array(), $changeset->getOldFile());
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht('This file was moved from %s.', $from);
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This image was moved from %s.', $from);
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This directory was moved from %s.', $from);
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This binary file was moved from %s.', $from);
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This symlink was moved from %s.', $from);
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This submodule was moved from %s.', $from);
            break;
        }
        break;

      case DifferentialChangeType::TYPE_COPY_HERE:
        $from = phutil_tag('strong', array(), $changeset->getOldFile());
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht('This file was copied from %s.', $from);
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This image was copied from %s.', $from);
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This directory was copied from %s.', $from);
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This binary file was copied from %s.', $from);
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This symlink was copied from %s.', $from);
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This submodule was copied from %s.', $from);
            break;
        }
        break;

      case DifferentialChangeType::TYPE_MOVE_AWAY:
        $paths = phutil_tag(
          'strong',
          array(),
          implode(', ', $changeset->getAwayPaths()));
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht('This file was moved to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This image was moved to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This directory was moved to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This binary file was moved to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This symlink was moved to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This submodule was moved to %s.', $paths);
            break;
        }
        break;

      case DifferentialChangeType::TYPE_COPY_AWAY:
        $paths = phutil_tag(
          'strong',
          array(),
          implode(', ', $changeset->getAwayPaths()));
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht('This file was copied to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This image was copied to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This directory was copied to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This binary file was copied to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This symlink was copied to %s.', $paths);
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This submodule was copied to %s.', $paths);
            break;
        }
        break;

      case DifferentialChangeType::TYPE_MULTICOPY:
        $paths = phutil_tag(
          'strong',
          array(),
          implode(', ', $changeset->getAwayPaths()));
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            $messages[] = pht(
              'This file was deleted after being copied to %s.',
              $paths);
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht(
              'This image was deleted after being copied to %s.',
              $paths);
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht(
              'This directory was deleted after being copied to %s.',
              $paths);
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht(
              'This binary file was deleted after being copied to %s.',
              $paths);
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht(
              'This symlink was deleted after being copied to %s.',
              $paths);
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht(
              'This submodule was deleted after being copied to %s.',
              $paths);
            break;
        }
        break;

      default:
        switch ($file) {
          case DifferentialChangeType::FILE_TEXT:
            // This is the default case, so we only render this header if
            // forced to since it's not very useful.
            if ($force) {
              $messages[] = pht('This file was not modified.');
            }
            break;
          case DifferentialChangeType::FILE_IMAGE:
            $messages[] = pht('This is an image.');
            break;
          case DifferentialChangeType::FILE_DIRECTORY:
            $messages[] = pht('This is a directory.');
            break;
          case DifferentialChangeType::FILE_BINARY:
            $messages[] = pht('This is a binary file.');
            break;
          case DifferentialChangeType::FILE_SYMLINK:
            $messages[] = pht('This is a symlink.');
            break;
          case DifferentialChangeType::FILE_SUBMODULE:
            $messages[] = pht('This is a submodule.');
            break;
        }
        break;
    }

    // If this is a text file with at least one hunk, we may have converted
    // the text encoding. In this case, show a note.
    $show_encoding = ($file == DifferentialChangeType::FILE_TEXT) &&
                     ($changeset->getHunks());

    if ($show_encoding) {
      $encoding = $this->getOriginalCharacterEncoding();
      if ($encoding != 'utf8') {
        if ($encoding) {
          $messages[] = pht(
            'This file was converted from %s for display.',
            phutil_tag('strong', array(), $encoding));
        } else {
          $messages[] = pht(
            'This file uses an unknown character encoding.');
        }
      }
    }

    if (!$messages) {
      return null;
    }

    foreach ($messages as $key => $message) {
      $messages[$key] = phutil_tag('li', array(), $message);
    }

    return phutil_tag(
      'ul',
      array(
        'class' => 'differential-meta-notice',
      ),
      $messages);
  }

  protected function renderPropertyChangeHeader() {
    $changeset = $this->getChangeset();
    list($old, $new) = $this->getChangesetProperties($changeset);

    // If we don't have any property changes, don't render this table.
    if ($old === $new) {
      return null;
    }

    $keys = array_keys($old + $new);
    sort($keys);

    $key_map = array(
      'unix:filemode' => pht('File Mode'),
      'file:dimensions' => pht('Image Dimensions'),
      'file:mimetype' => pht('MIME Type'),
      'file:size' => pht('File Size'),
    );

    $rows = array();
    foreach ($keys as $key) {
      $oval = idx($old, $key);
      $nval = idx($new, $key);
      if ($oval !== $nval) {
        if ($oval === null) {
          $oval = phutil_tag('em', array(), 'null');
        } else {
          $oval = phutil_escape_html_newlines($oval);
        }

        if ($nval === null) {
          $nval = phutil_tag('em', array(), 'null');
        } else {
          $nval = phutil_escape_html_newlines($nval);
        }

        $readable_key = idx($key_map, $key, $key);

        $row = array(
          $readable_key,
          $oval,
          $nval,
        );
        $rows[] = $row;

      }
    }

    $classes = array('', 'oval', 'nval');
    $headers = array(
      pht('Property'),
      pht('Old Value'),
      pht('New Value'),
    );
    $table = id(new AphrontTableView($rows))
      ->setHeaders($headers)
      ->setColumnClasses($classes);
    return phutil_tag(
      'div',
      array(
        'class' => 'differential-property-table',
      ),
      $table);
  }

  public function renderShield($message, $force = 'default') {
    $end = count($this->getOldLines());
    $reference = $this->getRenderingReference();

    if ($force !== 'text' &&
        $force !== 'whitespace' &&
        $force !== 'none' &&
        $force !== 'default') {
      throw new Exception("Invalid 'force' parameter '{$force}'!");
    }

    $range = "0-{$end}";
    if ($force == 'text') {
      // If we're forcing text, force the whole file to be rendered.
      $range = "{$range}/0-{$end}";
    }

    $meta = array(
      'ref'   => $reference,
      'range' => $range,
    );

    if ($force == 'whitespace') {
      $meta['whitespace'] = DifferentialChangesetParser::WHITESPACE_SHOW_ALL;
    }

    $content = array();
    $content[] = $message;
    if ($force !== 'none') {
      $content[] = ' ';
      $content[] = javelin_tag(
        'a',
        array(
          'mustcapture' => true,
          'sigil'       => 'show-more',
          'class'       => 'complete',
          'href'        => '#',
          'meta'        => $meta,
        ),
        pht('Show File Contents'));
    }

    return $this->wrapChangeInTable(
      javelin_tag(
        'tr',
        array(
          'sigil' => 'context-target',
        ),
        phutil_tag(
          'td',
          array(
            'class' => 'differential-shield',
            'colspan' => 6,
          ),
          $content)));
  }

  abstract protected function renderColgroup();


  protected function wrapChangeInTable($content) {
    if (!$content) {
      return null;
    }

    return javelin_tag(
      'table',
      array(
        'class' => 'differential-diff remarkup-code PhabricatorMonospaced',
        'sigil' => 'differential-diff',
      ),
      array(
        $this->renderColgroup(),
        $content,
      ));
  }

  protected function renderInlineComment(
    PhabricatorInlineCommentInterface $comment,
    $on_right = false) {

    return $this->buildInlineComment($comment, $on_right)->render();
  }

  protected function buildInlineComment(
    PhabricatorInlineCommentInterface $comment,
    $on_right = false) {

    $user = $this->getUser();
    $edit = $user &&
            ($comment->getAuthorPHID() == $user->getPHID()) &&
            ($comment->isDraft())
            && $this->getShowEditAndReplyLinks();
    $allow_reply = (bool)$user && $this->getShowEditAndReplyLinks();

    return id(new DifferentialInlineCommentView())
      ->setInlineComment($comment)
      ->setOnRight($on_right)
      ->setHandles($this->getHandles())
      ->setMarkupEngine($this->getMarkupEngine())
      ->setEditable($edit)
      ->setAllowReply($allow_reply);
  }

}
