# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright (C) 2013 Canonical Ltd.
# Author: Sergio Schvezov <sergio.schvezov@canonical.com>

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function

import hashlib
import logging
import os.path

from phabletutils.device import AndroidBridge
from phabletutils import cdimage
from phabletutils import community
from phabletutils import downloads
from phabletutils import hashes
from phabletutils import resources
from phabletutils import projects
from phabletutils import settings
from phabletutils import ubuntuimage

log = logging.getLogger()


def detect_device(serial, device=None):
    '''If no argument passed determine them from the connected device.'''
    # Check CyanogenMod property
    if not device:
        adb = AndroidBridge(serial)
        adb.start()
        device = adb.getprop('ro.cm.device').strip()
    # Check Android property
    if not device:
        device = adb.getprop('ro.product.device').strip()
    log.info('Device detected as %s' % device)
    # property may not exist or we may not map it yet
    if device not in settings.supported_devices:
        raise EnvironmentError('Unsupported device, autodetect fails device')
    return device


def setup_cdimage_files(project_name, uri, download_dir, series,
                        device, legacy=False):
    downloads.setup_download_directory(download_dir)
    templ_arch_any = settings.files_arch_any[project_name]
    templ_arch_all = settings.files_arch_all[project_name]
    file_names = {}
    for key in templ_arch_any:
        file_names[key] = templ_arch_any[key] % (series, device)
    for key in templ_arch_all:
        file_names[key] = templ_arch_all[key] % series
    if legacy:
        hash_func = hashlib.md5
        hash_dict = {}
        for key in file_names:
            file_hash = hashes.load_hash(uri, '%s.md5sum' %
                                         file_names[key], download_dir)
            hash_dict[file_names[key]] = file_hash[file_names[key]]
    else:
        hash_func = hashlib.sha256
        hash_dict = hashes.load_hash(uri, 'SHA256SUMS', download_dir)
    files = {}
    for key in file_names:
        if uri:
            file_uri = '%s/%s' % (uri, file_names[key])
        else:
            file_uri = None
        files[key] = resources.File(
            file_path=os.path.join(download_dir, file_names[key]),
            file_uri=file_uri,
            file_hash=hash_dict[file_names[key]],
            file_hash_func=hash_func)
    return files


def setup_cdimage_touch(args):
    device = detect_device(args.serial, args.device)
    series = args.series
    base_uri = '%s/%s' % (settings.cdimage_uri_base, args.project)

    if args.base_path:
        uri = None
        download_dir = args.base_path
    else:
        daily_uri = '%s/daily-preinstalled' % (base_uri, )
        build = cdimage.get_build(daily_uri, args.pending)
        uri = '%s/%s' % (daily_uri, build)
        download_dir = downloads.get_full_path(
            os.path.join(settings.download_dir, args.project, build))
    files = setup_cdimage_files(
        args.project, uri, download_dir, series, device)
    return cdimage_project(files, args)


def setup_cdimage_legacy(args):
    series = args.series
    uri = args.uri
    device = detect_device(args.serial, args.device)
    if args.base_path:
        download_dir = args.base_path
    elif args.revision or args.latest_revision:
        build = args.build
        uri = args.uri
        download_dir = downloads.get_full_path(
            os.path.join(args.project, series, build))
    else:
        base_uri = '%s/%s' % (settings.cdimage_uri_base, args.project)
        daily_uri = '%s/daily-preinstalled' % (base_uri, )
        build = cdimage.get_build(daily_uri)
        uri = '%s/%s' % (daily_uri, build)
        download_dir = downloads.get_full_path(
            os.path.join(settings.download_dir, args.project, build))
    files = setup_cdimage_files(
        args.project, uri, download_dir, series, device, legacy=True)
    return cdimage_project(files, args)


def cdimage_project(files, args):
    if args.bootstrap:
        return projects.UbuntuTouchBootstrap(
            system=files['system_img'],
            boot=files['boot_img'],
            recovery=files['recovery_img'],
            ubuntu=files['ubuntu_zip'])
    else:
        if args.device_path:
            files['device_zip'] = args.device_path
        if args.ubuntu_path:
            files['ubuntu_zip'] = args.ubuntu_path
        return projects.UbuntuTouchRecovery(
            device=files['device_zip'],
            ubuntu=files['ubuntu_zip'],
            wipe=args.wipe)


def setup_ubuntu_system(args):
    device = detect_device(args.serial, args.device)
    uri = args.alternate_server
    if args.revision <= 0:
        json = ubuntuimage.get_json_from_index(device,
                                               args.channel,
                                               args.revision,
                                               uri)
    else:
        json = ubuntuimage.get_json_from_version(device,
                                                 args.channel,
                                                 args.revision,
                                                 uri)
    download_dir_part = os.path.join(settings.download_dir, args.project)
    if uri != settings.system_image_uri:
        base_uri = downloads.get_sha256_hash(uri)
        download_dir_part = os.path.join(download_dir_part, base_uri)
    download_dir = downloads.get_full_path(download_dir_part)

    files, command_part = ubuntuimage.get_files(download_dir, uri, json)
    return projects.UbuntuTouchSystem(
        file_list=files,
        command_part=command_part,
        device=device,
        backup=args.backup)


def setup_community(args):
    config_dir = community.branch_project(args.device)
    json_dict = community.load_manifest(config_dir)
    download_dir = community.get_download_dir(
        args.device, json_dict['revision'])
    files = community.get_files(json_dict, download_dir, args.series)
    return projects.UbuntuTouchRecovery(
        device=files['device'],
        ubuntu=files['ubuntu'],
        storage=json_dict['storage'],
        wipe=args.wipe)


def list_revisions(args):
    # Easy hack to get rid of the logger and inhibit requests from logging
    log.setLevel(logging.FATAL)
    revisions = cdimage.get_available_revisions(args.uri)
    cdimage.display_revisions(revisions)
