;;;
;;; Experimental pure CL driver tailored for pgloader. It only supports:
;;;
;;;  - login
;;;  - Bulk Copy Protocol for DB_QUERYOUT
;;;
;;; See http://msdn.microsoft.com/en-us/library/dd304523.aspx

(defpackage #:pgloader.mssql.bcp
  (:use :cl #:pgloader.params))

(in-package #:pgloader.mssql.bcp)

#|
Packet

 Header
  Type     1 byte unsigned char
  Status   1-byte unsigned char
  Length   2-byte unsigned short int, big-endian, includes packet header
  SPID     2-byte value, always send 0x0000
  PacketID 1-byte unsigned char, currently ignored
  Window   1-byte, 0x00

 Data
  Token Stream      SQL command with Binary Data, RPC, FeatureExtAck
                    Login Response, Row Data, Return Status, Return Parameters,
                    DONE, Session State, Error/Info

    Token Specifier
    Token Specific Data

  Tokenless Stream  All necessary information is in the packet header
|#
(defconstant +packet-type-sql-batch+       1)
(defconstant +packet-type-old-login+       2)
(defconstant +packet-type-rpc+             3)
(defconstant +packet-type-server-response+ 4)
(defconstant +packet-type-attention+       6)
(defconstant +packet-type-bulk-copy+       7)
(defconstant +packet-type-transaction+     14)
;; (defconstant +packet-type-query-5.0        #x0f)
(defconstant +packet-type-login            16)
(defconstant +packet-type-sspi             17)
(defconstant +packet-type-prelogin         18)

(defconstant +packet-status-normal+     #x00)
(defconstant +packet-status-eom+        #x01)
(defconstant +packet-status-ignore+     #x02)
(defconstant +packet-status-reset+      #x08)
(defconstant +packet-status-reset-skip+ #x10)

(defstruct mssql-packet type last-packet-p size data)

(defclass mssql-connection ()
  ((host :initarg :host :accessor ms-host)
   (port :initarg :port :accessor ms-port)
   (user :initarg :user :accessor ms-user)
   (pass :initarg :pass :accessor ms-pass)
   (dbname :initarg :dbname :accessor ms-dbname)
   (socket :initarg :socket :accessor ms-socket)
   (stream :initarg :stream :accessor ms-stream)
   (version :initarg :version :initform #x402 :accessor ms-version))
  (:documentation "MS SQL Driver connection object."))

(defun read-packet-header (stream)
  (let* ((type          (read-byte stream))
         (last-packet-p (= #x01 (read-byte stream)))
         (size          0)
         (unknown       (make-array 4 :element-type '(unsigned-byte 8))))
    (setf size (+ (read-byte stream)
                  (ash (read-byte stream) 8)))

    ;; then discard 4 bytes
    (read-sequence unknown stream)

    ;; return the packet structure with the header filled in.
    (make-mssql-packet :type type :last-packet-p last-packet-p :size size)))

(defun read-component-packet (stream packet position &optional block-size)
  "Read a packet that continues a previous one."
  ;; we need to ignore the 8 bytes header
  (read-sequence (make-array 8 :element-type '(unsigned-byte 8)))
  (let* ((chunk-size (- block-size 8)))
    (read-sequence chunk-size stream :start position)))

(defun read-packet (stream)
  "Read a single packet from the SQL Server / Sybase protocol"
  (let ((packet (read-packet-header stream)))
    (loop )
    packet))

(defun connect (&key host (port 1143) user password database)
  "Connect to a MSSQL database service, using TCP."
  (let* ((socket (usocket:socket-connect host port
                                          :protocol :stream
                                          :timeout 3
                                          :element-type '(unsigned-byte 8)))
         (stream (usocket:socket-stream socket)))
    (unless stream
      (error "Failed to connect to ~a:~a" host port))
    (make-instance 'mssql-connection
                   :host host
                   :port port
                   :user user
                   :pass password
                   :dbname database
                   :socket socket
                   :stream stream)))

(defun disconnect (connection)
  "Disconnect from given CONNECTION."
  (when (ms-stream connection)
    (usocket:socket-close (ms-socket connection))
    (setf (ms-socket connection) nil
          (ms-stream connection) nil)))

(defun write-string-upto (bytes string &optional (stream *standard-output*))
  "Write only up to BYTES bytes from STRING into STREAM."
  (write-sequence (babel:string-to-octets string :encoding :utf-8) stream
                  :end (min (length string) bytes))
  (when (< (length string) bytes)
    ;; complete with zeros
    (write-sequence (make-array (- bytes (length string))
                                :initial-element #x0
                                :element-type '(unsigned-byte 9))
                    stream)))

(defun write-int (int size &optional (stream *standard-output*))
  "Write the integer value INT to STREAM."
  (declare (fixnum int))
  (loop :repeat size
     :for i fixnum :from 0 :by 8
     :do (write-byte (ldb (byte 8 i) int) stream)))

(defun getpid ()
  "Get the current process PID."
  #+sbcl (sb-posix:getpid)
  #+ccl  (#_getpid))

(defun get-login-packet (connection)
  "Prepare the login packet."
  (let ((login-max-string-size 29)
        (local-server-name     (getenv-default "ASA_DATABASE" (machine-instance)))
        (*standard-output*     (ms-stream connection))
        (stream                (ms-stream connection)))
    ;; see src/tds/login.c in FreeTDS source code for details.
    (write-string-upto login-max-string-size (ms-host connection))
    (write-string-upto login-max-string-size (ms-user connection))
    (write-string-upto login-max-string-size (ms-pass connection))
    (write-string-upto login-max-string-size (format nil "~d" (getpid)))

    ;; Some magic bytes which depend on endianness, nice!
    (write-sequence #+little-endian #(#x03 #x01 #x06 #x0a #x09 #x01) ; le1
                    #+big-endian    #(#x02 #x00 #x06 #x04 #x08 #x01) ; be1
                    stream)

    (write-byte #x1 stream)            ; bulk copy
    (write-sequence #(#x0 #x0) stream) ; magic2

    (write-int (if (= #x402 (ms-version connection)) 512 0) 4 stream)

    (write-sequence #(#x0 #x0 #x0) stream) ; magic3

    ;; appname
    (write-string-upto login-max-string-size "pgloader")
    (write-string-upto login-max-string-size local-server-name)

    (if (= #x402 (ms-version connection))
        (write-string-upto 255 (ms-pass connection))

        (let* ((len (length (ms-pass connection)))
               (len (if (< len 254) len 0)))
          (write-byte 0 stream)
          (write-byte len stream)
          (write-string-upto 253 (ms-pass connection))
          (write-byte (+ len 2) stream)))

    ;; protocol version
    (write-sequence (ecase (ms-version connection)
                      (#x402 #(#x4 #x2 #x0 #x0))
                      (#x406 #(#x4 #x6 #x0 #x0))
                      (#x500 #(#x5 #x0 #x0 #x0)))
                    stream)

    ;; client program name
    (write-string-upto 10 "pgloader")

    ;; program version
    (if (= #x402 (ms-version connection))
        (write-int 0 4 stream)
        (write-sequence (ecase (ms-version connection)
                          (#x402 #(#x4 #x2 #x0 #x0))
                          (#x406 #(#x4 #x2 #x0 #x0))
                          (#x500 #(#x5 #x0 #x0 #x0)))
                        stream))

    ;; some more magic
    (write-sequence #+little-endian #(#x0 13 17) ; le2
                    #+big-endian #(#x0 12 16)    ; be2
                    stream)

    ;; language
    (write-string-upto login-max-string-size "us-english")
    (write-byte 1 stream)              ; suppress language
    (write-sequence #(#x0 #x0) stream) ; magic5
    (write-byte 0 stream)              ; encryption level
    (write-sequence #(#x0 #x0 #x0 #x0 #x0 #x0 #x0 #x0 #x0 #x0) stream) ; magic6

    ;; charset
    (write-string-upto login-max-string-size "utf8")
    (write-byte 1 stream)

    ;; network packet size
    (write-string-upto 6 (format nil "~d" 512)) ; defaults to 512

    (ecase (ms-version connection)
      (#x402 (write-sequence (make-array 8
                                         :initial-element #x0
                                         :element-type '(unsigned-byte 8))
                             stream))

      (#x406 (write-sequence (make-array 4
                                         :initial-element #x0
                                         :element-type '(unsigned-byte 8))
                             stream))

      (#x500 (write-sequence (make-array 4
                                         :initial-element #x0
                                         :element-type '(unsigned-byte 8))
                             stream)
             ;; TODO capabilities
             ))))


(defun test-mssql (&key
                     (host "54.148.14.220")
                     (port 1433)
                     (user "dim")
                     (password "h4ckm3")
                     (database "magnus_customer_sample"))
  (declare (ignore port))
  (let ((conn (mssql:connect database user password host)))
    (mssql:disconnect conn)))

(defun test-bcp (&key
                   (host "54.148.14.220")
                   (port 1433)
                   (user "dim")
                   (password "h4ckm3")
                   (database "magnus_customer_sample"))
  (let ((conn (connect :host host :port port :user user :password password
                       :database database)))
    (login conn)
    (disconnect conn)))

