/*
 * pfmon.h
 *
 * Copyright (C) 2001-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#ifndef __PFMON_H__
#define __PFMON_H__
#include <sys/types.h>
#include <inttypes.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <sys/resource.h>
#include <getopt.h>
#include <perfmon/perfmon.h>
#include <perfmon/pfmlib.h>

#define PFMON_VERSION		"3.0"

/*
 * undef to remove debug code
 */
#define PFMON_DEBUG	1

/*
 * type used for context identifier which is a file descriptor
 * with perfmon-2
 */
typedef int pfmon_ctxid_t;

/*
 * generic type to describe a perfmon context structure.
 * we use void because the area is dynamically allocated 
 * and contains a pfarg_context_t followed by an option 
 * sampling format set options
 */
typedef void	pfmon_ctx_arg_t;

/* 
 * max number of cpus (threads) supported
 */
#define PFMON_MAX_CPUS		1024 /* MUST BE power of 2 */
#define PFMON_CPUMASK_BITS	(sizeof(unsigned long)<<3)
#define PFMON_CPUMASK_COUNT	(PFMON_MAX_CPUS/PFMON_CPUMASK_BITS)

#define PFMON_MAX_FILENAME_LEN	256	/* max for output/input files */
#define PFMON_MAX_CMDLINE_LEN	1024	/* per-task mode max cmdline length /proc/cmdline */
#define PFMON_MAX_EVTNAME_LEN	128	/* maximum length for an event name */

#define PFMON_MAX_PMCS		PFMLIB_MAX_PMCS	/* max number of PMCS we can deal with */
#define PFMON_MAX_PMDS		PFMLIB_MAX_PMDS	/* max number of PMDS we can deal with */

/*
 * limits of for start/stop triggers, number cannot be greater than max number of 
 * hardware breakpoints you can set.
 */
#define PFMON_MAX_TRIGGER_CODE	4	/* simultaneous code trigger ranges we can support */
#define PFMON_MAX_TRIGGER_DATA	4	/* simultaneous data trigger ranges we can support */

/*
 * structure to encapsulate all the various in/out parameters
 * to the libpfm library
 */
typedef struct {
	pfmlib_input_param_t	inp;		/* common input parameters to libpfm            */
	pfmlib_output_param_t	outp;		/* common output parameters from libpfm         */
	void			*mod_inp;	/* model specific input parameters to libpfm    */
	void			*mod_outp;	/* model specific output parameters from libpfm */
} pfmon_lib_param_t;

typedef unsigned long pfmon_cpumask_t[PFMON_CPUMASK_COUNT];

/*
 * pfmon sampling descriptor. contains information about a
 * particular sampling session
 */
typedef struct {
	void		 *smpl_hdr;	/* virtual address of sampling buffer header */
	FILE		 *smpl_fp;	/* sampling file descriptor */
	void		 *sym_hash;	/* hash descriptor for symbol resolution */
	uint64_t	 *aggr_count;	/* entry count for aggregated session (protected by aggr_lock) */
	uint64_t	 entry_count;	/* number of entries recorded for this buffer */
	uint64_t	 last_ovfl;	/* last buffer overflow observed XXX: format specific */
	uint64_t	 last_count;	/* last number of entries in ovferflowed buffer */
	pid_t		 pid;		/* process which owns this context */
	unsigned int	 cpu;		/* on which CPU does this apply to (system wide) */
	int processing;
} pfmon_smpl_desc_t;

/*
 * sampling output format description
 */
typedef uint64_t pfmon_pmu_mask_t;

typedef struct {
	int			(*process_samples)(pfmon_smpl_desc_t *smpl);
	int			(*check_new_samples)(pfmon_smpl_desc_t *smpl);
	int			(*check_version)(pfmon_smpl_desc_t *smpl);
	int			(*print_header)(pfmon_smpl_desc_t *smpl);
	int			(*print_footer)(pfmon_smpl_desc_t *smpl);
	int			(*initialize_ctx_arg)(void *addr, unsigned int max_pmds_sample);
	size_t			(*get_fmt_arg_size)(void);
	int			(*parse_options)(int code, char *optarg, pfmon_lib_param_t *evt);
	void			(*show_options)(void);
	int			(*initialize_module)(void);
	int			(*initialize_session)(pfmon_smpl_desc_t *smpl);
	int			(*terminate_session)(pfmon_smpl_desc_t *smpl);
	int			(*validate_options)(void);
	int			(*validate_events)(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd);
	char			*name;		/* name of the format */
	char			*description;	/* one line of text describing the format */
	pfmon_pmu_mask_t	pmu_mask;	/* mask of which PMUs are supported */
	pfm_uuid_t		uuid;		/* UUID for the kernel module used */
} pfmon_smpl_module_t;

#define PFMON_CHECK_VERSION(h)	(PFM_VERSION_MAJOR((h)) != PFM_VERSION_MAJOR(PFMON_SMPL_VERSION))

#define PFMON_PMU_MASK(t)	(1ULL<<(t))

/*
 * sampling rates type definition
 */
typedef struct {
	uint64_t	value;	/* sampling period */
	uint64_t	mask;	/* bitmask used with randomization */
	uint64_t	seed;	/* seed value for randomization */
	unsigned int	flags;	/* type of value */
} pfmon_smpl_rate_t;

#define PFMON_RATE_NONE_SET	0x0	/* sampling rate non set by user */
#define PFMON_RATE_VAL_SET	0x1	/* sampling rate set by user */
#define PFMON_RATE_MASK_SET	0x2	/* randomization mask set by user */
#define PFMON_RATE_SEED_SET	0x4	/* randomization seed set by user */

/*
 * pfmon monitor definition
 */
typedef struct {
	unsigned int	event;	/* event index (from libpfm) */
	unsigned int	plm;	/* priv level  */
	unsigned int	flags;	/* flags */
} pfmon_monitor_t;

#define PFMON_MONITOR_FL_RANDOMIZE	0x1 /* monitor use randomization */

typedef struct {
	unsigned int start:1;		/* true if a start breakpoint */
	unsigned int active:1;		/* true if breakpoint is active */
	unsigned int repeat:1;		/* reuse breakpoint more than once, default oneshot */
	unsigned int inherit:1;		/* true if follow fork/clone/vfork */
	unsigned int rw:2;		/* read/write for debug register (data) */
	unsigned int func:1;		/* function trigger: must stop when leaving function */
	unsigned int reserved:25;	/* for future use */
} pfmon_trigger_state_t;

typedef struct {
	uintptr_t		brk_address;	/* where to place the breakpoint */
	pfmon_trigger_state_t	attr;
	unsigned int		br_idx;		/* which logical debug register was used */
	unsigned int		stop_trg_idx;	/* pfmon_trigger containing stop point (func=1) */
} pfmon_trigger_t;

/* some handy shortcuts */
#define trg_attr_start		attr.start
#define trg_attr_active		attr.active
#define trg_attr_repeat		attr.repeat
#define trg_attr_inherit	attr.inherit
#define trg_attr_rw		attr.rw
#define trg_attr_func		attr.func

/*
 * session description structure
 */
typedef struct {
	unsigned int	monitoring:1;	/* measurement started on the task */
	unsigned int	dispatched:1;	/* task is assigned to a worker CPU */
	unsigned int	seen_stopsig:1;	/* detection of task creation for ptrace */
	unsigned int	detaching:1;	/* in the process of detaching the task */
	unsigned int	attached:1;	/* task was monitored via direct/indirect PTRACE_ATTACH */
	unsigned int	singlestep:1;	/* waiting for single step notification */
	unsigned int	reserved:26;
} pfmon_sdesc_flags_t;

typedef struct _pfmon_desc_t {
	struct _pfmon_desc_t	*next;				/* pid hash chain */
	struct _pfmon_desc_t	*fd_next;			/* fd hash chain */
	unsigned short		type;				/* task descriptor type */
	unsigned short		refcnt;				/* #reference to object (max=2) */
	pfmon_sdesc_flags_t	flags;				/* active, inactive */

	pid_t			pid;				/* process identification */
	pid_t			ppid;				/* parent process identification */
	pid_t			tid;				/* thread identification */
	pid_t			ptid;				/* parent thread identification */

	pfmon_ctxid_t		ctxid;				/* perfmon session identifier */
	pfmon_smpl_desc_t	csmpl;				/* sampling descriptor */
	pthread_mutex_t		lock;				/* used to protect refcnt */

	unsigned int		exec_count;			/* number of times the task exec'd */

	void			*data;

	pfmon_ctx_arg_t		*ctx;				/* master context + fmt argument */
	pfarg_reg_t 		pd[PFMON_MAX_PMDS];		/* master pmd */
	pfarg_reg_t 		pc[PFMON_MAX_PMCS];		/* master pmc */

	struct timeval		tv_start;			/* time of activation */

	unsigned int		pc_count;			/* #entries in master pmc */
	unsigned int		pd_count;			/* #entries in master pmd */
	unsigned int		cpu;				/* worker assigned to sdesc, if any */

	pfmon_trigger_t		code_triggers[PFMON_MAX_TRIGGER_CODE];
	pfmon_trigger_t		data_triggers[PFMON_MAX_TRIGGER_DATA];

	unsigned int		num_code_triggers;
	unsigned int		num_data_triggers;

	char			cmdline[PFMON_MAX_CMDLINE_LEN];		/* process command line */
	char			new_cmdline[PFMON_MAX_CMDLINE_LEN];	/* process command line from exec() */
} pfmon_sdesc_t;

#define fl_monitoring	flags.monitoring
#define fl_dispatched	flags.dispatched
#define fl_seen_stopsig flags.seen_stopsig
#define fl_detaching	flags.detaching
#define fl_attached	flags.attached

#define PFMON_SDESC_ATTACH	0	/* task was attached */
#define PFMON_SDESC_FORK	1	/* created via fork() */
#define PFMON_SDESC_VFORK	2	/* created via vfork() */
#define PFMON_SDESC_CLONE	3	/* create via thread clone2() */

#define LOCK_SDESC(h)	pthread_mutex_lock(&(h)->lock)
#define UNLOCK_SDESC(h)	pthread_mutex_unlock(&(h)->lock)

typedef struct {
	struct {
		int opt_plm;		/* which privilege level to monitor (more than one possible) */
		int opt_debug;		/* print debug information */
		int opt_verbose;	/* verbose output */
		int opt_append;		/* append to output file */
		int opt_block;		/* block child task on counter overflow */
		int opt_syst_wide;

		int opt_with_header;      /* generate header on output results (smpl or not) */
		int opt_use_smpl;	  /* true if sampling is requested */
		int opt_aggr;	  	  /* aggregate results */
		int opt_print_cnt_mode;	  /* mode for printing counters */
		int opt_show_rusage;	  /* show process time */
		int opt_sysmap_syms;	  /* use System.map format for symbol file */
		int opt_check_evt_only;   /* stop after checking the event combination is valid */
		int opt_smpl_print_counts;/* print counters values when sampling session ends */
		int opt_attach;		  /* set to 1 if attach pid is specified by user */
		int opt_reset_non_smpl;	  /* reset non-overflowed, non-sampling counters on overflow */
		int opt_dont_start;	  /* do not explicitely start monitoring */
		int opt_follow_fork;	  /* follow fork in per-task mode */
		int opt_follow_vfork;	  /* follow vfork in per-task mode */
		int opt_follow_pthread;	  /* follow clone (pthreads) in per-task mode */
		int opt_follow_exec;	  /* follow exec in per-task mode */
		int opt_follow_exec_excl; /* follow exec pattern is excluding */
		int opt_follow_all;	  /* follow all of the above */
		int opt_follows;	  /* set to true if one of the follow* is used */
		int opt_cmd_no_verbose;	  /* redirect command stdout to /dev/null */
		int opt_code_trigger_repeat;  /* repeat start/stop monitoring each time code trigger start/stop is crossed */
		int opt_code_trigger_follow;  /* code trigger start/stop used in all monitored tasks (not just first one) */
		int opt_data_trigger_repeat;  /* repeat start/stop monitoring each time data trigger start/stop is touched */
		int opt_data_trigger_follow;  /* data trigger start/stop used in all monitored tasks (not just first one) */
		int opt_data_trigger_ro;  /* data trigger activated for read access only (default RW) */
		int opt_data_trigger_wo;  /* data trigger activated for write access only (default RW) */
		int opt_block_restart;	  /* for debug only: force a keypress before PFM_RESTART */
		int opt_split_exec;	  /* split result output on execve() */
		int opt_support_gen;	  /* operating in a generic PMU mode is supported */
		int opt_addr2sym;	  /* try converting addresses to symbol */
		int opt_no_ovfl_notify;	  /* do not notify sampling buffer overflow (saturation) */
	} program_opt_flags;

	char  **argv;			/* full command line */
	char  **command;		/* keep track of the command to execute (per-task) */

	char *outfile;			/* basename for output filename for counters */
	char *smpl_outfile;		/* basename for sampling output file */
	char *symbol_file;		/* name of file which holds symbol table */

	char *fexec_pattern;		/* follow-exec pattern */

	pid_t	attach_pid;		/* process to attach to */

	pfmon_cpumask_t cpu_mask;	/* which CPUs to use in system wide mode */

	unsigned int	trigger_delay; 	/* number of seconds to wait before start a session */
	unsigned int	session_timeout;  /* number of seconds to wait before stopping a session */

	char	      	*priv_lvl_str;  	/* per-event privilege level option string */
	char		*code_trigger_start;	/* code trigger start cmdline */
	char		*code_trigger_stop;	/* code trigger stop cmdline */

	char		*data_trigger_start;	/* data trigger start cmdline */
	char		*data_trigger_stop;	/* data trigger stop cmdline */
	char		*cpu_list;		/* CPU list cmdline */

	pfmon_trigger_t	code_triggers[PFMON_MAX_TRIGGER_CODE];
	pfmon_trigger_t	data_triggers[PFMON_MAX_TRIGGER_DATA];

	unsigned int	num_code_triggers;	/* number of code triggers */
	unsigned int	num_data_triggers;	/* number of data triggers */

	unsigned int	max_sym_modules;	/* how many symbol table we have loaded */

	unsigned long	online_cpus;	/* _SC_NPROCESSORS_ONLN */
	unsigned long	config_cpus;	/* _SC_NPROCESSORS_CONF */
	unsigned long	selected_cpus;	/* number of CPUS  selected by user */
	unsigned int 	nibrs;		/* number of code debug registers */
	unsigned int 	ndbrs;		/* number of data debug registers */
	unsigned int 	max_counters;	/* maximum number of counters */
	unsigned int	page_size;	/* page size used by host kernel */
	unsigned int	clock_tick;	/* _SC_CLK_TCK */
	int		libpfm_generic;	/* generic PMU type */

	unsigned int	pfm_version;		/* kernel perfmon version */
	unsigned int	pfm_smpl_version;	/* kernel perfmon sampling format version */

	unsigned int 	monitor_count;		/* how many counters specified by user */
	pfmon_monitor_t events[PFMON_MAX_PMDS];	/* event information */
	size_t		max_event_name_len;	/* max len for event name (pretty printing) */

	unsigned long		common_smpl_pmds;		/* PMDs to include in every samples */
	unsigned long		common_reset_pmds;		/* PMDs to reset for any counter overflow */
	unsigned long 		smpl_pmds[PFMON_MAX_PMCS];	/* for each monitor, which PMDs are recorded in sampling buffer */
	unsigned long 		rev_smpl_pmds[PFMON_MAX_PMCS];	/* reverse map of smpl_pmds, index by PMC number */

	pfmon_smpl_rate_t 	long_rates[PFMON_MAX_PMDS];	/* long sampling period rates */
	pfmon_smpl_rate_t 	short_rates[PFMON_MAX_PMDS];	/* short sampling period rates */

	int 			rev_pc[PFMON_MAX_PMDS];		/* pmd[x] -> monitor_events[y] */

	pfmon_smpl_module_t	*smpl_mod;			/* which sampling module to use */

	size_t			ctx_arg_size;
	void 			*model_options;			/* point to model specific pfmon options */
} program_options_t;

#define PFMON_CPUMASK_SET(m, g)		((m)[(g)/PFMON_CPUMASK_BITS] |= (1UL << ((g) % PFMON_CPUMASK_BITS)))
#define PFMON_CPUMASK_CLEAR(m, g)	((m)[(g)/PFMON_CPUMASK_BITS] &= ~(1UL << ((g) % PFMON_CPUMASK_BITS)))
#define PFMON_CPUMASK_ISSET(m, g)	((m)[(g)/PFMON_CPUMASK_BITS] & (1UL << ((g) % PFMON_CPUMASK_BITS)))

#define opt_plm			program_opt_flags.opt_plm
#define opt_debug		program_opt_flags.opt_debug
#define opt_verbose		program_opt_flags.opt_verbose
#define opt_append		program_opt_flags.opt_append
#define opt_block		program_opt_flags.opt_block
#define opt_fclone		program_opt_flags.opt_fclone
#define opt_syst_wide		program_opt_flags.opt_syst_wide
#define opt_with_header		program_opt_flags.opt_with_header
#define opt_use_smpl		program_opt_flags.opt_use_smpl
#define opt_aggr		program_opt_flags.opt_aggr
#define opt_print_cnt_mode	program_opt_flags.opt_print_cnt_mode
#define opt_show_rusage		program_opt_flags.opt_show_rusage
#define opt_sysmap_syms		program_opt_flags.opt_sysmap_syms
#define opt_check_evt_only	program_opt_flags.opt_check_evt_only
#define opt_smpl_print_counts   program_opt_flags.opt_smpl_print_counts
#define opt_attach		program_opt_flags.opt_attach
#define opt_reset_non_smpl	program_opt_flags.opt_reset_non_smpl
#define opt_dont_start		program_opt_flags.opt_dont_start
#define opt_follow_fork		program_opt_flags.opt_follow_fork
#define opt_follow_vfork	program_opt_flags.opt_follow_vfork
#define opt_follow_pthread	program_opt_flags.opt_follow_pthread
#define opt_follow_exec		program_opt_flags.opt_follow_exec
#define opt_follow_exec_excl	program_opt_flags.opt_follow_exec_excl
#define opt_follow_all		program_opt_flags.opt_follow_all
#define opt_follows		program_opt_flags.opt_follows
#define opt_cmd_no_verbose	program_opt_flags.opt_cmd_no_verbose
#define opt_code_trigger_repeat	program_opt_flags.opt_code_trigger_repeat
#define opt_code_trigger_follow	program_opt_flags.opt_code_trigger_follow
#define opt_data_trigger_repeat	program_opt_flags.opt_data_trigger_repeat
#define opt_data_trigger_follow	program_opt_flags.opt_data_trigger_follow
#define opt_data_trigger_ro	program_opt_flags.opt_data_trigger_ro
#define opt_data_trigger_wo	program_opt_flags.opt_data_trigger_wo
#define opt_block_restart	program_opt_flags.opt_block_restart
#define opt_split_exec		program_opt_flags.opt_split_exec
#define opt_support_gen		program_opt_flags.opt_support_gen
#define opt_addr2sym		program_opt_flags.opt_addr2sym
#define opt_no_ovfl_notify	program_opt_flags.opt_no_ovfl_notify

typedef struct {
	int	(*pfmon_prepare_registers)(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd);
	int	(*pfmon_install_registers)(pfmon_ctxid_t id, pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd);
	int	(*pfmon_print_header)(FILE *fp);
	int	(*pfmon_initialize)(pfmon_lib_param_t *);
	void	(*pfmon_usage)(void);
	int	(*pfmon_parse_options)(int code, char *optarg, pfmon_lib_param_t *evt);
	int	(*pfmon_post_options)(pfmon_lib_param_t *evt);
	void	(*pfmon_detailed_event_name)(int evt);
	char	*name;			/* support module name */
	int	pmu_type;		/* indicate the PMU type, must be one from pfmlib.h */
	unsigned int default_plm; 	/* default privilege level to monitor at */
	char	*default_event;		/* a default event name */
} pfmon_support_t;

extern pfmon_support_t *pfmon_current;

extern program_options_t options;

/* from pfmon_util.c */
extern void warning(char *fmt, ...);
extern void dbgprintf(char *fmt, ...);
extern void fatal_error(char *fmt, ...);
extern void gen_reverse_table(pfmon_lib_param_t *evt, int *rev_pc);
extern int enable_pmu(pfmon_ctxid_t id);
extern int session_start(pfmon_ctxid_t id);
extern int session_stop(pfmon_ctxid_t id);
extern void session_free(pfmon_ctxid_t id);
extern int load_context(pfmon_sdesc_t *sdesc);
extern int unload_context(pfmon_sdesc_t *sdesc);
extern int gen_event_list(char *arg, pfmon_monitor_t *events);
extern int gen_smpl_rates(char *arg, unsigned int max_count, pfmon_smpl_rate_t *rates, unsigned int *count);
extern int gen_smpl_randomization(char *arg, unsigned int max_count, pfmon_smpl_rate_t *rates, unsigned int *count);
extern int find_current_cpu(pid_t pid, unsigned int *cur_cpu);
extern int register_exit_function(void (*func)(int));
extern void print_standard_header(FILE *fp, unsigned int cpu, pfmon_sdesc_t *sdesc);
extern void vbprintf(char *fmt, ...);
extern void vbprintf_unblock(void);
extern void vbprintf_block(void);
extern void gen_code_range(char *arg, uintptr_t *start, uintptr_t *end);
extern void gen_data_range(char *arg, uintptr_t *start, uintptr_t *end);
extern void counter2str(uint64_t value, char *str);
extern void show_task_rusage(const struct timeval *start, const struct timeval *end, const struct rusage *ru);
extern int is_regular_file(char *name);
extern void print_simple_cpuinfo(FILE *fp, const char *msg);
extern int pfm_uuid2str(pfm_uuid_t uuid, size_t maxlen, char *str);
extern int pfmon_extract_cmdline(pid_t pid, char *cmdline, int maxlen);
extern void pfmon_backtrace(void);
extern unsigned int find_cpu_speed(void);
extern int pfmon_pin_self(unsigned int cpu);
extern char * priv_level_str(unsigned int plm);
extern pid_t gettid(void); 
extern pid_t pfmon_set_affinity(pid_t pid, size_t size, pfmon_cpumask_t mask); 
extern int pfmon_print_address(FILE *fp, void *hash_desc, uintptr_t addr);
extern int perfmon_debug(int m);
extern int extract_cache_size(unsigned int level, unsigned int type, unsigned long *size);

/* helper functions provided by arch-specific code */
extern int  pfmon_set_code_breakpoint(pid_t pid, int dbreg, uintptr_t address);
extern int  pfmon_clear_code_breakpoint(pid_t pid, int dbreg, uintptr_t address);
extern int  pfmon_set_data_breakpoint(pid_t pid, int dbreg, uintptr_t address, int rw);
extern int  pfmon_clear_data_breakpoint(pid_t pid, int dbreg, uintptr_t address);
extern int  pfmon_resume_after_code_breakpoint(pid_t pid);
extern int  pfmon_resume_after_data_breakpoint(pid_t pid);
extern void pfmon_print_cpuinfo(FILE *fp);
extern void pfmon_print_simple_cpuinfo(FILE *fp, char *msg);
extern void pfmon_arch_initialize(void);
extern int  pfmon_enable_all_breakpoints(pid_t pid);
extern int  pfmon_disable_all_breakpoints(pid_t pid);
extern int  pfmon_validate_code_trigger_address(uintptr_t addr);
extern int  pfmon_validate_data_trigger_address(uintptr_t addr);
extern void pfmon_segv_handler_info(struct siginfo *info, struct sigcontext *sc);
extern int  pfmon_get_breakpoint_addr(pid_t pid, uintptr_t *addr, int *is_data);
extern int  pfmon_get_return_pointer(pid_t pid, uintptr_t *rp);


/* from pfmon.c */
extern int pfmon_register_pmu_options(struct option *cmd, size_t sz);
extern int pfmon_register_smpl_mod_options(struct option *cmd, size_t sz);
extern int install_registers(pfmon_sdesc_t *sdesc, pfmon_lib_param_t *evt);

/* from pfmon_results.c */
extern int print_results(pfarg_reg_t *pd, pfmon_sdesc_t *sdesc, unsigned int cpu);

/* pfmon_smpl.c */
extern void pfmon_setup_sampling_rates(char *long_args, char *short_args, char *random_args);
extern int pfmon_reset_sampling(pfmon_sdesc_t *sdesc);
extern int pfmon_setup_common_sampling_output(pfmon_smpl_desc_t *csmpl);
extern int pfmon_setup_private_sampling_output(pfmon_sdesc_t *sdesc);
extern void pfmon_close_sampling_output(pfmon_smpl_desc_t *csmpl, pid_t tid, unsigned int cpu);
extern int pfmon_process_smpl_buf(pfmon_sdesc_t *sdesc, int need_restart);
extern int pfmon_find_smpl_module(char *name, pfmon_smpl_module_t **mod, int ignore_cpu);
extern void pfmon_list_smpl_modules(void);
extern void pfmon_smpl_module_info(pfmon_smpl_module_t *mod);
extern void pfmon_smpl_initialize(void);
extern void pfmon_smpl_mod_usage(void);

/* from pfmon_system.c */
extern int measure_system_wide(pfmon_lib_param_t *evt, pfmon_ctx_arg_t *ctx, pfarg_reg_t *pc, pfarg_reg_t *pd, char **argv);

/* from pfmon_task.c */
extern int measure_task(pfmon_lib_param_t *evt, pfmon_ctx_arg_t *ctx, pfarg_reg_t *pc, pfarg_reg_t *pd, char **argv);

/* from pfmon_symbols.c */
#define PFMON_TEXT_SYMBOL	1
#define PFMON_DATA_SYMBOL	2

#define PFMON_DFL_SYM_HASH_SIZE	12	/* log2(): number of collision lists */
#define PFMON_DFL_SYM_ENTRIES	4096	/* number of symbols that can be stored in hash table */

extern int load_kernel_syms(void);
extern int load_elf_syms(const char *filename);
extern int find_sym_addr(char *sym, int type, uintptr_t *start, uintptr_t *end);
extern int find_sym_byaddr(uintptr_t addr, int type, char **name, char **module, uintptr_t *start, uintptr_t *end);
extern int is_exact_sym(uintptr_t addr, int type);
extern void print_syms(void);
extern void pfmon_syms_hash_free(void *cookie);
extern int pfmon_syms_hash_find(void *hash_desc, uintptr_t addr, char **name, char **module, uintptr_t *start_addr);
extern int pfmon_syms_hash_alloc(unsigned long hash_log_size, unsigned long max_entries, void **cookie);

/* from pfmon_conf.c */
extern void load_config_file(void);
extern int find_opcode_matcher(char *name, unsigned long *val);
extern void print_opcode_matchers(void);

/*
 * Some useful inline functions
 */
#ifdef PFMON_DEBUG
#include <unistd.h>
#define DPRINT(a) \
	do { \
		if (options.opt_debug) { fprintf(stderr, "%s.%d: [%d] ", __FUNCTION__, __LINE__, getpid()); dbgprintf a; } \
	} while (0)
#else
#define DPRINT(a)
#endif

#define M_PMD(x)		(1UL<<(x))

#ifdef __ia64__
#include "pfmon_ia64.h"
#endif

#endif /*__PFMON_H__ */
