#!/usr/bin/env python

"""
PETSc for Python
================

Python bindings for PETSc libraries.
"""


## try:
##     import setuptools
## except ImportError:
##     pass


# --------------------------------------------------------------------
# Metadata
# --------------------------------------------------------------------

from configure import metadata

name     = 'petsc4py'
version  = open('VERSION.txt').read().strip()
descr    = __doc__.strip().split('\n'); del descr[1:3]
devstat  = ['Development Status :: 3 - Alpha']
download = 'http://cheeseshop.python.org/packages/source/%s/%s/%s-%s.tar.gz'

metadata['name'] = name
metadata['version'] = version
metadata['description'] = descr.pop(0)
metadata['long_description'] = '\n'.join(descr)
metadata['classifiers'] += devstat
metadata['download_url'] = download % (name[0], name, name, version)

# --------------------------------------------------------------------
# Extension modules
# --------------------------------------------------------------------

def get_ext_modules(Extension):
    from os   import walk
    from glob import glob
    from os.path import join, sep as pathsep, extsep, abspath

    # generare dependencies
    extroot = join('petsc', 'lib', 'ext')
    depends = []
    for pth, dirs, files in walk(join(extroot, 'swig')):
        depends += glob(join(pth, '*%si' % extsep))
    for pth, dirs, files in walk(join(extroot, 'src')):
        depends += glob(join(pth, '*%s[h,c]' % extsep))
    seprepl = lambda p: p.replace(pathsep,'/').replace(extsep,'.')
    depends = map(seprepl, depends)
    extdir  = 'petsc/lib/ext'
    petsc_c   = Extension('petsc4py.lib._petscext',
                          sources=[extdir + '/' + 'petscext_c.i',
                                   extdir + '/' + 'petsclib.c'],
                          depends=depends,
                          include_dirs=[extdir],
                          language='c')
    petsc_cxx = Extension('petsc4py.lib._petscext',
                          sources=[extdir + '/' + 'petscext_cpp.i',
                                   extdir + '/' + 'petsclib.cpp'],
                          depends=depends,
                          include_dirs=[extdir],
                          language='c++')
    return [petsc_c, petsc_cxx]

from distutils.core import Command

class bdist_dpkg(Command):
    """Make a nice .deb package
    """
    
    description = "Make a deb package using dpkg (debuild)"
    user_options = []  # distutils complains if this is not here.

    def initialize_options(self):  # distutils wants this
        pass
    
    def finalize_options(self):    # this too
        pass

    def run(self):
        """
        debian/changelog contains a version like this:

        0.4~pre+svn739-1

        This method parses it, then checkouts the svn revision as directed (739
        in this example), but applies the current top svn debian dir to it, and
        executes "debuild" in that temporary directory.
        """
        import os
        def get_changelog_version_revision():
            """Reads the first line in changelog, parses 0.4~pre+svn739-1 and
            returns ("0.4~pre",739,1)
            """
            l = file("debian/changelog").readline()
            import re
            m = re.match("petsc4py \((\S+)\+svn(\d+)\-(\d+)\) ",l)
            if m:
                g = m.groups()
                if len(g) == 3:
                    #version, svn revision, debian revision
                    #('0.4~pre', '739', '1') 
                    v, r, dr = g
                    return v, int(r), int(dr)
            print l
            raise "Don't understant the syntax in changelog"
        version,revision,drevision = get_changelog_version_revision()
        os.system("mkdir -p dist")
        tmpdir = "petsc4py-%s+svn%d" % (version, revision)
        print "exporting svn (%d) to dist/%s" % (revision,tmpdir)
        os.system("svn -q export -r %d " % revision +
                "http://petsc4py.googlecode.com/svn/trunk/ dist/%s" % tmpdir)  
        os.system("rm -rf dist/%s/debian" % tmpdir)
        print "creating dist/petsc4py_%s+svn%d.orig.tar.gz" % (version, revision)
        os.system("cd dist; tar zcf petsc4py_%s+svn%d.orig.tar.gz %s" \
                %(version, revision, tmpdir))
        print "creating the deb package"
        os.system("cp -a debian dist/%s/debian" % tmpdir)
        os.system("rm -rf dist/%s/debian/.svn" % tmpdir)
        os.system("cd dist/%s; export PETSC_DIR=/usr/lib/petsc; export PETSC_ARCH=`petscarch`; debuild -sa -us -uc" % tmpdir)
        #os.system("cd dist/%s; debuild" % tmpdir)
        os.system("rm -rf dist/%s" % tmpdir)
        print "-"*50
        print "Done. Files genereated in the dist/ directory"

class clean(Command):
    """Cleans *.pyc and debian trashs, so you should get the same copy as 
    is in the svn.
    """
    
    description = "Clean everything"
    user_options = [("all","a","the same")]  

    def initialize_options(self):  
        self.all = None
    
    def finalize_options(self):   
        pass

    def run(self):
        import os
        #os.system("py.cleanup")
        os.system("rm -rf build")
        os.system("rm -rf dist")

class test_sympy_dpkg(Command):
    
    description = "Run tests for the deb package"
    user_options = []  # distutils complains if this is not here.
    
    def initialize_options(self):  # distutils wants this
        pass
    
    def finalize_options(self):    # this too
        pass
    
    def run(self):
        import os
        from glob import glob
        g = glob("dist/*.changes")
        assert len(g) == 1
        changes = g[0]
        g = glob("dist/*.dsc")
        assert len(g) == 1
        dsc = g[0]
        g = glob("dist/*.deb")
        assert len(g) == 1
        deb = g[0]
        print "testing this package:"
        print "  ",dsc
        print "  ",changes
        print "  ",deb
        print
        print "run these commands:"
        print "lintian -i %s" % changes
        print "linda -i %s" % changes
        print "sudo pbuilder build %s" % dsc
        print "sudo piuparts -p %s" % deb

# --------------------------------------------------------------------
# Setup
# --------------------------------------------------------------------

from configure import setup
from configure import Extension
from configure import config
from configure import build
from configure import build_src
from configure import build_py
from configure import build_ext
from configure import sdist

def main():
    import os
    if not os.environ.has_key("PETSC_DIR"):
        os.environ["PETSC_DIR"] = "/usr/lib/petsc"
    if not os.environ.has_key("PETSC_ARCH"):
        os.environ["PETSC_ARCH"] = "linux-gnu-c-opt"
    setup(packages     = ['petsc4py',
                          'petsc4py.lib'],
          package_dir  = {'petsc4py'     : 'petsc',
                          'petsc4py.lib' : 'petsc/lib'},
          package_data = {'petsc4py.lib': ['petsc.cfg']},
          ext_modules  = get_ext_modules(Extension),
          cmdclass     = {'config'     : config,
                          'build'      : build,
                          'build_py'   : build_py,
                          'build_src'  : build_src,
                          'build_ext'  : build_ext,
                          'bdist_dpkg' : bdist_dpkg,
                          'test_dpkg'  : test_sympy_dpkg,
                          'clean'      : clean,
                          'sdist'      : sdist},
          **metadata)

# --------------------------------------------------------------------

if __name__ == '__main__':
    from distutils import sysconfig
    cfgvars = sysconfig.get_config_vars()
    cflags = cfgvars['OPT'].split()
    for flag in ('-g', '-g3', '-Wstrict-prototypes'):
        try:
            cflags.remove(flag)
        except ValueError:
            pass
    cfgvars['OPT'] = str.join(' ', cflags) 

if __name__ == '__main__':
    ## from distutils import log
    ## log.set_verbosity(log.DEBUG)
    main()
