# $Id: MenuBase.pm,v 1.15 2004/04/03 10:29:44 jodrell Exp $
# This file is part of PerlPanel.
# 
# PerlPanel is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# PerlPanel is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with PerlPanel; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Copyright: (C) 2003-2004 Gavin Brown <gavin.brown@uk.com>
#
package PerlPanel::MenuBase;
use vars qw(@ICON_DIRECTORIES);
use File::Basename qw(basename);
use Gtk2::SimpleList;
use strict;

our @ICON_DIRECTORIES = (
	sprintf('%s/.perlpanel/icon-files', $ENV{HOME}),
	sprintf('%s/.icons', $ENV{HOME}),
	sprintf('%s/.icons/gnome/48x48/apps', $ENV{HOME}),
	'%s/share/icons/gnome/48x48/apps',
	'%s/share/pixmaps',
);

=pod

=head1 NAME

PerlPanel::MenuBase - a base class for PerlPanel menu applets.

=head1 SYNOPSIS

	package PerlPanel::Applet::MyMenu;
	use base 'PerlPanel::MenuBase';
	use strict;

	sub create_menu {

		my $self = shift;

		$self->menu->append($self->menu_item(
			'Hello World!',
			$icon,
			sub { print "Hello World!\n" }
		));
		return 1;
	}

	1;

=head1 DESCRIPTION

C<PerlPanel::MenuBase> is a base class that does as much as possible to
abstract the nuts-and-bolts details of building a PerlPanel menu applet. If you
use C<PerlPanel::MenuBase> to write a menu applet, you don't need to worry
about menu hierarchies or icons - all that's done for you. Instead to can
concentrate on building your menu backend.

=head1 USAGE

C<PerlPanel::MenuBase> is a base class - that means, you must write a Perl
module that inherits from it. The C<use base> line in the example above is one
way you can do this. Then you simply override the C<configure()> and
C<create_menu()> methods with your own.

=cut

sub new {
	my $self		= {};
	$self->{package}	= shift;
	bless($self, $self->{package});
	return $self;
}

sub configure {
	my $self = shift;

	$self->{widget}	= Gtk2::Button->new;
	$self->{menu}	= Gtk2::Menu->new;

	$self->widget->signal_connect('clicked', sub { $self->popup });

	$self->create_menu;

	$self->add_control_items if ($self->show_control_items);

	return 1;
}

sub widget {
	return $_[0]->{widget};
}

sub menu {
	return $_[0]->{menu};
}

sub expand {
	return 0;
}

sub fill {
	return 0;
}

sub end {
	return 'start';
}

=pod

=head1 STANDARD METHODS

	$self->add_control_items;

This method appends the standard set of PerlPanel control options to the
menu. The menu will subsequently look like this:

	|				|
	| ----------------------------- |
	| Lock Screen			|
	| Run Program...		|
	| Take Screenshot...		|
	| ----------------------------- |
	| Shut Down...			|
	| Reboot...			|
	| ----------------------------- |
	| Configure...			|
	| Reload			|
	| ----------------------------- |
	| About...			|
	+-------------------------------+

=cut

sub add_control_items {
	my $self = shift;
	if (scalar($self->menu->get_children) > 0) {
		$self->menu->append(Gtk2::SeparatorMenuItem->new);
	}
	chomp(my $xscreensaver = `which xscreensaver-command 2> /dev/null`);
	if (-x $xscreensaver) {
		$self->menu->append($self->menu_item(_('Lock Screen'), PerlPanel::get_applet_pbf_filename('lock'), sub { system("$xscreensaver -lock &") }));
	}
	$self->menu->append($self->menu_item(_('Run Program...'), PerlPanel::get_applet_pbf_filename('commander'), sub {
		require('Commander.pm');
		PerlPanel::Applet::Commander->run;
	}));
	$self->menu->append($self->menu_item(_('Take Screenshot...'), PerlPanel::get_applet_pbf_filename('screenshot'), sub {
		require('ScreenShot.pm');
		my $screenshot = PerlPanel::Applet::ScreenShot->new;
		$screenshot->configure;
		$screenshot->prompt;
	}));
	$self->menu->append(Gtk2::SeparatorMenuItem->new);
	# here we callously assume that the presence of this file means that bog-standard users can poweroff and reboot:	
	if (-e '/etc/pam.d/poweroff') {
		$self->menu->append($self->menu_item(_('Shut Down...'), sprintf('%s/share/pixmaps/%s/shutdown.png', $PerlPanel::PREFIX, lc($PerlPanel::NAME)), sub {
			PerlPanel::question(
				_('Are you sure you want to shut down?'),
				sub { system("poweroff") },
				sub { },
			);
		}));
	}
	if (-e '/etc/pam.d/reboot') {
		$self->menu->append($self->menu_item(_('Reboot...'), sprintf('%s/share/pixmaps/%s/reboot.png', $PerlPanel::PREFIX, lc($PerlPanel::NAME)), sub {
			PerlPanel::question(
				_('Are you sure you want to reboot?'),
				sub { system("reboot") },
				sub { },
			);
		}));
	}
	$self->menu->append(Gtk2::SeparatorMenuItem->new);
	$self->menu->append($self->menu_item(_('Configure...'), PerlPanel::get_applet_pbf_filename('configurator'), sub {
		require('Configurator.pm');
		my $configurator = PerlPanel::Applet::Configurator->new;
		$configurator->configure;
		$configurator->init;
	}));
	$self->menu->append($self->menu_item(_('Reload'), PerlPanel::get_applet_pbf_filename('reload'), sub { PerlPanel::reload }));

	my $item = $self->menu_item(_('Add To Panel'), 'gtk-add');
	my $menu = Gtk2::Menu->new;
	$item->set_submenu($menu);

	my @files;
	foreach my $dir (@PerlPanel::APPLET_DIRS) {
		opendir(DIR, $dir) or next;
		push(@files, grep { /\.pm$/ } readdir(DIR));
		closedir(DIR);
	}

	@files = sort(@files);

	require('Configurator.pm');
	my $registry = {};
	PerlPanel::Applet::Configurator::load_appletregistry($registry);
	foreach my $file (@files) {
		my ($appletname, undef) = split(/\./, $file, 2);
		my $item = $self->menu_item(
			$appletname,
			PerlPanel::get_applet_pbf($appletname, PerlPanel::icon_size),
			sub {$self->add_applet_dialog($appletname)},
		);
		PerlPanel::tips->set_tip($item, $registry->{registry}->{$appletname});
		$menu->append($item);
	}

	$self->menu->append($item);

	$self->menu->append(Gtk2::SeparatorMenuItem->new);

	$self->menu->append($self->menu_item(_('About...'), PerlPanel::get_applet_pbf_filename('about'), sub {
		require('About.pm');
		my $about = PerlPanel::Applet::About->new;
		$about->configure;
		$about->about;
	}));
	return 1;
}

=pod

	my $item = $self->menu_item($label, $icon, $callback);

This returns a ready-prepared Gtk2::ImageMenuItem. This method does a lot of
hard work for you - C<$label> is set as the text label for the item, and if
defined, C<$callback> is connected to the C<'activate'> signal.

C<$icon> can be either a file, a C<Gtk::Gdk::Pixbuf>, or a stock ID.
C<menu_item> will automagically resize the icon to fit in with the rest of
the menu.

=cut

sub menu_item {
	my ($self, $label, $icon, $callback) = @_;
	my $item = Gtk2::ImageMenuItem->new_with_label($label);
	my $pbf;
	if (-e $icon) {
		# it's a file:
		$pbf = Gtk2::Gdk::Pixbuf->new_from_file($icon);
	} elsif (ref($icon) eq 'Gtk2::Gdk::Pixbuf') {
		# it's a pixbuf:
		$pbf = $icon;
	} else {
		# assume it's a stock ID:
		$pbf = $self->widget->render_icon($icon, PerlPanel::menu_icon_size_name);
	}
	if (ref($pbf) ne 'Gtk2::Gdk::Pixbuf') {
		$pbf = Gtk2::Gdk::Pixbuf->new('rgb', 1, 8, PerlPanel::menu_icon_size, PerlPanel::menu_icon_size);
	}
	my $x0 = $pbf->get_width;
	my $y0 = $pbf->get_height;
	if ($x0 > PerlPanel::menu_icon_size || $y0 > PerlPanel::menu_icon_size) {
		my ($x1, $y1);
		if ($x0 > $y0) {
			# image is landscape:
			$x1 = PerlPanel::menu_icon_size;
			$y1 = int(($y0 / $x0) * PerlPanel::menu_icon_size);
		} else {
			# image is portrait:
			$x1 = int(($x0 / $y0) * PerlPanel::menu_icon_size);
			$y1 = PerlPanel::menu_icon_size;
		}
		$pbf = $pbf->scale_simple($x1, $y1, 'bilinear');
	}
	$item->set_image(Gtk2::Image->new_from_pixbuf($pbf));
	if (defined($callback)) {
		$item->signal_connect('activate', $callback);
	}
	return $item;
}

sub popup {
	my $self = shift;
	$self->menu->show_all;
	$self->menu->popup(undef, undef, sub { return $self->popup_position(@_) }, undef, undef, 0);
	return 1;
}

sub popup_position {
	my $self = shift;
	my ($x, undef) = PerlPanel::get_widget_position($self->widget);
	$x = 0 if ($x < 5);
	if (PerlPanel::position eq 'top') {
		return ($x, PerlPanel::panel->allocation->height);
	} else {
		$self->menu->realize;
		return ($x, PerlPanel::screen_height() - $self->menu->allocation->height - PerlPanel::panel->allocation->height);
	}
}

=pod

	my $icon = $self->get_icon($string, $is_submenu_parent);

This returns a scalar containing either a filename for an icon, or a stock ID.
This method is best used when used as the C<$icon> argument to the
C<menu_item()> method above.

C<menu_item()> searches a series of directories looking for an appropriate icon.
These directories are listed in @PerlPanel::ICON_DIRECTORIES.

=cut

sub get_icon {
	my ($self, $executable, $is_submenu_parent) = @_;

	$executable =~ s/\s/-/g if ($is_submenu_parent == 1);

	my $file = $self->detect_icon($executable);

	if (-e $file) {
		return $file;

	} else {
		return ($is_submenu_parent == 1 ? 'gtk-open' : 'gtk-execute');

	}
}

sub detect_icon {
	my ($self, $executable) = @_;

	return undef if ($executable eq '');
	my $program = lc(basename($executable));
	($program, undef) = split(/\s/, $program, 2);

	foreach my $dir (@ICON_DIRECTORIES) {
		my $file = sprintf('%s/%s.png', sprintf($dir, $PerlPanel::PREFIX), $program);
		if (-e $file) {
			return $file;
		}
	}

	return undef;
}

sub add_applet_dialog {
	my ($self, $applet) = @_;
	# place the new applet next to the menu:
	my $idx = 0;
	foreach my $applet ($PerlPanel::OBJECT_REF->{hbox}->get_children) {
		last if ($applet eq $self->widget);
		$idx++;
	}
	if ($idx >= 0) {
		splice(@{$PerlPanel::OBJECT_REF->{config}{applets}}, $idx+1, 0, $applet);
		PerlPanel::reload;
	}
	return 1;
}

=pod

=head1 SEE ALSO

=over

=item * L<perlpanel>

=item * L<perlpanel-applet-howto>

=item * L<Gtk2>

=back

=cut

1;
