/* Copyright © 2007, 2008 Jakub Wilk
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 */

#ifndef PDF2DJVU_COMPOPPLER_H
#define PDF2DJVU_COMPOPPLER_H

#include <string>
#include <ostream>
#include <vector>
#include <stdexcept>
#include <stdint.h>

#include <PDFDoc.h>
#include <GfxState.h>
#include <SplashOutputDev.h>
#include <Link.h>

#include <splash/Splash.h>
#include <splash/SplashBitmap.h>
#include <splash/SplashFont.h>
#include <splash/SplashGlyphBitmap.h>
#include <splash/SplashPath.h>

namespace pdf 
{

/* type definitions — splash output device
 * =======================================
 */

  namespace splash
  {
    typedef ::Splash Splash;
    typedef ::SplashColor Color;
    typedef ::SplashFont Font;
    typedef ::SplashCoord Coord;
    typedef ::SplashPath Path;
    typedef ::SplashGlyphBitmap GlyphBitmap;
    typedef ::SplashBitmap Bitmap;
    typedef ::SplashOutputDev OutputDevice;
  #if POPPLER_VERSION >= 602
    typedef ::SplashClipResult ClipResult;
  #endif
  }

/* miscellaneous type definitions
 * ==============================
 */

  typedef ::Stream Stream;
  typedef ::Object Object;
  typedef ::Dict Dict;
  typedef ::Catalog Catalog;
  typedef ::GooString String;

/* type definitions — annotations
 * ==============================
 */

  namespace ant
  {
    typedef ::Annot Annotation;
#if POPPLER_VERSION >= 700
    typedef ::AnnotColor Color;
#endif
  }

/* type definitions — hyperlinks
 * =============================
 */

  namespace link
  {
    typedef ::Link Link;
    typedef ::LinkAction Action;
    typedef ::LinkDest Destination;
    typedef ::LinkGoTo GoTo;
    typedef ::LinkURI URI;
#if POPPLER_VERSION < 509
    typedef ::LinkBorderStyle BorderStyle; 
#endif
  }

/* type definitions — rendering subsystem
 * ======================================
 */

  namespace gfx
  {
    typedef ::GfxSubpath Subpath;
    typedef ::GfxPath Path;
    typedef ::GfxState State;
    typedef ::GfxImageColorMap ImageColorMap;
#if POPPLER_VERSION >= 500    
    typedef ::GfxColorComp ColorComponent;
    typedef ::GfxColor Color;
    typedef ::GfxRGB RgbColor;
    typedef ::GfxDeviceCMYKColorSpace DeviceCmykColorSpace;
#endif
  }

/* class pdf::Renderer : pdf::splash::OutputDevice
 * ===============================================
 */

  class Renderer : public pdf::splash::OutputDevice
  {
  public:
    Renderer(pdf::splash::Color &paper_color, bool monochrome = false) :
#if POPPLER_VERSION < 500
      pdf::splash::OutputDevice(monochrome ? splashModeMono1 : splashModeRGB8Packed, gFalse, paper_color)
#else
      pdf::splash::OutputDevice(monochrome ? splashModeMono1 : splashModeRGB8, 4, gFalse, paper_color)
#endif
    { }

#if POPPLER_VERSION < 500
    void drawChar(gfx::State *state, double x, double y, double dx, double dy, double origin_x, double origin_y,
      CharCode code, Unicode *unistr, int len)
    {
      this->drawChar(state, x, y, dx, dy, origin_x, origin_y, code, -1, unistr, len);
    }

    virtual void drawChar(gfx::State *state, double x, double y, double dx, double dy, double origin_x, double origin_y,
      CharCode code, int n_bytes, Unicode *unistr, int len)
    {
      this->pdf::splash::OutputDevice::drawChar(state, x, y, dx, dy, origin_x, origin_y, code, unistr, len);
    }

    virtual void drawMaskedImage(gfx::State *state, Object *object, Stream *stream, int width, int height,
      gfx::ImageColorMap *color_map, Stream *mask_stream, int mask_width, int mask_height, GBool mask_invert) {}
    virtual void drawSoftMaskedImage(gfx::State *state, Object *object, Stream *stream,
      int width, int height, gfx::ImageColorMap *color_map, Stream *mask_stream,
      int mask_width, int mask_height,	gfx::ImageColorMap *mask_color_map) {}

    pdf::splash::Font *getCurrentFont()
    {
      return NULL;
    }
#endif  

#if POPPLER_VERSION >= 509
    void processLink(pdf::link::Link *link, pdf::Catalog *catalog)
    {
      this->drawLink(link, catalog);
    }
#endif
    virtual void drawLink(pdf::link::Link *link, pdf::Catalog *catalog);
    virtual void drawLink(pdf::link::Link *link, const std::string &border_color, pdf::Catalog *catalog)  { }
    std::vector<std::string> link_border_colors;
  protected:
    static void convert_path(gfx::State *state, pdf::splash::Path &splash_path);
  };


/* class pdf::Pixmap::iterator
 * ===========================
 */

  class PixmapIterator
  {
  private:
    const uint8_t *row_ptr;
    const uint8_t *ptr;
    size_t row_size;
  public:  
    PixmapIterator(const uint8_t *raw_data, size_t row_size)
    {
      this->row_ptr = this->ptr = raw_data;
      this->row_size = row_size;
    }

    PixmapIterator &operator ++(int)
    {
      ptr += 3;
      return *this;
    }

    void next_row()
    {
      ptr = row_ptr = row_ptr + row_size;
    }

    uint8_t operator[](int n) const
    {
      return this->ptr[n];
    }
  };


/* class pdf::Pixmap
 * =================
 */

  class Pixmap
  {
  private:
    Pixmap(const Pixmap&); // not defined
    Pixmap& operator=(const Pixmap&); // not defined
    const uint8_t *raw_data;
    pdf::splash::Bitmap *bmp;
    size_t row_size;
    size_t byte_width;
    bool monochrome;
    int width, height;
  public:
    typedef PixmapIterator iterator;

    int get_width() const 
    {
      return width;
    }
    int get_height() const
    {
      return height;
    }

    explicit Pixmap(Renderer *renderer)
    {
#if POPPLER_VERSION < 500
      bmp = renderer->getBitmap();
      raw_data = const_cast<const uint8_t*>(bmp->getDataPtr().rgb8p);
#else
      bmp = renderer->takeBitmap();
      raw_data = const_cast<const uint8_t*>(bmp->getDataPtr());
#endif
      width = bmp->getWidth();
      height = bmp->getHeight();
      row_size = bmp->getRowSize();
      this->monochrome = false;
      switch (bmp->getMode())
      {
      case splashModeMono1:
        this->byte_width = (width + 7) / 8;
        this->monochrome = true;
        break;
      case splashModeMono8:
        this->byte_width = width;
        break;
      case splashModeRGB8:
#if POPPLER_VERSION >= 500
      case splashModeBGR8:
#else
      case splashModeRGB8Packed:
      case splashModeBGR8Packed:
#endif
        this->byte_width = width * 3;
        break;
#if POPPLER_VERSION >= 500
      case splashModeXBGR8:
        this->byte_width = width * 4;
        break;
#endif
      }
    }

    ~Pixmap()
    {
#if POPPLER_VERSION >= 500
      delete bmp;
#endif
    }

    PixmapIterator begin() const
    {
      return PixmapIterator(raw_data, row_size);
    }

    friend std::ostream &operator<<(std::ostream &, const Pixmap &);
  };


/* class pdf::OwnedObject : pdf::Object
 * ====================================
 */

  class OwnedObject : public Object
  {
  public:
    ~OwnedObject()
    {
      this->free();
    } 
  };


/* class pdf::NFKC
 * ===============
 */

  class NFKC
  {
  protected:
    Unicode* data; 
    int _length;
  public:
    explicit NFKC(Unicode *, int length);
    ~NFKC();
    size_t length() const
    {
      return static_cast<size_t>(this->_length);
    }
    operator const Unicode*() const
    {
      return this->data;
    }
  };


/* class pdf::Environment
 * ======================
 */

  class Environment
  {
  public:
    Environment();
    void set_antialias(bool value);
    class UnableToSetParameter : public std::runtime_error
    {
    public:
      UnableToSetParameter(const std::string &message)
      : std::runtime_error(message)
      { }
    };
  };


/* class pdf::Document
 * ===================
 */

  class Document : public ::PDFDoc
  {
  public:
    Document(const std::string &file_name);
    void display_page(Renderer *renderer, int npage, double hdpi, double vdpi, bool crop, bool do_links);
    void get_page_size(int n, bool crop, double &width, double &height);
    class LoadError : public std::runtime_error
    {
    public:
      LoadError()
      : std::runtime_error("Unable to load document")
      { }
    };
  };


/* utility functions
 * =================
 */

  void set_color(pdf::splash::Color &result, uint8_t r, uint8_t g, uint8_t b);

#if POPPLER_VERSION >= 500
  namespace gfx
  {
    static inline double color_component_as_double(pdf::gfx::ColorComponent c)
    {
      return ::colToDbl(c);
    }

    static inline pdf::gfx::ColorComponent double_as_color_component(double x)
    {
      return ::dblToCol(x);
    }
  }
#endif

/* glyph-related functions
 * =======================
 */

  bool get_glyph(pdf::splash::Splash *splash, pdf::splash::Font *font, 
    double x, double y, // x, y are transformed (i.e. output device) coordinates
    int code, pdf::splash::GlyphBitmap *bitmap);

/* dictionary lookup
 * =================
 */

  pdf::Object *dict_lookup(pdf::Object &dict, const char *key, pdf::Object *object);
  pdf::Object *dict_lookup(pdf::Object *dict, const char *key, pdf::Object *object);
  pdf::Object *dict_lookup(pdf::Dict *dict, const char *key, pdf::Object *object);

/* path-related functions
 * ======================
 */

  double get_path_area(pdf::splash::Path &path);

/* Unicode → UTF-8 conversion
 * ==========================
 */

  void write_as_utf8(std::ostream &stream, Unicode unicode_char);
  void write_as_utf8(std::ostream &stream, char pdf_char);
  void write_as_utf8(std::ostream &stream, const char *pdf_chars);
}

#endif

// vim:ts=2 sw=2 et
