// Package tmp provides temporary directory helpers.
//
// tmp stores temporary items in the system's
// temporary directory unless a corresponding
// environment variable is set ( see os.TempDir ).
//
// On Unix systems, it uses $TMPDIR if non-empty, else /tmp.
// On Windows, it uses GetTempPath, returning the first non-empty
// value from %TMP%, %TEMP%, %USERPROFILE%, or the Windows directory.
// On Plan 9, it returns /tmp.
//
// The directory is neither guaranteed to exist nor have accessible
// permissions.
package tmp

import (
	"io/ioutil"
	"os"
)

var tmpDir = os.TempDir()

// Dir creates a new temporary directory in the system temporary
// directory with a name beginning with prefix and returns the path
// of the new directory.
// Multiple programs calling Dir simultaneously
// will not choose the same directory.
// It is the caller's responsibility
// to remove the file when no longer needed.
func Dir(prefix string) (string, error) {
	return ioutil.TempDir(tmpDir, prefix)
}

// File creates a new temporary file in the system temporary
// directory, opens the file for reading and writing, and
// returns the resulting *os.File.
// The filename is generated by taking pattern and adding a random
// string to the end. If pattern includes a "*", the random string
// replaces the last "*".
// Multiple programs calling File simultaneously
// will not choose the same file. The caller can use f.Name()
// to find the pathname of the file. It is the caller's responsibility
// to remove the file when no longer needed.
func File(pattern string) (*os.File, error) {
	return ioutil.TempFile(tmpDir, pattern)
}
