#include <cassert>

#include <qmessagebox.h>
#include <qaction.h>
#include <q3popupmenu.h>
#include <QMainWindow>
#include <QMutex>

#include <unistd.h>		// for getuid and geteuid
#include <sys/types.h>	//

#include <tagcoll/Expression.h>
#include <tagcoll/Filter.h>
#include <tagcoll/StdioParserInput.h>
#include <tagcoll/Exception.h>

#include <apt-front/cache/cache.h>
#include <apt-front/cache/component/packagetags.h>
#include <apt-front/cache/component/tags.h>

#include "debtagsplugincontainer.h"

// NUtil
#include "helpers.h"

// NApplication
#include "applicationfactory.h"
#include "runcommand.h"

// NPlugin
#include <iprovider.h>
#include <plugincontainer.h>
#include <iprogressobserver.h>
#include "debtagsplugin.h"
#include "relatedplugin.h"
#include "debtagsactionplugin.h"
#include "debtagspluginfactory.h"

#include "debtagssettingswidget.h"

using namespace aptFront::cache::component;
using namespace aptFront::cache::entity;

extern "C" 
{ 
	NPlugin::PluginContainer* new_debtagsplugin() 
	{
		return new NPlugin::DebtagsPluginContainer;
	} 

	NPlugin::PluginInformation get_pluginInformation()
	{
		return NPlugin::PluginInformation("debtagsplugin", "2.1", "Benjamin Mesing");
	} 
}

/** Initialize the plugin. */
__attribute__ ((constructor)) void init() 
{
}
      
      
// __attribute__ ((destructor)) void fini() 
// {
//   /* code here is executed just before dlclose() unloads the module */
// } 


namespace NPlugin
{

DebtagsPluginContainer::DebtagsPluginContainer()
{
	// this assumes, that only one instance of the DebtagsPluginContainer is created
	// (no two versions of libapt-front may be opened at any time)
	// This is ok for our purpose - though usually there should be a singleton ensuring
	// this constraint...
	DebtagsPluginFactory::getInstance()->setContainer(this);
	_pCommand = 0;
	_pRelatedPlugin = 0;
	_pDebtagsPlugin = 0;
	_pDebtagsActionPlugin = 0;
	_pSettingsWidget = 0;
	addPlugin("DebtagsPlugin");
	addPlugin("RelatedPlugin");
	addPlugin("DebtagsActionPlugin");
	
	_debtagsEnabled=false;
	
}
 
DebtagsPluginContainer::~DebtagsPluginContainer()
{
	delete _pCommand;
//	qDebug("Closing global aptFront Cache");
//	cache::Global::get().close();
}

/////////////////////////////////////////////////////
// PluginContainer Interface
/////////////////////////////////////////////////////

bool DebtagsPluginContainer::init(IProvider* pProvider)
{
	BasePluginContainer::init(pProvider, DebtagsPluginFactory::getInstance());
	updateDebtags();
	if (debtagsEnabled())
	{
		// use dynamic cast here because of the virtual base class 
		// (static_cast is not allowed there)
		_pRelatedPlugin = dynamic_cast<RelatedPlugin*>(requestPlugin("RelatedPlugin"));
		_pDebtagsPlugin = dynamic_cast<DebtagsPlugin*>(requestPlugin("DebtagsPlugin"));
		_pDebtagsActionPlugin = dynamic_cast<DebtagsActionPlugin*>(requestPlugin("DebtagsActionPlugin"));
		
		connect( _pDebtagsActionPlugin->qDebtagsUpdateAction(), SIGNAL(triggered(bool)), SLOT(onDebtagsUpdate()) );
	}
	return debtagsEnabled();
}

vector< pair<QString, QAction*> > DebtagsPluginContainer::actions()
{
	vector< pair<QString, QAction*> > result;
	return result;
}

QWidget* DebtagsPluginContainer::getSettingsWidget(QWidget* pParent)
{
	_pSettingsWidget = new DebtagsSettingsWidget(_hiddenFacets, pParent, "DebtagsSettingsWidget");
 	return _pSettingsWidget;
}

void DebtagsPluginContainer::applySettings()
{
	assert(_pSettingsWidget);
	_hiddenFacets = _pSettingsWidget->hiddenFacets();
	updateVocabulary();
}



/////////////////////////////////////////////////////
// BasePluginContainer Interface
/////////////////////////////////////////////////////


QDomElement DebtagsPluginContainer::loadContainerSettings(const QDomElement source)
{
	if (source.tagName() != "ContainerSettings")
		return source;
	float settingsVersion;
	NXml::getAttribute(source, settingsVersion, "settingsVersion", 0.0f);
	
	QDomNodeList hiddenFacets = source.elementsByTagName("HiddenFacet");
	for (int i=0; i < hiddenFacets.count(); ++i)
	{
		string hiddenFacet = toString( hiddenFacets.item(i).toElement().text() );
		_hiddenFacets.insert(hiddenFacet);
	}
	if (debtagsEnabled())
		updateVocabulary();
	return NXml::getNextElement(source);
}

void DebtagsPluginContainer::saveContainerSettings(NXml::XmlData& outData, QDomElement parent) const
{
	QDomElement containerElement = outData.addElement(parent, "ContainerSettings");
	outData.addAttribute(containerElement, 0.1f, "settingsVersion");
	for (set<string>::const_iterator it = _hiddenFacets.begin(); it != _hiddenFacets.end(); ++it)
	{
		QDomElement hiddenFacetElement = outData.addElement(containerElement, "HiddenFacet");
		outData.addText(hiddenFacetElement, *it);
	}
}





/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPluginContainer::onDebtagsUpdate()
{
	_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(false);
	// this will fetch us the update of the db
	NApplication::ApplicationFactory fac;
	_pCommand = fac.getRunCommand("DebtagsUpdateProcess");
	connect(_pCommand, SIGNAL(quit()), SLOT(onDebtagsUpdateFinished()) );
	_pCommand->addArgument("/usr/bin/debtags");
	_pCommand->addArgument("update");
	
	try 
	{
		if ( !_pCommand->startAsRoot() )
		{
			provider()->reportError( tr("Command not executed"), tr("For an unknwon reason, the command could "
				"not be executed.") );
			delete _pCommand;
			_pCommand = 0;
			_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(true);
		}
	}
	catch (const NException::RuntimeException& e)
	{
		provider()->reportError(tr("Command not executed"), toQString(e.description()));
		delete _pCommand;
		_pCommand = 0;
		_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(true);
	}
}

void DebtagsPluginContainer::onDebtagsUpdateFinished()
{
	if (_pCommand->processExitedSuccessful())
	{
		provider()->reloadAptFrontCache();
		updateDebtags();
	}
	delete _pCommand;
	_pCommand = 0;
	_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(true);
}


void DebtagsPluginContainer::updateVocabulary(bool informPlugins)
{
	if (informPlugins)
	{
		if (_pDebtagsPlugin)
			_pDebtagsPlugin->debtagsDataChanged();
		if (_pRelatedPlugin)
			_pRelatedPlugin->debtagsDataChanged();
	}
}

void DebtagsPluginContainer::updateDebtags()
{
	NUtil::IProgressObserver* pProgressObserver = provider()->progressObserver();
	if (pProgressObserver)
		pProgressObserver->setText("Loading Debtags Plugin");
/*	try 
	{
		// this is set to 0 if no listener is available
		cache::Global::get().close();
		qDebug("Opening aptfront cache for debtags");
		cache::Global::get().open(cache::Cache::OpenDefault
				| cache::Cache::OpenReadOnly
				| cache::Cache::OpenTags
				| cache::Cache::OpenDebtags);
	}
	catch (Tagcoll::SystemException e)*/
	
	if (!cache::component::PackageTags::hasTagDatabase())
	{
		setDebtagsEnabled(false);	// disable the debtags system
		provider()->reportError(
			tr("Tag Database Not Available" ),
			tr(
				"<p>The tag database is not available and the debtags plugin was disabled!</p>"
				"<p>"
				"You must execute <tt><b>debtags update</b></tt> as root on the commandline to download "
				"the database. If debtags is not on your system you can install it via "
				"<tt><b>apt-get install debtags</b></tt><br>"
				"Afterwards you can enable the debtags plugin via the plugin menu -> Control Plugins."
				"</p>")
/*				"<p>"
				"The original error message reported by debtags was:<br>"
				"<tt>"
			) +
			toQString(e.desc()) +
			"</tt>"
			"</p>"*/
		);
		return;
	}
/*	catch (exception::Error e)
	{
		setDebtagsEnabled(false);	// disable the debtags system
		provider()->reportError(
			tr("Unknown Error when reading the Debtags database" ),
			tr("An unknown error occured when reading the debtags database.<br>"
				"The error reported by debtags is:.<br><tt>") +
			toQString(e.what()) +
			"</tt>"
		);
		return;
	}*/

	setDebtagsEnabled(true);
	if (debtagsEnabled())
	{
		// load the new vocabulary
		updateVocabulary(false);
	}
	// inform the plugins about the new collections
	if (_pDebtagsPlugin)
		_pDebtagsPlugin->debtagsDataChanged();
	if (_pRelatedPlugin)
		_pRelatedPlugin->debtagsDataChanged();
	if (pProgressObserver)
		pProgressObserver->setProgress(100);
}

void DebtagsPluginContainer::setDebtagsEnabled(bool enabled)
{
	_debtagsEnabled = enabled;
}

Tagcoll::Collection<cache::entity::Package, cache::entity::Tag>* DebtagsPluginContainer::collection() const	
{
	return &(provider()->aptFrontCache().debtags().tagdb());
}

const Tagcoll::OpSet<cache::entity::Facet> DebtagsPluginContainer::facets() const
{
	Tags& tags = provider()->aptFrontCache().tags();
	Tagcoll::OpSet<Facet> hidden;
	for (set<string>::const_iterator i = _hiddenFacets.begin();
		i != _hiddenFacets.end(); i++)
	{
		Facet f = tags.facetByName(*i);
		if (f.valid())
			hidden += f;
	}
	return provider()->aptFrontCache().tags().facets() - hidden; 
}




}	// namespace NPlugin

