#include <sstream>
#include <iostream>

#include <qcheckbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <q3listbox.h>
#include <QMainWindow>
#include <qmessagebox.h>
#include <qpoint.h>
#include <q3popupmenu.h>
#include <QVBoxLayout>

// Tagcoll
#include <tagcoll/Expression.h>
#include <tagcoll/Filter.h>
#include <tagcoll/Exception.h>


#include <helpers.h>
#include <exception.h>

#include "debtagsplugin.h"

// NPlugin
#include "iprovider.h"

// NUtil
#include "debtagshelper.h"


#include "selectioninputanddisplay.h"
#include "tagselectionwidget.h"
#include "choosentagsdisplay.h"



namespace NPlugin
{

const QString DebtagsPlugin::PLUGIN_NAME = "DebtagsPlugin";


/////////////////////////////////////////////////////
// Constructors/ Destructors
/////////////////////////////////////////////////////

DebtagsPlugin::DebtagsPlugin(const DebtagsPluginContainer& container) :
	_container(container)
{ 
	_pChooserWidget = 0;
	_pTagsDisplay = 0;
	_pProvider = 0;
	_pIncludeSelection = 0;
	_pExcludeSelection = 0;
	_isInactive = true;
}

DebtagsPlugin::~DebtagsPlugin() 
{ 
	delete _pChooserWidget;
	delete _pTagsDisplay;
	delete _pIncludeSelection;
	delete _pExcludeSelection;
}

/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////

void DebtagsPlugin::init(IProvider* pProvider)
{
	QMainWindow* pWindow = pProvider->mainWindow();
	_pProvider = pProvider;

	_pChooserWidget = new TagChooserWidget(pWindow, "TagChooserWidget");
	_pTagsDisplay = new ChoosenTagsDisplay(pWindow, "ChoosenTagsDisplay");
	
	// this will be garbage collected if the parent _pChooserWidget will be deleted
	NWidgets::TagSelectionWidget* pIncludeSelectionWidget = 
		new NWidgets::TagSelectionWidget(_pChooserWidget, &_container, "IncludeSelection");
	NWidgets::TagSelectionWidget* pExcludeSelectionWidget = 
		new NWidgets::TagSelectionWidget(_pChooserWidget, &_container, "ExcludeSelection");
	
	_pTagsDisplay->show();
	// currently disabled as the exclude tag option is not too useful and wastes some space
	_pChooserWidget->_pExcludeTagsCheck->setShown(false);
	// assume that the vocabulary is not accessible if one of the pointers is not set
	bool debtagsEnabled = (_container.collection() != 0);
	_pIncludeSelection = new NWidgets::SelectionInputAndDisplay
	(
		pWindow,
		"IncludeSelection",
		pIncludeSelectionWidget,
		_pTagsDisplay->_pIncludeTagsView,
		_pTagsDisplay->_pIncludeViewLabel
	);
	_pIncludeSelection->loadVocabulary(_container.facets());
	_pExcludeSelection = new NWidgets::SelectionInputAndDisplay
	(
		pWindow,
		"ExcludeSelection",
		pExcludeSelectionWidget,
		_pTagsDisplay->_pExcludeTagsView,
		_pTagsDisplay->_pExcludeViewLabel
	);
	_pExcludeSelection->loadVocabulary(_container.facets());
	static_cast<QVBoxLayout*>
		(_pChooserWidget->layout())->insertWidget(1, _pIncludeSelection->tagSelectionWidget());
	static_cast<QVBoxLayout*>
		(_pChooserWidget->layout())->insertWidget(4, _pExcludeSelection->tagSelectionWidget());
	connect(
		_pIncludeSelection->tagSelectionListView(), SIGNAL(tagItemsSelected(const set<TagItem*>&)),
		this, SLOT(evaluateSearch()) 
	);
	connect(
		_pExcludeSelection->tagSelectionListView(), SIGNAL(tagItemsSelected(const set<TagItem*>&)),
		this, SLOT(evaluateSearch()) 
	);
	

	showExcludeWidgets(false);	// hide the exclude selection part by default
	connect(_pChooserWidget->_pExcludeTagsCheck, SIGNAL(toggled(bool)), this, SLOT(showExcludeWidgets(bool)));
	
	if ( _container.collection()==0 )
		setWidgetsEnabled(false);
}


QString DebtagsPlugin::title() const 
{ 
	return QString("Debtags Plugin"); 
}

QString DebtagsPlugin::briefDescription() const 
{ 
	return QString("Offers information and search using the debtags system"); 
}

QString DebtagsPlugin::description() const 
{ 
	return QString("This plugin shows the tags for a program in the detailed view.\n"
		"It also offers searching by tags that can be selected from a list."); 
}

/////////////////////////////////////////////////////
// Search Plugin Interface 
/////////////////////////////////////////////////////

QWidget* DebtagsPlugin::shortInputAndFeedbackWidget() const
{
	return _pTagsDisplay;
}

const Tagcoll::OpSet<string>& DebtagsPlugin::searchResult() const
{ 
	return _searchResult; 
};

void DebtagsPlugin::clearSearch()
{
	_pIncludeSelection->tagSelectionListView()->deselectAll();
	_pExcludeSelection->tagSelectionListView()->deselectAll();
}

/////////////////////////////////////////////////////
// Information Plugin Interface 
/////////////////////////////////////////////////////

QString DebtagsPlugin::informationText(const string& package)
{
	{	// add the tags of the package to the description
		entity::Package packageEntity= NUtil::getPackageByName(package);
		Tagcoll::OpSet<string> tagset = NUtil::tagsToStrings(_container.collection()->getTags(packageEntity));
		if (tagset.empty()) return _emptyString;
		QString detailsString = "<b>Tags:</b> ";
		for (Tagcoll::OpSet<string>::iterator it = tagset.begin(); ; )
		{
			detailsString += toQString( *it );
			if ( ++it == tagset.end() )
			{
				detailsString.append("\n");
				break;
			}
			detailsString += ", ";
		}
		return detailsString;
	}
}


/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPlugin::showExcludeWidgets( bool display )
{
	_pExcludeSelection->setShown(display);
	_pChooserWidget->_pExcludeInputLabel->setShown(display);
}


/** This creates a search expression which can be used as input for an ExpressionFiler.
  * @returns the filter string
  */
std::string DebtagsPlugin::createSearchExpression()
{
	ostringstream oexpr;
	// the list view that displays the selected tags
	Q3ListBox* pTagsView = _pTagsDisplay->_pIncludeTagsView;
	for (uint i = 0; i < pTagsView->count(); ++i)
	{
		oexpr << toString(pTagsView->text(i));
		// if we do not have the last item or we have tags to exclude
		if ( i+1< pTagsView->count() || _pTagsDisplay->_pExcludeTagsView->count()!=0)	
			oexpr << " && ";
	}
	// the list view that displays the selected tags
	pTagsView = _pTagsDisplay->_pExcludeTagsView;
	for (uint i = 0; i < pTagsView->count(); ++i)
	{
		oexpr << "!" << toString(pTagsView->text(i));
		if ( i+1<pTagsView->count())	// if we do not have the last item 
			oexpr << " && ";
	}
	return oexpr.str();	
}


void DebtagsPlugin::evaluateSearch()
{
	_pProvider->reportBusy(this, tr("Performing full text search on Package Database"));
//	statusBar()->message(tr("Searching Package Database for tags"));
	_searchResult.clear();
	string expression = createSearchExpression();
	if (expression.empty())	// if nothing is selected, do not show anything
	{
		_isInactive = true;
	}
	else
	{
		_isInactive = false;
		if (_pTagsDisplay->_pExcludeTagsView->count()!=0)	// if we want to exclude some tags
		{
			// send the package collection through the filter into the result
			// and than copy the result to the _searchResult
			Tagcoll::FilterChain<int, string> filters;
			#warning this must be fixed!!
			#warning this must be fixed!!
			#warning this must be fixed!!
			/// @todo fix the Expression Filter (hopefully Enrico will make it a template)
/*			Tagcoll::ExpressionFilter filter;
			if ( !filter.setExpression(expression) )	// no error should occur!
				cerr << "Programmers error please report to bensmail@gmx.net: Error in tag expression "
					<< endl;
			filters.appendFilter(&filter);
			Tagcoll::TagCollection result;
			filters.setConsumer(&result);
			_container.collection()->output(filters);
			_searchResult = result.getAllItems();*/
		}
		else	// do not use TagcollFilter
		{
			_searchResult = NUtil::packagesToStrings( _container.collection()->getItems(
				_pIncludeSelection->tagSelectionListView()->getSelectedTags()) );
		}
	}
	_pProvider->reportReady(this);
	emit searchChanged(this);
}


void DebtagsPlugin::debtagsDataChanged()
{
	clearSearch();
	if (_container.collection()==0)
		setWidgetsEnabled(false);
	else
	{
		_pIncludeSelection->loadVocabulary(_container.facets());
		_pExcludeSelection->loadVocabulary(_container.facets());
		setWidgetsEnabled(true);
	}
//	clearSearch();
}

void DebtagsPlugin::setWidgetsEnabled(bool enabled)
{
	_pChooserWidget->setEnabled(enabled);
	_pTagsDisplay->setEnabled(enabled);
}



}	// namespace NPlugin

