/*
 * libexplain - Explain errno values returned by libc functions
 * Copyright (C) 2009 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful,but
 * WITHOUT ANY WARRANTY; without even the implied warranty
 * ofMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNULesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <libexplain/ac/linux/ata.h>
#include <libexplain/ac/linux/hdreg.h>
#include <libexplain/ac/sys/ioctl.h>

#include <libexplain/iocontrol/hdio_getgeo.h>
#include <libexplain/iocontrol/hdio_get_identity.h>
#include <libexplain/string_buffer.h>

#include <ioctl-scan/report.h>
#include <ioctl-scan/scan/linux_hdreg.h>


#ifdef HDIO_GETGEO

static void
hdio_getgeo(int fildes)
{
    struct hd_geometry data;

    if (ioctl(fildes, HDIO_GETGEO, &data) >= 0)
    {
        explain_string_buffer_t sb;
        char            text[1000];

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_iocontrol_hdio_getgeo.print_data_returned
        (
            0,
            &sb,
            0,
            fildes,
            HDIO_GETGEO,
            &data
        );
        report("HDIO_GETGEO", "Disk Geometry", text);
    }
}

#endif

#ifdef HDIO_GET_32BIT

static void
hdio_get_32bit(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_32BIT, &data) >= 0)
    {
        report_long("HDIO_GET_32BIT", "hdio_get_32bit", data, 0);
    }
}

#endif

#ifdef HDIO_GET_ACOUSTIC

static void
hdio_get_acoustic(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_ACOUSTIC, &data) >= 0)
    {
        report_long("HDIO_GET_ACOUSTIC", "hdio_get_acoustic", data, 0);
    }
}

#endif

#ifdef HDIO_GET_ADDRESS

static void
hdio_get_address(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_ADDRESS, &data) >= 0)
    {
        report_long("HDIO_GET_ADDRESS", "hdio_get_address", data, 0);
    }
}

#endif

#ifdef HDIO_GET_BUSSTATE

static void
hdio_get_busstate(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_BUSSTATE, &data) >= 0)
    {
        report_long("HDIO_GET_BUSSTATE", "hdio_get_busstate", data, 0);
    }
}

#endif

#ifdef HDIO_GET_DMA

static void
hdio_get_dma(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_DMA, &data) >= 0)
    {
        report_long("HDIO_GET_DMA", "hdio_get_dma", data, 0);
    }
}

#endif

#ifdef HDIO_GET_IDENTITY

static void
hdio_get_identity(int fildes)
{
#ifdef HAVE_LINUX_ATA_H
    short           data[ATA_ID_WORDS];

    if (ioctl(fildes, HDIO_GET_IDENTITY, data) >= 0)
    {
        explain_string_buffer_t sb;
        char            text[1000];

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_iocontrol_hdio_get_identity.print_data_returned
        (
            0,
            &sb,
            0,
            fildes,
            HDIO_GET_IDENTITY,
            data
        );
        report("HDIO_GET_IDENTITY", "hdio_get_identity", text);
    }
#else
    (void)fildes;
#endif
}

#endif

#ifdef HDIO_GET_MULTCOUNT

static void
hdio_get_multcount(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_MULTCOUNT, &data) >= 0)
    {
        report_long("HDIO_GET_MULTCOUNT", "hdio_get_multcount", data, 0);
    }
}

#endif

#ifdef HDIO_GET_NICE

static void
hdio_get_nice(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_NICE, &data) >= 0)
    {
        report_long("HDIO_GET_NICE", "hdio_get_nice", data, 0);
    }
}

#endif

#ifdef HDIO_GET_NOWERR

static void
hdio_get_nowerr(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_NOWERR, &data) >= 0)
    {
        report_long("HDIO_GET_NOWERR", "hdio_get_nowerr", data, 0);
    }
}

#endif

#ifdef HDIO_GET_QDMA

static void
hdio_get_qdma(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_QDMA, &data) >= 0)
    {
        report_long("HDIO_GET_QDMA", "hdio_get_qdma", data, 0);
    }
}

#endif

#ifdef HDIO_GET_UNMASKINTR

static void
hdio_get_unmaskintr(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_UNMASKINTR, &data) >= 0)
    {
        report_long("HDIO_GET_UNMASKINTR", "hdio_get_unmaskintr", data, 0);
    }
}

#endif

#ifdef HDIO_GET_WCACHE

static void
hdio_get_wcache(int fildes)
{
    long            data;

    if (ioctl(fildes, HDIO_GET_WCACHE, &data) >= 0)
    {
        report_long("HDIO_GET_WCACHE", "hdio_get_wcache", data, 0);
    }
}

#endif


void
scan_linux_hdreg(int fildes)
{
#ifdef HDIO_GETGEO
    hdio_getgeo(fildes);
#endif
#ifdef HDIO_GET_32BIT
    hdio_get_32bit(fildes);
#endif
#ifdef HDIO_GET_ACOUSTIC
    hdio_get_acoustic(fildes);
#endif
#ifdef HDIO_GET_ADDRESS
    hdio_get_address(fildes);
#endif
#ifdef HDIO_GET_BUSSTATE
    hdio_get_busstate(fildes);
#endif
#ifdef HDIO_GET_DMA
    hdio_get_dma(fildes);
#endif
#ifdef HDIO_GET_IDENTITY
    hdio_get_identity(fildes);
#endif
#ifdef HDIO_GET_MULTCOUNT
    hdio_get_multcount(fildes);
#endif
#ifdef HDIO_GET_NICE
    hdio_get_nice(fildes);
#endif
#ifdef HDIO_GET_NOWERR
    hdio_get_nowerr(fildes);
#endif
#ifdef HDIO_GET_QDMA
    hdio_get_qdma(fildes);
#endif
#ifdef HDIO_GET_UNMASKINTR
    hdio_get_unmaskintr(fildes);
#endif
#ifdef HDIO_GET_WCACHE
    hdio_get_wcache(fildes);
#endif
}
