/*
 * $Id: bit_io.c,v 1.12 2008/03/25 19:31:44 simakov Exp $
 *
 * EPSILON - wavelet image compression library.
 * Copyright (C) 2006-2007 Alexander Simakov, <xander@entropyware.info>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://epsilon-project.sourceforge.net
 */

#include <common.h>
#include <bit_io.h>

void init_bits(bit_buffer *bb, unsigned char *buf, int size)
{
    assert(size > 0);

    bb->start = bb->next = buf;
    bb->end = bb->start + size;
    bb->bits = bb->pending = 0;
}

int write_bits(bit_buffer *bb, int value, int size)
{
    assert(size <= 24);

    /* No more space available */
    if (bb->next >= bb->end) {
        return BIT_BUFFER_OVERFLOW;
    }

    /* Save requested number of bits */
    bb->bits |= (value << bb->pending);
    bb->pending += size;

    /* Write complete octets */
    while (bb->pending >= 8) {
        if (bb->next >= bb->end) {
            return BIT_BUFFER_OVERFLOW;
        }

        *bb->next++ = (unsigned char) (bb->bits & 0xff);

        bb->bits >>= 8;
        bb->pending -= 8;
    }

    return BIT_BUFFER_OK;
}

int read_bits(bit_buffer *bb, int *value, int size)
{
    assert(size <= 24);

    /* Read missing bytes */
    while (bb->pending < size) {
        if (bb->next >= bb->end) {
            return BIT_BUFFER_UNDERFLOW;
        }

        bb->bits |= (*bb->next++ << bb->pending);
        bb->pending += 8;
    }

    *value = bb->bits & (~(~0 << size));

    bb->bits >>= size;
    bb->pending -= size;

    return BIT_BUFFER_OK;
}

int flush_bits(bit_buffer *bb)
{
    if (bb->pending) {
        return write_bits(bb, 0UL, 8);
    } else {
        return BIT_BUFFER_OK;
    }
}
