/***************************************************************************
 *                            VcdFilter.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements  the info function for a VCD-file to DFIO filter
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "VcdFilter.h"
#include "DFIOMgr.h"
#include "FileAccObj.h"
#include <memory.h>
#include <stdlib.h>
#include <ctype.h>
#include <tcl.h>

#undef  DEBUG_VCD_FILTER

#ifdef DEBUG_VCD_FILTER
#define DEBUG(x) x
#else
#define DEBUG(x)
#endif

static void quicksort(struct symbol **a, int p, int r);
static int hash(char *s);

/****************************************************************************
 * The code in this module was borrowed, nearly verbatim, from the file
 * vcd2lxt.c from the GTKwave distribution...
 *
 * Many thanks to Tony for distilling the intricacies of VCD ;-)
 ****************************************************************************/
/* 
 * Copyright (c) Tony Bybell 1999.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * vcd.c 			23jan99ajb
 * evcd parts 			29jun99ajb
 * profiler optimizations 	15jul99ajb
 * stripped out of gtkwave	21jul99ajb
 * fix for duplicate nets       19dec00ajb
 * lxt conversion added		20nov01ajb
 *
 */

#define SYMPRIME 65519

typedef struct Node	 *nptr;
typedef struct HistEnt	 *hptr;


typedef struct HistEnt {
    hptr next;			/* next transition in history */
    TimeType time;		/* time of transition */
    TimeType previous_width;	/* to avoid thrashing */

    union {
	unsigned char val;	/* value: "0XU1"[val] */
	char *vector;		/* pointer to a whole vector */
    } v;

} HistEnt;


typedef struct ExtNode {
    int msi, lsi;
} ExtNode;

struct Node {
    char     *nname;	/* ascii name of node */
    ExtNode  *ext;	/* extension to node for vectors */    
    HistEnt  head;	/* first entry in transition history */
    hptr     curr;      /* ptr. to current history entry */

    hptr     *harray;   /* fill this in when we make a trace.. contains  */
			/*  a ptr to an array of histents for bsearching */
    int      numhist;	/* number of elements in the harray */
    char     notdead;	/* indicates if this node gets a non-x value */
    int      numtrans;  /* number of transitions */

    struct Node *substnode;  /* pointer to substitutions on buses */
    struct Node *substhead;  /* pointer to substitution head (originator) on buses */
};

typedef struct symbol {
    struct symbol *nextinaet;	/* for aet node chaining */
    struct HistEnt *h;		/* points to previous one */

    struct symbol *next;	/* for hash chain */
    char *name;
    char selected;		/* for the clist object */

    struct Node *n;
} symbol;


struct slist {  
    struct slist *next;
    char *str;
    int len;
};


typedef struct vcdsymbol {
    struct vcdsymbol *next;
    DFIOTrace        *trace;
    char *name;
    char *id;
    int vartype;
    int msi, lsi;
    int size;
    char *value;
    struct queuedevent *ev; /* only if vartype==V_EVENT */
    struct Node **narray;
} vcdsymbol;


struct queuedevent {
    struct queuedevent *next;
    struct vcdsymbol *sym;
    TimeType last_event_time;    /* make +1 == 0 if there's not an event there too */
};


static char *tokens[] = {
    "var", "end", "scope", "upscope",
    "comment", "date", "dumpall", "dumpoff", "dumpon",
    "dumpvars", "enddefinitions",
    "dumpports", "dumpportsoff", "dumpportson", "dumpportsall",
    "timescale", "version",
    "", "", ""
};

static char *vartypes[]={
    "event", "parameter",
    "integer", "real", "reg", "supply0",
    "supply1", "time", "tri", "triand", "trior",
    "trireg", "tri0", "tri1", "wand", "wire", "wor", "port",
    "$end", "", "", "", ""
};

/******************************************************
 * Init()
 ******************************************************/
void VcdFilter::Init()
{
    numfacs = deadcnt = 0;
    facs = NULL;
    firstnode = curnode = NULL;
    min_time = max_time = -1;
    vcdlineno = 1;
    header_over = 0;
    dumping_off = 0;
    start_time  = -1;
    end_time    = -1;
    current_time = -1;
    time_scale   = 1;	/* multiplier is 1, 10, 100 */
    vcd_hier_delimeter[0] = '.';
    vcd_hier_delimeter[1] = 0;
    slistroot = slistcurr = NULL;
    slisthier = NULL;
    slisthier_len = 0;

    yylen         = 0;
    yylen_cache   = 0;
    vcdsymroot    = NULL;
    vcdsymcurr    = NULL;
    sorted        = NULL;
    numsyms       = 0;
    queuedevents  = NULL;
    d_keywords    = 0;
}

/******************************************************
 * VcdFilter()
 ******************************************************/
VcdFilter::VcdFilter(Tcl_Interp  *interp) :
    DFIOFilter(interp, "DFIOFilter::VcdFilter", "vcd", 
            "Vcd Filter", ".vcd")
{
    filterCapability = FiltCapabilty_Import;

    hier_delimeter = '.';
    deadchar       = 'X';
    vcd_explicit_zero_subscripts=-1;  /* 0=yes, -1=no */
    atomic_vectors = 1;

    T_MAX_STR     = 1024;
    yytext        = (char *)malloc(T_MAX_STR+1);
    sym           = (struct symbol **)calloc(SYMPRIME,sizeof(struct symbol *));
}

/******************************************************
 * isValidFile()
 ******************************************************/
Int32 VcdFilter::isValidFile(FILE *fp)
{
    return 0;
}

/********************************************************************
 * Import()
 ********************************************************************/
Int32 VcdFilter::Import(DFIO **dfio, FileAccObj &file)
{
    this->file = file;
    data = *dfio;

    /**** Need to clear stuff out first... ****/
    Init();

    build_slisthier();
    vcd_parse();
    add_tail_histents();
    vcd_build_symbols();

    /* if (dostats) vcd_sortfacs(); */

    min_time=start_time*time_scale;
    max_time=end_time*time_scale;

    if (!d_keywords) {
        Tcl_AppendResult(interp, "invalid VCD file - no $end tokens", 0);
        return -1;
    }

    return 0;
}

/********************************************************************
 * Export()
 ********************************************************************/
Int32 VcdFilter::Export(DFIO **dfio, FileAccObj &file)
{
    this->file = file;
    data = *dfio;

    return -1;
}

/********************************************************************
 * Vcd_filter_Init()
 *
 * Registers this filter with the IVI backplane
 ********************************************************************/
extern "C" int Vcd_filter_Init(Tcl_Interp *interp)
{
    VcdFilter    *vcdf;

    if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
        return TCL_ERROR;
    }

    Tcl_PkgProvide(interp, "vcd_filter", "1.0");

    vcdf = new VcdFilter(interp);

    DFIOMgr_RegisterDFIOFilter(vcdf);

    return TCL_OK;
}

extern "C" int Vcd_filter_SafeInit(Tcl_Interp *interp)
{
    return Vcd_filter_SafeInit(interp);
}

/*******************************************************************
 *******************************************************************/

/* #include "v2l_analyzer.h" */

#undef VCD_BSEARCH_IS_PERFECT           /* bsearch is imperfect under linux, but OK under AIX */

static FILE *vcd_handle=NULL;
static char vcd_is_compressed=0;

/*
 * atoi 64-bit version..
 * y/on     default to '1'
 * n/nonnum default to '0'
 */
static TimeType atoi_64(char *str) {
    TimeType val=0;
    unsigned char ch, nflag=0;

    switch(*str) {
	case 'y':
	case 'Y':
	    return(LLDescriptor(1));

	case 'o':
	case 'O':
	    str++;
	    ch=*str;
	    if((ch=='n')||(ch=='N'))
		return(LLDescriptor(1));
	    else
		return(LLDescriptor(0));

	case 'n':
	case 'N':
	    return(LLDescriptor(0));

	default:
	    break;
	}

    while((ch=*(str++))) {
	if((ch>='0')&&(ch<='9')) {
	    val=(val*10+(ch&15));
	} else if((ch=='-')&&(val==0)&&(!nflag)) {
	    nflag=1;
	} else if(val) {
	    break;
	}
    }
    return(nflag?(-val):val);
}


#define NUM_TOKENS 17
#define NUM_VTOKENS 19
#define T_GET tok=get_token();if((tok==T_END)||(tok==T_EOF))break;

/*
 * bsearch compare
 */
static int vcdsymbsearchcompare(const void *s1, const void *s2) {
    char *v1;
    struct vcdsymbol *v2;

    v1=(char *)s1;
    v2=*((struct vcdsymbol **)s2);

    return(strcmp(v1, v2->id));
}

/*
 * actual bsearch
 */
struct vcdsymbol *VcdFilter::bsearch_vcd(char *key) {
    struct vcdsymbol **v;
    struct vcdsymbol *t;

    v=(struct vcdsymbol **)bsearch(key, sorted, numsyms,
				   sizeof(struct vcdsymbol *),
				   vcdsymbsearchcompare);

    if(v) {
#ifndef VCD_BSEARCH_IS_PERFECT
	for(;;) {
	    t=*v;

	    if((v==sorted)||(strcmp((*(--v))->id, key))) {
		return(t);
	    }
	}
#else
	return(*v);
#endif
    } else {
        return(NULL);
    }
}


/*
 * sort on vcdsymbol pointers
 */
static int vcdsymcompare(const void *s1, const void *s2) {
    struct vcdsymbol *v1, *v2;

    v1=*((struct vcdsymbol **)s1);
    v2=*((struct vcdsymbol **)s2);

    return(strcmp(v1->id, v2->id));
}


/********************************************************************
 * create_sorted_table()
 *
 * create sorted (by id) table
 ********************************************************************/
void VcdFilter::create_sorted_table() 
{
    struct vcdsymbol *v;
    struct vcdsymbol **pnt;

    if(numsyms) {
        pnt = sorted = 
            (struct vcdsymbol **)calloc(numsyms, sizeof(struct vcdsymbol *));
        v=vcdsymroot;
        while(v) {
            *(pnt++)=v;
            v=v->next;
        }

        qsort(sorted, numsyms, sizeof(struct vcdsymbol *), vcdsymcompare);
    }
}


/********************************************************************
 * getch()
 ********************************************************************/
int VcdFilter::getch() {
    Uint8   ch;

    if (file.read_uint8(&ch) < 1) {
        if (file.eof()) {
            return EOF;
        } else {
            return -1;
        }
    } 

    if (ch=='\n') {
        vcdlineno++;
    }
    return ch;
}

static char *varsplit=NULL, *vsplitcurr=NULL;

/********************************************************************
 * getch_patched()
 ********************************************************************/
int VcdFilter::getch_patched() {
    char ch;

    ch=*vsplitcurr;
    if(!ch) {
        return(-1);
    } else {
        vsplitcurr++;
        return((int)ch);
    }
}

/********************************************************************
 * get_token()
 *
 * simple tokenizer
 ********************************************************************/
int VcdFilter::get_token() {
    int ch;
    int i, len=0;
    int is_string=0;
    char *yyshadow;

    for(;;) {
	ch=getch();
	if(ch<0) return(T_EOF);
	if(ch<=' ') continue;	/* val<=' ' is a quick whitespace check      */
	break;			/* (take advantage of fact that vcd is text) */
    }
    if(ch=='$') {
	yytext[len++]=ch;
	for(;;) {
	    ch=getch();
	    if(ch<0) return(T_EOF);
	    if(ch<=' ') continue;
	    break;
	}
    } else {
	is_string=1;
    }

    for(yytext[len++]=ch;;yytext[len++]=ch) {
	if(len==T_MAX_STR) {
	    yytext = (char *)realloc(yytext, (T_MAX_STR=T_MAX_STR*2)+1);
	}
	ch=getch();
	if(ch<=' ') break;
    }
    yytext[len]=0;	/* terminator */

    if(is_string) {
	yylen=len;
	return(T_STRING);
    }

    yyshadow=yytext;
    do {
	yyshadow++;
	for(i=0;i<NUM_TOKENS;i++) {
	    if(!strcmp(yyshadow,tokens[i])) {
                return(i);
	    }
        }
    } while(*yyshadow=='$'); /* fix for RCS ids in version strings */

    return(T_UNKNOWN_KEY);
}

static int var_prevch=0;


/********************************************************************
 * get_vartoken_patched()
 ********************************************************************/
int VcdFilter::get_vartoken_patched() {
    int ch;
    int i, len=0;
        
    if(!var_prevch) {
        for(;;) {
	    ch=getch_patched();
	    if (ch<0) { 
                free(varsplit); varsplit=NULL; return(V_END); 
            }

	    if ((ch==' ')||(ch=='\t')||(ch=='\n')||(ch=='\r')) continue;
	    break;   
	}
    } else {
        ch=var_prevch;
        var_prevch=0;
    }
                 
    if(ch=='[') return(V_LB);
    if(ch==':') return(V_COLON);
    if(ch==']') return(V_RB);
 
    for(yytext[len++]=ch;;yytext[len++]=ch) {
	if(len==T_MAX_STR) {
	    yytext = (char *)realloc(yytext, (T_MAX_STR=T_MAX_STR*2)+1);
	}
        ch=getch_patched();
        if(ch<0) break;
        if((ch==':')||(ch==']')) {
	    var_prevch=ch;
	    break;
	}
    }
    yytext[len]=0;  /* terminator */
                
    for(i=0;i<NUM_VTOKENS;i++) {
        if(!strcmp(yytext,vartypes[i])) {
	    if(ch<0) { free(varsplit); varsplit=NULL; }
	    return(i);
	}
    }

    yylen=len;
    if(ch<0) { free(varsplit); varsplit=NULL; }
    return(V_STRING);
}


/********************************************************************
 * get_vartoken
 ********************************************************************/
int VcdFilter::get_vartoken() {
    int ch;
    int i, len=0;

    if(varsplit) {
        int rc=get_vartoken_patched();
        if(rc!=V_END) return(rc);  
        var_prevch=0;
    }
                
    if(!var_prevch) {
        for(;;) {
	    ch=getch();
	    if(ch<0) return(V_END);
	    if((ch==' ')||(ch=='\t')||(ch=='\n')||(ch=='\r')) continue;
	    break;
	}
    } else {
        ch=var_prevch;
        var_prevch=0;
    }
                 
    if(ch=='[') return(V_LB);
    if(ch==':') return(V_COLON);
    if(ch==']') return(V_RB);  
        
    for(yytext[len++]=ch;;yytext[len++]=ch) {
        if(len==T_MAX_STR) {
	    yytext = (char *)realloc(yytext, (T_MAX_STR=T_MAX_STR*2)+1);
	}
        ch=getch();
        if((ch==' ')||(ch=='\t')||(ch=='\n')||(ch=='\r')||(ch<0)) break;
        if((ch=='[')&&(yytext[0]!='\\')) {
	    varsplit=yytext+len;            /* keep looping so we get the *last* one */
	} else if(((ch==':')||(ch==']'))&&(!varsplit)&&(yytext[0]!='\\')) {
	    var_prevch=ch;
	    break;
	}
    }
    yytext[len]=0;  /* absolute terminator */
    if((varsplit)&&(yytext[len-1]==']')) {
        char *vst;
        vst = new char[strlen(varsplit)+1];
        strcpy(vst, varsplit);

        *varsplit=0x00;         /* zero out var name at the left bracket */
        len=varsplit-yytext;

        varsplit=vsplitcurr=vst;
        var_prevch=0;
    } else {
        varsplit=NULL;
    }
        
    for(i=0;i<NUM_VTOKENS;i++) {
        if(!strcmp(yytext,vartypes[i])) {
	    return(i);
	}
    }

    yylen=len;
    return(V_STRING);
}

/********************************************************************
 * get_strtoken()
 ********************************************************************/
int VcdFilter::get_strtoken() {
    int ch;
    int len=0;

    if(!var_prevch) {
	for(;;) {
	    ch=getch();
	    if(ch<0) return(V_END);
	    if((ch==' ')||(ch=='\t')||(ch=='\n')||(ch=='\r')) continue;
	    break;
	}
    } else {
	ch=var_prevch;
	var_prevch=0;
    }
      
    for(yytext[len++]=ch;;yytext[len++]=ch) {
	if(len==T_MAX_STR) {
	    yytext = (char *)realloc(yytext, (T_MAX_STR=T_MAX_STR*2)+1);
	}
	ch=getch();
	if((ch==' ')||(ch=='\t')||(ch=='\n')||(ch=='\r')||(ch<0)) break;
    }
    yytext[len]=0;        /* terminator */

    yylen=len;
    return(V_STRING);
}

/********************************************************************
 * sync_end()
 ********************************************************************/
void VcdFilter::sync_end(char *hdr) 
{
    int tok;

    if(hdr) DEBUG(fprintf(stderr,"%s",hdr));
    for(;;) {
	tok=get_token();
	if((tok==T_END)||(tok==T_EOF)) break;
	if(hdr)DEBUG(fprintf(stderr," %s",yytext));
    }
    if(hdr) DEBUG(fprintf(stderr,"\n"));
}

/********************************************************************
 * build_slisthier()
 ********************************************************************/
char *VcdFilter::build_slisthier() {
    struct slist *s;
    int len=0;

    if(!slistroot) {
	if(slisthier) {
	    free(slisthier);
	}

	slisthier_len=0;
	slisthier = new char[1];
	*slisthier=0;
	return(slisthier);
    }

    s=slistroot; len=0;
    while(s) {
	len+=s->len+(s->next?1:0);
	s=s->next;
    }

    slisthier =new char[(slisthier_len=len)+1];
    s=slistroot; len=0;
    while(s) {
	strcpy(slisthier+len,s->str);
	len+=s->len;
	if(s->next) {
	    strcpy(slisthier+len,vcd_hier_delimeter);
	    len++;
	}
	s=s->next;
    }
    return(slisthier);
}


/********************************************************************
 * append_vcd_slisthier()
 ********************************************************************/
void VcdFilter::append_vcd_slisthier(char *str) {
    struct slist *s = (struct slist *)calloc(1,sizeof(struct slist));

    s->len =strlen(str);
    s->str = new char[s->len+1];
    strcpy(s->str,str);

    if(slistcurr) {
	slistcurr->next=s;
	slistcurr=s;
    } else {
	slistcurr=slistroot=s;
    }

    build_slisthier();
    DEBUG(fprintf(stderr, "SCOPE: %s\n",slisthier));
}


/********************************************************************
 * parse_valuechange()
 ********************************************************************/
void VcdFilter::parse_valuechange() {
    struct vcdsymbol *v;
    char *vector;
    int vlen;

    switch(yytext[0]) {
    case '0':
    case '1':
    case 'x':
    case 'X':
    case 'z':
    case 'Z':
	if(yylen>1) {
	    v=bsearch_vcd(yytext+1);	
	    if(!v) {
		fprintf(stderr,"Near line %d, Unknown VCD identifier: "
                        "'%s'\n",vcdlineno,yytext+1);
	    } else if(v->vartype!=V_EVENT) {
		char vl[2];
		vl[0]=yytext[0]; vl[1]=0;
                v->trace->setValueBitStr(0, vl);

		v->value[0]=yytext[0];
		DEBUG(fprintf(stderr,"%s = '%c'\n",v->name,v->value[0]));
		add_histent(current_time,v->narray[0],v->value[0],1, NULL);
	    } else {
		char vl[2];

		v->value[0]=(dumping_off)?'x':'1'; /* only '1' is relevant */
		if(current_time!=(v->ev->last_event_time+1)) {
		    /* dump degating event */
		    DEBUG(fprintf(stderr,"#"TTFormat" %s = '%c' "
                                "(event)\n",v->ev->last_event_time+1,
                                v->name,'0'));
		    add_histent(v->ev->last_event_time+1,v->narray[0],
                            '0',1, NULL);
		}
		DEBUG(fprintf(stderr,"%s = '%c' (event)\n",v->name,v->value[0]));
		add_histent(current_time,v->narray[0],v->value[0],1, NULL);

		vl[0]='1'; vl[1]=0;
                v->trace->setValueBitStr(0, vl);
/*		lt_emit_value_bit_string(lt, v->ltsym, 0, vl); */
		vl[0]='0'; vl[1]=0;
/*		lt_emit_value_bit_string(lt, v->ltsym, 0, vl); */

		v->ev->last_event_time=current_time;
	    }
	} else {
	    fprintf(stderr,"Near line %d, Malformed VCD identifier\n", vcdlineno);
	}
	break;

    case 'b':
    case 'B':
	/* extract binary number then.. */
	vector = (char *)malloc(yylen_cache=yylen); 
	strcpy(vector,yytext+1);
	vlen=yylen-1;

	get_strtoken();
	v=bsearch_vcd(yytext);	
	if(!v) {
	    fprintf(stderr,"Near line %d, Unknown identifier: '%s'\n",vcdlineno, yytext);
	    free(vector);
	} else {
	    if(vlen<v->size) { 	/* fill in left part */
		char extend;
		int i, fill;

		extend=(vector[0]=='1')?'0':vector[0];

		fill=v->size-vlen;				
		for(i=0;i<fill;i++) {
		    v->value[i]=extend;
		}
		strcpy(v->value+fill,vector);
	    } else if(vlen==v->size) { 	/* straight copy */
		strcpy(v->value,vector);
	    } else {			/* too big, so copy only right half */
		int skip;

		skip=vlen-v->size;
		strcpy(v->value,vector+skip);
	    }
	    DEBUG(fprintf(stderr,"%s = '%s'\n",v->name, v->value));
            v->trace->setValueBitStr(0, v->value);

	    if((v->size==1)||(!atomic_vectors)) {
		int i;
		for(i=0;i<v->size;i++) {
		    add_histent(current_time,v->narray[i],v->value[i],1, NULL);
		}
		free(vector);
	    } else {
		if(yylen_cache!=(v->size+1)) {
		    free(vector);
		    vector = (char *)malloc(v->size+1);
		}
		strcpy(vector,v->value);
		add_histent(current_time, v->narray[0],0,1,vector);
		free(vector);
	    }
	}
	break;

    case 'p':
	/* extract port dump value.. */
	vector = (char *)malloc(yylen_cache=yylen); 
	strcpy(vector,yytext+1);
	vlen=yylen-1;

	get_strtoken();	/* throw away 0_strength_component */
	get_strtoken(); /* throw away 0_strength_component */
	get_strtoken(); /* this is the id                  */
	v=bsearch_vcd(yytext);	
	if(!v) {
	    fprintf(stderr,"Near line %d, Unknown identifier: '%s'\n",vcdlineno, yytext);
	    free(vector);
	} else {
	    if(vlen<v->size) { 	/* fill in left part */
		char extend;
		int i, fill;

		extend='0';
		fill=v->size-vlen;
		for(i=0;i<fill;i++) {
		    v->value[i]=extend;
		}
		evcd_strcpy(v->value+fill,vector);
	    } else if(vlen==v->size) { 	/* straight copy */
		evcd_strcpy(v->value,vector);
	    } else {			/* too big, so copy only right half */
		int skip = vlen-v->size;
		evcd_strcpy(v->value,vector+skip);
	    }
	    DEBUG(fprintf(stderr,"%s = '%s'\n",v->name, v->value));
            v->trace->setValueBitStr(0, v->value);

	    if((v->size==1)||(!atomic_vectors)) {
		int i;
		for(i=0;i<v->size;i++) {
		    add_histent(current_time,v->narray[i],v->value[i],1, NULL);
		}
		free(vector);
	    } else {
		if(yylen_cache<v->size) {
		    free(vector);
		    vector = (char *)malloc(v->size+1);
		}
		strcpy(vector,v->value);
		add_histent(current_time, v->narray[0],0,1,vector);
		free(vector);
	    }
	}
	break;

    case 'r':
    case 'R': {
	double *d = (double *) malloc(sizeof(double));

	sscanf(yytext+1,"%lg",d);
		
	get_strtoken();
	v=bsearch_vcd(yytext);	
	if(!v) {
	    fprintf(stderr,"Near line %d, Unknown identifier: '%s'\n",vcdlineno, yytext);
	    free(d);
	} else {
            fprintf(stderr, "WARNING :: double value...\n");

/*	    lt_emit_value_double(lt, v->ltsym, 0, *d); */
/*	    lt_emit_value_bit_string(lt, v->ltsym, 0, v->value); */
	    add_histent(current_time, v->narray[0],'g',1,(char *)d);
	    free(d);
	}

	break;
    }

    }
}

/********************************************************************
 * evcd_strcpy()
 ********************************************************************/
void VcdFilter::evcd_strcpy(char *dst, char *src) {
    static char *evcd="DUNZduLHXTlh01?FAaBbCcf";
    static char  *vcd="01xz0101xz0101xzxxxxxxx";

    char ch;
    int i;

    while((ch=*src)) {
	for(i=0;i<23;i++) {
	    if(evcd[i]==ch) {
		*dst=vcd[i];
		break;
	    }
	}	
	if(i==23) *dst='x';

	src++;
	dst++;
    }

    *dst=0;	/* null terminate destination */
}


/********************************************************************
 * vcd_parse()
 ********************************************************************/
void VcdFilter::vcd_parse() {
    int tok;

    for(;;) {
	switch(tok=get_token()) {
	case T_COMMENT:
	    sync_end("COMMENT:");
	    break;
	case T_DATE:
	    sync_end("DATE:");
	    break;
	case T_VERSION:
	    sync_end("VERSION:");
	    break;
	case T_TIMESCALE: {
	    int vtok;
	    int i;
	    char prefix=' ';

	    vtok=get_token();
	    if((vtok==T_END)||(vtok==T_EOF)) break;
	    time_scale=atoi_64(yytext);
	    if(!time_scale) time_scale=1;
	    for(i=0;i<yylen;i++) {
		if((yytext[i]<'0')||(yytext[i]>'9')) {
		    prefix=yytext[i];
		    break;
		}
	    }
	    if(prefix==' ') {
		vtok=get_token();
		if((vtok==T_END)||(vtok==T_EOF)) break;
		prefix=yytext[0];		
	    }

	    switch(prefix) {
	    case 's':
	    case ' ': data->setTimeUnit(0);   break;
	    case 'm': 
                data->setTimeUnit(-3);
                break;
	    case 'u': 
                data->setTimeUnit(-6);
                break;
	    case 'n': 
                data->setTimeUnit(-9);
                break;
	    case 'p': 
                data->setTimeUnit(-12);
                break;
	    case 'f': 
                data->setTimeUnit(-15);
                break;
	    default:	/* unknown */
                data->setTimeUnit(-9);
                break;
	    }
            data->setUnitMultiplier(time_scale);

	    sync_end(NULL);
	    break;
	}
	case T_SCOPE:
	    T_GET;
	    T_GET;
	    if(tok==T_STRING) {
		struct slist *s;
		s=(struct slist *)calloc(1,sizeof(struct slist));
		s->len=yylen;
		s->str=(char *)malloc(yylen+1);
		strcpy(s->str,yytext);

		if(slistcurr) {
		    slistcurr->next=s;
		    slistcurr=s;
		} else {
		    slistcurr=slistroot=s;
		}

		build_slisthier();
		DEBUG(fprintf(stderr, "SCOPE: %s\n",slisthier));
	    }
	    sync_end(NULL);
	    break;
	case T_UPSCOPE:
	    if(slistroot) {
		struct slist *s = slistroot;

		if(!s->next) {
		    free(s->str);
		    free(s);
		    slistroot=slistcurr=NULL;
		} else
		    for(;;) {
			if(!s->next->next) {
			    free(s->next->str);
			    free(s->next);
			    s->next=NULL;
			    slistcurr=s;
			    break;
			}
			s=s->next;
		    }
		build_slisthier();
		DEBUG(fprintf(stderr, "SCOPE: %s\n",slisthier));
	    }
	    sync_end(NULL);
	    break;
	case T_VAR: {
	    int vtok;
	    struct vcdsymbol *v=NULL;

	    var_prevch=0;
	    varsplit=NULL;
	    vtok=get_vartoken();
	    if(vtok>V_PORT) goto bail;

	    v=(struct vcdsymbol *)calloc(1,sizeof(struct vcdsymbol));
	    v->vartype=vtok;
	    v->msi=v->lsi=vcd_explicit_zero_subscripts; /* indicate [un]subscripted status */

	    if(vtok==V_PORT) {
		vtok=get_vartoken();
		if(vtok==V_STRING) {
		    v->size=atoi_64(yytext);
		    if(!v->size) v->size=1;
		} else if(vtok==V_LB) {
		    vtok=get_vartoken();
		    if(vtok==V_END) goto err;
		    if(vtok!=V_STRING) goto err;
		    v->msi=atoi_64(yytext);
		    vtok=get_vartoken();
		    if(vtok==V_RB) {
			v->lsi=v->msi;
			v->size=1;
		    } else {
			if(vtok!=V_COLON) goto err;
			vtok=get_vartoken();
			if(vtok!=V_STRING) goto err;
			v->lsi=atoi_64(yytext);
			vtok=get_vartoken();
			if(vtok!=V_RB) goto err;

			if(v->msi>v->lsi) {
			    v->size=v->msi-v->lsi+1;
			} else {
			    v->size=v->lsi-v->msi+1;
			}
		    }
		} else
		    goto err;

		vtok=get_strtoken();
		if(vtok==V_END) goto err;
		v->id=(char *)malloc(yylen+1);
		strcpy(v->id, yytext);

		vtok=get_vartoken();
		if(vtok!=V_STRING) goto err;
		if(slisthier_len) {
		    v->name=(char *)malloc(slisthier_len+1+yylen+1);
		    strcpy(v->name,slisthier);
		    strcpy(v->name+slisthier_len,vcd_hier_delimeter);
		    strcpy(v->name+slisthier_len+1,yytext);
		} else {
		    v->name=(char *)malloc(yylen+1);
		    strcpy(v->name,yytext);
		}
	    } else {	/* regular vcd var, not an evcd port var */
		vtok=get_vartoken();
		if(vtok==V_END) goto err;
		v->size=atoi_64(yytext);
		if(!v->size) v->size=1;
		vtok=get_strtoken();
		if(vtok==V_END) goto err;
		v->id=(char *)malloc(yylen+1);
		strcpy(v->id, yytext);

		vtok=get_vartoken();
		if(vtok!=V_STRING) goto err;
		if(slisthier_len) {
		    v->name=(char *)malloc(slisthier_len+1+yylen+1);
		    strcpy(v->name,slisthier);
		    strcpy(v->name+slisthier_len,vcd_hier_delimeter);
		    strcpy(v->name+slisthier_len+1,yytext);
		} else {
		    v->name=(char *)malloc(yylen+1);
		    strcpy(v->name,yytext);
		}

		vtok=get_vartoken();
		if(vtok==V_END) goto dumpv;
		if(vtok!=V_LB) goto err;
		vtok=get_vartoken();
		if(vtok!=V_STRING) goto err;
		v->msi=atoi_64(yytext);
		vtok=get_vartoken();
		if(vtok==V_RB) {
		    v->lsi=v->msi;
		    goto dumpv;
		}
		if(vtok!=V_COLON) goto err;
		vtok=get_vartoken();
		if(vtok!=V_STRING) goto err;
		v->lsi=atoi_64(yytext);
		vtok=get_vartoken();
		if(vtok!=V_RB) goto err;
	    }

	  dumpv:
	    if(v->vartype==V_REAL) {
		v->size=1;		/* override any data we parsed in */
		v->msi=v->lsi=0;
	    } else if((v->size>1)&&(v->msi<=0)&&(v->lsi<=0)) {
		if(v->vartype==V_EVENT) {
		    v->size=1;
		} else {
		    /* any criteria for the direction here? */
		    v->msi=v->size-1;	
		    v->lsi=0;
		}
	    } else if((v->msi>v->lsi)&&((v->msi-v->lsi+1)!=v->size)) {
		if(v->vartype!=V_EVENT) goto err;
		v->size=v->msi-v->lsi+1;
	    } else if((v->lsi>=v->msi)&&((v->lsi-v->msi+1)!=v->size)) {
		if(v->vartype!=V_EVENT) goto err;
		v->size=v->msi-v->lsi+1;
	    }

	    /* initial conditions */
	    v->value=(char *)malloc(v->size+1);
	    v->value[v->size]=0;
	    v->narray=(struct Node **)calloc(v->size,sizeof(struct Node *));
	    {
		int i;
		for(i=0;i<v->size;i++) {
		    v->value[i]='x';

		    v->narray[i]=(struct Node *)calloc(1,sizeof(struct Node));
		    v->narray[i]->head.time=-1;
		    v->narray[i]->head.v.val=1;
		}
	    }

	    if(v->vartype==V_EVENT) {
		struct queuedevent *q;
		v->ev=q=(struct queuedevent *)calloc(1,sizeof(struct queuedevent));
		q->sym=v;
		q->last_event_time=-1;		
		q->next=queuedevents;
		queuedevents=q;		
	    }

	    if(!vcdsymroot) {
		vcdsymroot=vcdsymcurr=v;
	    } else {
		vcdsymcurr->next=v;
		vcdsymcurr=v;
	    }
	    numsyms++;

	    if((v->vartype==V_INTEGER)||(v->vartype==V_REAL)) {
                v->trace = data->newTrace(v->name, 1, v->msi, v->lsi, 0);
                data->addTrace(v->trace);
                /*
                (v->vartype==V_INTEGER)?LT_SYM_F_INTEGER:((v->vartype==V_REAL)?LT_SYM_F_DOUBLE:LT_SYM_F_BITS)); 
                 */
	    } else {
                v->trace = data->newTrace(v->name, 1, 
                        (v->msi<0)?0:v->msi, (v->lsi<0)?0:v->lsi, 0);
                data->addTrace(v->trace);
                /*
		(v->vartype==V_INTEGER)?LT_SYM_F_INTEGER:((v->vartype==V_REAL)?LT_SYM_F_DOUBLE:LT_SYM_F_BITS)); 
                */
	    }

	    DEBUG(fprintf(stderr,"VAR %s %d %s %s[%d:%d]\n",
			  vartypes[v->vartype], v->size, v->id, v->name, 
			  v->msi, v->lsi));
	    goto bail;
	  err:
	    if(v) {
		if(v->name) free(v->name);
		if(v->id) free(v->id);
		if(v->value) free(v->value);
		free(v);
	    }

	  bail:
	    if(vtok!=V_END) sync_end(NULL);
	    break;
	}
	case T_ENDDEFINITIONS:
	    if(!header_over) {
		header_over=1;	/* do symbol table management here */
		create_sorted_table();
		if(!sorted) {
		    fprintf(stderr, "No symbols in VCD file..nothing to do!\n");
		    exit(1);
		}
	    }
	    break;
	case T_STRING:
	    if(header_over) {
				/* catchall for events when header over */
		if(yytext[0]=='#') {
		    TimeType time;
		    time=atoi_64(yytext+1);
					
		    if(start_time<0) {
			start_time=time;
		    }
		    current_time=time;
		    if(end_time<time) end_time=time;        /* in case of malformed vcd files */
                    data->setTime(current_time);
		    DEBUG(fprintf(stderr,"#"TTFormat"\n",time));
		} else {
		    parse_valuechange();
		}
	    }
	    break;
	case T_DUMPALL:	/* dump commands modify vals anyway so */
	case T_DUMPPORTSALL:
	    break;	/* just loop through..                 */
	case T_DUMPOFF:
	case T_DUMPPORTSOFF:
	    dumping_off=1;
	    break;
	case T_DUMPON:
	case T_DUMPPORTSON:
	    dumping_off=0;
	    break;
	case T_DUMPVARS:
	case T_DUMPPORTS:
	    if(current_time<0) {
		start_time=current_time=end_time=0;
                data->setTime(current_time);
	    }
	    break;  
	case T_END:	/* either closure for dump commands or */
            d_keywords++;
	    break;	/* it's spurious                       */
	case T_UNKNOWN_KEY:
	    sync_end(NULL);	/* skip over unknown keywords */
	    break;
	case T_EOF:
	    return;
	default:
	    DEBUG(fprintf(stderr,"UNKNOWN TOKEN\n"));
	}
    }
}


/********************************************************************
 * add_histent()
 *
 * this function doesn't do anything useful anymore except for to 
 * mark statistics on the various nets...
 ********************************************************************/
void VcdFilter::add_histent(TimeType time, struct Node *n, 
        char ch, int regadd, char *vector) 
{
    struct HistEnt *he;

    if(!vector) {
	if(!n->curr) {
	    he = (struct HistEnt *)calloc(1,sizeof(struct HistEnt));
	    he->time = -1;
	    he->v.val = 1;

	    n->curr = he;
	    n->head.next = he;

	    add_histent(time,n,ch,regadd, vector);
	} else {
	    if(regadd) {
		time *= time_scale;
	    }

	    if(toupper(ch)!=deadchar) n->notdead = 1;
	    n->numtrans++;
       }
    } else if(ch=='g') {	/* real number */
	if(!n->curr) {
	    he = (struct HistEnt *)calloc(1,sizeof(struct HistEnt));
	    he->time = -1;
	    he->v.vector = NULL;

	    n->curr = he;
	    n->head.next = he;

	    add_histent(time,n,ch,regadd, vector);
	} else {
	    n->notdead = 1;
	    n->numtrans++;
	}
    } else if(!n->curr) {
	he = (struct HistEnt *)calloc(1,sizeof(struct HistEnt));
	he->time = -1;
	he->v.vector = NULL;

	n->curr = he;
	n->head.next = he;

	add_histent(time,n,ch,regadd, vector);
    } else {
	int i, nlen = strlen(vector);
	if(nlen) {
	    n->numtrans++;
	    for(i=0;i<nlen;i++) {
		if (toupper(vector[i])!=deadchar) {	
		    n->notdead = 1;
		    return;
		}
	    }
	}
    }
}


/********************************************************************
 * add_tail_histents
 ********************************************************************/
void VcdFilter::add_tail_histents(void) {
    /* dump out any pending events 1st */
    struct queuedevent *q;
    for (q=queuedevents ; q ; q=q->next) {
	struct vcdsymbol *v = q->sym;

	if (current_time != v->ev->last_event_time+1) {
	    /* dump degating event */
	    DEBUG(fprintf(stderr,"#"TTFormat" %s = '%c' (event)\n",
			  v->ev->last_event_time+1,v->name,'0'));
	    add_histent(v->ev->last_event_time+1,v->narray[0],'0',1, NULL);
	}
    }
}

/********************************************************************
 * vcd_build_symbols()
 ********************************************************************/
void VcdFilter::vcd_build_symbols() {
    int i, j, max_slen = -1, duphier = 0;
    char *str = NULL;

    for(i=0;i<numsyms;i++) {
	struct vcdsymbol *v, *vprime;
	int msi, delta, slen, substnode;

	v = sorted[i];
	msi = v->msi;
	delta = ((v->lsi-v->msi)<0)?-1:1;
	substnode = 0;

	slen = strlen(v->name);
	if (!str || slen>max_slen) {
	    max_slen = slen;
	    str = (char *)malloc(slen+32);	/* more than enough */
	}
	strcpy(str,v->name);

	if(v->msi>=0) {
	    strcpy(str+slen,vcd_hier_delimeter);
	    slen++;
	}

	vprime = bsearch_vcd(v->id);
	if(vprime!=v) /* hash mish means dup net */ {
	    if(v->size!=vprime->size) {
		fprintf(stderr,"ERROR: Duplicate IDs with differing width: %s %s\n", v->name, vprime->name);
	    } else {
		substnode = 1;

/*		v->ltsym->flags = LT_SYM_F_ALIAS;*/	/* alias to our other node that bsearch found */
/*		v->ltsym->aliased_to = vprime->ltsym; */
	    }
	}

	if ((v->size==1 || !atomic_vectors) && v->vartype!=V_REAL) {
	    for(j=0;j<v->size;j++) {
		if(v->msi>=0) {
		    if(!vcd_explicit_zero_subscripts)
			sprintf(str+slen,"%d",msi);
		    else
			sprintf(str+slen-1,"[%d]",msi);
		}

		if(symfind(str)) {
		    char *dupfix = (char *)malloc(max_slen+32);

		    DEBUG(fprintf(stderr,"Warning: %s is a duplicate VCD symbol.\n",str));

		    do
			sprintf(dupfix, "$DUP%d%s%s", duphier++,
				vcd_hier_delimeter, str);
		    while(symfind(dupfix));

		    strcpy(str, dupfix);
		    free(dupfix);
		    duphier = 0; /* reset for next duplicate resolution */
		}
		/* fallthrough */
		{
		    struct symbol *s = symadd(str,hash(str));

		    s->n = v->narray[j];
		    if(substnode) {
			struct Node *n, *n2;

			n = s->n;
			n2 = vprime->narray[j];
			/* nname stays same */
			n->head = n2->head;
			n->curr = n2->curr;
			/* harray calculated later */
			n->numhist = n2->numhist;

			n->substnode = n2;

			while(n2->substhead)
			    n2 = n2->substhead;
			n2->substhead = n;
		    }

		    s->n->nname = s->name;
		    s->h = s->n->curr;
		    if(!firstnode) {
			firstnode = curnode = s;
		    } else {
			curnode->nextinaet = curnode = s;
		    }

		    numfacs++;
		    DEBUG(fprintf(stderr,"Added: %s\n",str));
		}
		msi += delta;
	    }
	} else {		/* atomic vector */
	    if(v->vartype!=V_REAL) {
		sprintf(str+slen-1,"[%d:%d]",v->msi,v->lsi);
	    } else {
		*(str+slen-1) = 0;
	    }

	    if(symfind(str)) {
		char *dupfix = (char *)malloc(max_slen+32);

		DEBUG(fprintf(stderr,"Warning: %s is a duplicate VCD symbol.\n",str));

		do
		    sprintf(dupfix, "$DUP%d%s%s", duphier++,
			    vcd_hier_delimeter, str);
		while(symfind(dupfix));

		strcpy(str, dupfix);
		free(dupfix);
		duphier = 0; /* reset for next duplicate resolution */
	    }
	    /* fallthrough */
	    {
		struct symbol *s = symadd(str,hash(str));

		s->n = v->narray[0];
		if(substnode) {
		    struct Node *n, *n2;

		    n = s->n;
		    n2 = vprime->narray[0];
		    /* nname stays same */
		    n->head = n2->head;
		    n->curr = n2->curr;
		    /* harray calculated later */
		    n->numhist = n2->numhist;
		    n->ext = n2->ext;

		    n->substnode = n2;

		    while(n2->substhead)
			n2 = n2->substhead;
		    n2->substhead = n;
		} else {
		    struct ExtNode *en = (struct ExtNode *)
			malloc(sizeof(struct ExtNode));
		    en->msi = v->msi;
		    en->lsi = v->lsi;

		    s->n->ext = en;
		}

		s->n->nname = s->name;
		s->h = s->n->curr;
		if(!firstnode) {
		    firstnode = curnode = s;
		} else {
		    curnode->nextinaet = curnode = s;
		}

		numfacs++;
		DEBUG(fprintf(stderr,"Added: %s\n",str));
	    }
	}
    }

    free(str);
}

/********************************************************************
 * vcd_sortfacs()
 ********************************************************************/
void VcdFilter::vcd_sortfacs() 
{
    int i;

    facs=(struct symbol **)malloc(numfacs*sizeof(struct symbol *));
    curnode=firstnode;
    for(i=0;i<numfacs;i++) {
        char *subst, ch;

        facs[i]=curnode;
	subst=curnode->name;
        curnode=curnode->nextinaet;
        while((ch=(*subst))) {
	    if(ch==hier_delimeter) { *subst=0x01; } /* forces sort at hier boundaries */
	    subst++;
	}
    }
    quicksort(facs,0,numfacs-1);

    for(i=0;i<numfacs;i++) {
        char *subst, ch;
         
        subst=facs[i]->name;
        while((ch=(*subst))) {
	    if(ch==0x01) { *subst=hier_delimeter; } /* restore back to normal */
	    subst++;
	}
    }

    for(i=0;i<numfacs;i++) {        
	struct Node *n;
	int do_indent;

	if(!facs[i]->n->substnode) {
	    n=facs[i]->n;
	    printf("[%c] [%5d] %s", facs[i]->n->notdead?' ':'*',facs[i]->n->numtrans,facs[i]->name);
	    if(!facs[i]->n->notdead) {
		deadcnt++;
	    }
	    do_indent=0;
	} else {
	    n=facs[i]->n->substnode;
	    printf("[%c] [%5d] %s <-> %s", n->notdead?' ':'*',n->numtrans,facs[i]->name,n->nname);
	    if(!n->notdead) {
		deadcnt++;
	    }
	    do_indent=1;
	}

	n=n->substhead;
	while(n) {
	    if(strcmp(n->nname, facs[i]->name)) {
		if(do_indent) { printf("\n\t\t\t"); } else { do_indent=1; }
		printf(" <-> %s",n->nname);
	    }
	    n=n->substhead;
	}
	printf("\n");
    }

    printf("\n[%d] total facilities: [%d] facilit%s defined, [%d] facilit%s undefined.\n\n",
	   numfacs,numfacs-deadcnt,(numfacs-deadcnt!=1)?"ies":"y",deadcnt,
	   (deadcnt!=1)?"ies":"y");
}

/********************************************************************
 * hash()
 *
 * Generic hash function for symbol names...
 ********************************************************************/
static int hash(char *s) {
    char *p;
    unsigned int h=0, g;
    for(p=s;*p;p++) {
        h=(h<<4)+(*p);
        if((g=h&0xf0000000)) {
	    h=h^(g>>24);
	    h=h^g;
	}
    }
    return(h%SYMPRIME);
}

/********************************************************************
 *
 ********************************************************************/
/*
 * add symbol to table.  no duplicate checking
 * is necessary as aet's are "correct."
 */
struct symbol *VcdFilter::symadd(char *name, int hv) {
    struct symbol *s = (struct symbol *)calloc(1,sizeof(struct symbol));

    strcpy(s->name=(char *)malloc(strlen(name)+1),name);
    s->next=sym[hv];
    sym[hv]=s;
    return(s);
}

/*
 * find a slot already in the table...
 */
struct symbol *VcdFilter::symfind(char *s) {
    struct symbol *temp;

    for (temp=sym[hash(s)] ; temp ; temp=temp->next) {
	if(!strcmp(temp->name,s)) {
	    return(temp); /* in table already */
	}
    }
  
    return(NULL); /* not found, add here if you want to add*/
}

int sigcmp(char *s1, char *s2) {
    unsigned char c1, c2;
    int u1, u2;
                                
    for(;;) {
        c1=(unsigned char)*(s1++);
        c2=(unsigned char)*(s2++);
                
        if((!c1)&&(!c2)) return(0);
        if((c1<='9')&&(c2<='9')&&(c2>='0')&&(c1>='0')) {
	    u1=(int)(c1&15);
	    u2=(int)(c2&15);
                
	    while(((c2=(unsigned char)*s2)>='0')&&(c2<='9')) {
		u2*=10;
		u2+=(unsigned int)(c2&15);
		s2++;
	    }

	    while(((c2=(unsigned char)*s1)>='0')&&(c2<='9')) {
		u1*=10;
		u1+=(unsigned int)(c2&15);
		s1++;
	    }
        
	    if(u1==u2) continue;
	    else return((int)u1-(int)u2);
	} else {
	    if(c1!=c2) return((int)c1-(int)c2);
	}
    }       
}

static int partition(struct symbol **a, int p, int r) {
    struct symbol *x, *t;
    int i,j;

    x=a[p];
    i=p-1;
    j=r+1;

    while(1) {
        do {
	    j--;
	} while(sigcmp(a[j]->name,x->name)>0);

        do {
	    i++;
	} while(sigcmp(a[i]->name,x->name)<0);
 
        if(i<j) {
	    t=a[i];
	    a[i]=a[j];
	    a[j]=t; 
	} else {
	    return(j);
	}
    }
}

static void quicksort(struct symbol **a, int p, int r) 
{
    int q;

    if(p<r) {
        q = partition(a,p,r);
        quicksort(a,p,q);
        quicksort(a,q+1,r);
    }
}



