/****************************************************************************
 *                          GhdlRemoteSimPlugin.cc
 *
 * Author: Matthew Ballance
 * Desc:
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "GhdlRemoteSimPlugin.h"
#include "iviSplitString.h"
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>

static p_vpi_thunk            d_tp;

/********************************************************************
 * GhdlRemoteSimPlugin()
 ********************************************************************/
GhdlRemoteSimPlugin::GhdlRemoteSimPlugin(Tcl_Interp *interp) 
    : IviRemoteSimPlugin(interp)
{
    SetupVpiCBs();
}

/********************************************************************
 * ~GhdlRemoteSimPlugin()
 ********************************************************************/
GhdlRemoteSimPlugin::~GhdlRemoteSimPlugin()
{
}

/********************************************************************
 * BeginRun()
 ********************************************************************/
int GhdlRemoteSimPlugin::BeginRun(Uint32 runLen)
{
    s_vpi_time           time_s, tt;
    s_cb_data            cbdata;

    tt.type = vpiSimTime;
    d_sim->vpi_get_time(0, &tt);

    cbdata.reason = cbReadOnlySynch;
    time_s.low = runLen + tt.low;
    time_s.high = 0;
    time_s.type = vpiSimTime;

    cbdata.time = &time_s;
    cbdata.obj  = 0;
    cbdata.value = 0;

    cbdata.user_data = (char *)this;
    cbdata.cb_rtn    = &GhdlRemoteSimPlugin::TimeCB;
    d_sim->vpi_register_cb(&cbdata);

    return TCL_OK;
}

/********************************************************************
 * ConstructSim()
 ********************************************************************/
IviSim *GhdlRemoteSimPlugin::ConstructSim()
{
    VPIProcVecStruct  *pv;
    char               buf[1024];
    int                argc;
    char             **argv;

    pv = new VPIProcVecStruct;
    memset(pv, 0, sizeof(VPIProcVecStruct));

    pv->vpi_register_systf     = d_tp->vpi_register_systf;
//    pv->vpi_vprintf            = d_tp->vpi_vprintf;
    pv->vpi_mcd_close          = d_tp->vpi_mcd_close;
    pv->vpi_mcd_name           = d_tp->vpi_mcd_name;
    pv->vpi_mcd_open           = d_tp->vpi_mcd_open;
//    pv->vpi_mcd_vprintf        = d_tp->vpi_mcd_vprintf;
    pv->vpi_register_cb        = d_tp->vpi_register_cb;
    pv->vpi_remove_cb          = d_tp->vpi_remove_cb;
//    pv->vpi_sim_vcontrol       = d_tp->vpi_sim_vcontrol;
    pv->vpi_handle             = d_tp->vpi_handle;
    pv->vpi_iterate            = d_tp->vpi_iterate;
    pv->vpi_scan               = d_tp->vpi_scan;
    pv->vpi_handle_by_index    = d_tp->vpi_handle_by_index;
    pv->vpi_get_time           = d_tp->vpi_get_time;
    pv->vpi_get                = d_tp->vpi_get;
    pv->vpi_get_str            = d_tp->vpi_get_str;
    pv->vpi_get_value          = d_tp->vpi_get_value;
    pv->vpi_put_value          = d_tp->vpi_put_value;
    pv->vpi_free_object        = d_tp->vpi_free_object;
    pv->vpi_get_vlog_info      = d_tp->vpi_get_vlog_info;

    sprintf(buf, "sim %s", getInstName());
    iviSplitString(buf, &argc, &argv);

    d_sim = new GhdlRemoteIviSim(getInterp(), argc, argv, pv);

    iviFreeStringArr(argv);

    return d_sim;
}

/********************************************************************
 * TimeCB()
 ********************************************************************/
void GhdlRemoteSimPlugin::TimeCB()
{
    Uint64            iTime;
    s_vpi_time        tt;

    tt.type = vpiSimTime;
    d_sim->vpi_get_time(0, &tt);

    iTime.low  = tt.low;
    iTime.high = tt.high;

    SignalEndOfRun(iTime);
}

/********************************************************************
 * EndOfSimCB()
 ********************************************************************/
void GhdlRemoteSimPlugin::EndOfSimCB()
{
    Uint64            iTime;
    s_vpi_time        tt;

    tt.type = vpiSimTime;
    d_sim->vpi_get_time(0, &tt);

    iTime.low  = tt.low;
    iTime.high = tt.high;

    SignalEndOfSimulation(iTime);
}

/********************************************************************
 * EndOfCompileCB()
 ********************************************************************/
void GhdlRemoteSimPlugin::EndOfCompileCB()
{
    SignalReadyForSimulation();
}

/********************************************************************
 * SetupVpiCBs()
 ********************************************************************/
void GhdlRemoteSimPlugin::SetupVpiCBs()
{
    s_cb_data     cbdata;

    cbdata.reason          = cbEndOfCompile;
    cbdata.cb_rtn          = &GhdlRemoteSimPlugin::EndOfCompileCB;
    cbdata.user_data       = (char *)this;
    d_tp->vpi_register_cb(&cbdata);

    cbdata.reason          = cbEndOfSimulation;
    cbdata.cb_rtn          = &GhdlRemoteSimPlugin::EndOfSimCB;
    cbdata.user_data       = (char *)this;
    d_tp->vpi_register_cb(&cbdata);
}

/********************************************************************
 * hup_handler()
 ********************************************************************/
static void hup_handler(int sig)
{
    fprintf(stderr, "HUP handler\n");
    _exit(0);
}

extern "C" int IviCommon_RemoteInit(Tcl_Interp *interp);
extern "C" int Shm_dfio_Init(Tcl_Interp *interp);

/********************************************************************
 * GhdlRemoteSimPlugin_Init()
 ********************************************************************/
static void GhdlRemoteSimPlugin_Init()
{
    GhdlRemoteSimPlugin     *vplug;
    Tcl_Interp              *interp;

    if (getenv("GHDL_REMOTE_SIM_PAUSE")) {
        char buf[16];
        fprintf(stderr, "pid %d paused...\n", getpid());
        fgets(buf, 16, stdin);
    }

    interp = Tcl_CreateInterp();
    Tcl_Init(interp);

#ifndef __MINGW32__
    signal(SIGHUP, hup_handler);
#endif

    IviCommon_RemoteInit(interp);
    Shm_dfio_Init(interp);

    vplug = new GhdlRemoteSimPlugin(interp);

    if (!vplug->ok()) {
        fprintf(stderr, "ERROR: Cannot initialize IVI GHDL plugin\n");
    }
}


/********************************************************************
 * vlog_startup_routines
 ********************************************************************/
void (*vlog_startup_routines[])() = {
    GhdlRemoteSimPlugin_Init,
    0
};

/********************************************************************
 * vpi_register_sim
 ********************************************************************/
extern "C" int vpi_register_sim(p_vpi_thunk tp)
{
    d_tp = tp;
}

