/*
 *   Generic filesystem routines and wrappers
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#ifdef _WIN32
#include <string.h>
#else
#include <strings.h>
#include <unistd.h>
#endif


char *ifile_prefix=NULL; // No prefix by default
int ithe_devmode=0; // Don't allow file overrides by default

/*
static FILE *fpx;
#define LOGULE(x,y) {fpx=fopen("qslog","a");if(!fpx) return; fprintf(fpx,x,y); fclose(fpx);}
*/

#include "ithelib.h"

// Defines

#define MAXIFILES 8192

// Variables

static struct IFILEENTRY
	{
	char *name;
	long org;
	long len;
	char *tfn;
	time_t time;
	} ifiletab[MAXIFILES];
static int ifilenum=0;

// Functions

static int BSCMP(const void *key, const void *elem);
static int QSCMP(const void *elem1, const void *elem2);
static struct IFILEENTRY *ifile_search(char *filename);
static IFILE *iopen_readvfs(struct IFILEENTRY *ife);

// Code

/*
 *  iopen: top level file open routine.
 *         Does it's own priority checking
 */

IFILE *iopen(char *filename)
{
struct IFILEENTRY *ife;

// First, search the VFS table.
ife = ifile_search(filename);
if(ife)
	{
	// Got a result.  Any overriding files will have been dealt with
	// already by attach
	return iopen_readvfs(ife);
	}

return iopen_readfile(filename);
}


/*
 *  iopen_editor: top level file open routine, but with a different set of
 *                priorities to decide which file to open by default
 */

IFILE *iopen_editor(char *filename)
{
IFILE *ifp;
struct IFILEENTRY *ife;

// For the editor, open actual files by preference
ifp=iopen_readfile(filename);
if(!ifp)
	{
	ife = ifile_search(filename);
	if(ife)
		return iopen_readvfs(ife);
	}

return NULL;
}


/*
 *  Read direct from file.  You should call iopen, unless you have special reasons.
 */

IFILE *iopen_readfile(char *filename)
{
FILE *fp;
IFILE *ifp;
char buffer[1024];

fp=NULL;

// If we have a prefix, try that first

if(ifile_prefix)
	{
	strcpy(buffer,ifile_prefix);
	strcat(buffer,filename);
	#ifdef _WIN32
	strwinslash(buffer);
	#endif
	fp = fopen(buffer,"rb");
	}

//  Nothing?  Try without any prefix
if(!fp)
	fp = fopen(filename,"rb");

if(!fp)
	ithe_panic("iopen_readfile: fopen failed on file",filename);
//	return NULL;

// OK, we got a result with a physical file, load it and go
ifp = (IFILE *)M_get(1,sizeof(IFILE));
ifp->length = filelength(fileno(fp));
ifp->origin = 0;
ifp->fp = fp;

ifp->truename = M_get(1,strlen(filename)+1);
strcpy(ifp->truename,filename);

return ifp;
}


/*
 *  Read from a VFS file
 */

IFILE *iopen_readvfs(struct IFILEENTRY *ife)
{
FILE *fp;
IFILE *ifp;

ifp = (IFILE *)M_get(1,sizeof(IFILE));
ifp->length = ife->len;
ifp->origin = ife->org;
fp = fopen(ife->tfn,"rb");
if(!fp)
	ithe_panic("iopen_readvfs: fopen failed on file",ife->tfn);

fseek(fp,ifp->origin,SEEK_SET);
ifp->fp = fp;
ifp->truename = M_get(1,strlen(ife->tfn)+1);
strcpy(ifp->truename,ife->tfn);

return ifp;
}


/*
 *  Write to a file
 */

IFILE *iopen_write(char *filename)
{
FILE *fp;
IFILE *ifp;

// only physical files are supported for writing

fp = fopen(filename,"wb+");
if(!fp)
	ithe_panic("iopen_write: cannot create file:",filename);

// OK, we got a result with a physical file, start it and go
ifp = (IFILE *)M_get(1,sizeof(IFILE));
ifp->length = 0;
ifp->origin = 0;
ifp->fp = fp;

ifp->truename = M_get(1,strlen(filename)+1);
strcpy(ifp->truename,filename);

return ifp;
}


/*
 *  Does a file exist
 */

int iexist(char *filename)
{
char buffer[1024];
if(ifile_search(filename))
	return 3; // It is here

if(!access(filename,F_OK))
	return 2;
strcpy(buffer,ifile_prefix);
strcat(buffer,filename);

if(!access(buffer,F_OK))
	return 1;

return 0;
}

/*
 *  Get file date
 */

time_t igettime(char *filename)
{
struct IFILEENTRY *ife;
struct stat sb;
char buffer[1024];

ife=ifile_search(filename);
if(ife)
	return ife->time;

if(!stat(filename,&sb))
	return sb.st_mtime;

strcpy(buffer,ifile_prefix);
strcat(buffer,filename);
if(!stat(buffer,&sb))
	return sb.st_mtime;

return 0;
}


/*
 *  Attach a file to the VFS system
 */

int ifile_attach(char *filename, char *truename, long length, long origin)
{
struct stat sbuf;
int num,ctr;

//if(bsearch(filename,ifiletab,ifilenum,sizeof(struct IFILEENTRY),BSCMP))
//	return 0; // Already there, don't bother

if(stat(truename,&sbuf))
	return 0;  // Can't open file!  Abandon

num = -1;
// If it already exists, override it
for(ctr=0;ctr<ifilenum;ctr++)
	if(!istricmp(ifiletab[ctr].name,filename))
		{
		num=ctr;
		break;
		}

// If it doesn't exist, use a new number
if(num == -1)
	num = ifilenum++;

if(ifilenum > MAXIFILES)
	ithe_panic("ifile_attach: Too many files",filename);

ifiletab[num].name = filename;
ifiletab[num].org = origin;
ifiletab[num].tfn = truename;
ifiletab[num].len = length;
ifiletab[num].time = sbuf.st_mtime;

// Now, look and see if there are any overriding files
// (we'll do this by date)

if(ithe_devmode && !stat(filename,&sbuf))
	{
	// Remember, stat returns zero on success
	if(difftime(sbuf.st_mtime,ifiletab[num].time) > 0.0)
		{
		// The new file is later.  Use it instead
		ifiletab[num].name = filename;
		ifiletab[num].org = 0;
		ifiletab[num].tfn = filename;
		ifiletab[num].len = sbuf.st_size;
		ifiletab[num].time = sbuf.st_mtime;
		}
	}

return 1;
}

/*
 *  Sort the files after everything has been attached
 */

void ifile_hash()
{
qsort(ifiletab,ifilenum,sizeof(struct IFILEENTRY),QSCMP);
}


/*
 *  Close a file
 */

void iclose(IFILE *ifp)
{
if(!ifp)
	return;
fclose(ifp->fp);
M_free(ifp->truename);
M_free(ifp);
}

void iseek(IFILE *ifp, int offset, int whence)
{
if(!ifp)
	return;

switch(whence)
	{
	case SEEK_SET:
	fseek(ifp->fp,offset+ifp->origin,SEEK_SET);
	break;

	case SEEK_END:
	fseek(ifp->fp,ifp->origin+ifp->length+offset,SEEK_SET);
	break;

	default:
	fseek(ifp->fp,offset,SEEK_CUR);
	break;
	}
}

unsigned char igetc(IFILE *ifp)
{
return fgetc(ifp->fp);
}

unsigned short igetsh_reverse(IFILE *ifp)
{
return ((igetc(ifp)<<8) + igetc(ifp));
}

unsigned short igetsh_native(IFILE *ifp)
{
unsigned short s;
iread((void *)&s,2,ifp);
return s;
}

long igetl_reverse(IFILE *ifp)
{
return ((igetsh_reverse(ifp) << 16) + igetsh_reverse(ifp));
}

long igetl_native(IFILE *ifp)
{
return getw(ifp->fp);
}


unsigned long igetlu_reverse(IFILE *ifp)
{
return ((igetsh_reverse(ifp) << 16) + igetsh_reverse(ifp));
}

unsigned long igetlu_native(IFILE *ifp)
{
return getw(ifp->fp);
}

uqword igetll_reverse(IFILE *ifp)
{
return (((uqword)igetl_reverse(ifp) << 32) + igetl_reverse(ifp));
}

uqword igetll_native(IFILE *ifp)
{
uqword q;
iread((void *)&q,8,ifp);
return q;
}

int iread(unsigned char *buf, int l, IFILE *ifp)
{
return fread(buf,1,l,ifp->fp);
}

void iputc(unsigned char c, IFILE *ifp)
{
fputc(c,ifp->fp);
}

void iputsh_reverse(unsigned short s, IFILE *ifp)
{
iputc((s>>8)&0xff,ifp);
iputc(s&0xff,ifp);
}

void iputsh_native(unsigned short s, IFILE *ifp)
{
iwrite((void *)&s,2,ifp);
}

void iputl_reverse(long l, IFILE *ifp)
{
iputsh_reverse((l>>16)&0xffff,ifp);
iputsh_reverse(l&0xffff,ifp);
}

void iputl_native(long l, IFILE *ifp)
{
putw(l,ifp->fp);
}

void iputlu_reverse(unsigned long l, IFILE *ifp)
{
iputsh_reverse((l>>16)&0xffff,ifp);
iputsh_reverse(l&0xffff,ifp);
}

void iputlu_native(unsigned long l, IFILE *ifp)
{
putw(l,ifp->fp);
}

void iputll_reverse(uqword q, IFILE *ifp)
{
iputl_reverse((q>>32)&0xffffffff,ifp);
iputl_reverse(q&0xffffffff,ifp);
}

void iputll_native(uqword q, IFILE *ifp)
{
iwrite((void *)&q,8,ifp);
}

int iwrite(unsigned char *buf, int l, IFILE *ifp)
{
return fwrite(buf,1,l,ifp->fp);
}

unsigned int itell(IFILE *ifp)
{
return ftell(ifp->fp)-ifp->origin;
}

unsigned int ifilelength(IFILE *ifp)
{
return ifp->length;
}

int ieof(IFILE *ifp)
{
if(ftell(ifp->fp) >= ifp->origin+ifp->length)
	return 1;
return 0;
}

// Get a home subdirectory for config settings

void ihome(char *path)
{
char *home;
char temp[1024];

home=getenv("HOME");
if(!home)
	home=".";

#if defined(__SOMEUNIX__) || defined(__DJGPP__)
	sprintf(temp,"%s/.ire",home);
	mkdir(temp,S_IRUSR|S_IWUSR|S_IXUSR);
	strcat(temp,"/");
#else
	sprintf(temp,"%s/ire",home);
	mkdir(temp);
	strcat(temp,"/");
	#ifdef _WIN32
	strwinslash(temp);
	#endif
#endif

strcpy(path,temp);
}


/*
 *
 *
 *  Internal functions (not for human consumption)
 *
 *
 */

static int BSCMP(const void *key, const void *elem)
{
return istricmp((char *)key,((struct IFILEENTRY *)elem)->name);
}

static int QSCMP(const void *elem1, const void *elem2)
{
return istricmp(((struct IFILEENTRY *)elem1)->name,((struct IFILEENTRY *)elem2)->name);
}

/*
 *  Do a binary search of the VFS table for a filename
 */

struct IFILEENTRY *ifile_search(char *filename)
{
struct IFILEENTRY *ife;
char buffer[1024];

// If we have a file prefix, e.g. project directory, search with that first

if(ifile_prefix)
	{
	strcpy(buffer,ifile_prefix);
//	strcat(buffer,"/");
	strcat(buffer,filename);
	ife = bsearch(buffer,ifiletab,ifilenum,sizeof(struct IFILEENTRY),BSCMP);
	if(ife)
		return ife;
	}

return bsearch(filename,ifiletab,ifilenum,sizeof(struct IFILEENTRY),BSCMP);
}


#ifdef __BEOS__		// BEOS is not well
int getw(FILE *fp)
{
int w;
fread(&w,1,4,fp);
return w;
}

void putw(int w, FILE *fp)
{
fwrite(&w,1,4,fp);
}
#endif
