/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Tom Newton, 2006
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#include <sqlite3.h>

#define PLUGIN_NAME "SQLite IMSpector logging plugin"
#define PLUGIN_SHORT_NAME "SQLite"

#define CREATE_TABLE "CREATE TABLE IF NOT EXISTS messages ( " \
	"id integer PRIMARY KEY AUTOINCREMENT, " \
	"timestamp integer NOT NULL, " \
	"clientaddress text NOT NULL, " \
	"protocolname text NOT NULL, " \
	"outgoing integer NOT NULL, " \
	"type integer NOT NULL, " \
	"localid text NOT NULL, " \
	"remoteid text NOT NULL, " \
	"filtered integer NOT NULL, " \
	"categories text NOT NULL, " \
	"eventdata blob NOT NULL );"

#define INSERT_STATEMENT "INSERT INTO messages " \
	"(id, timestamp, clientaddress, protocolname, outgoing, type, localid, remoteid, filtered, categories, eventdata) " \
	"VALUES (NULL, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
#define NO_FIELDS 10

sqlite3 *db;
sqlite3_stmt *insertstatement;
bool localdebugmode = false;

extern "C"
{
	bool initloggingplugin(struct loggingplugininfo &ploggingplugininfo,
		class Options &options, bool debugmode);
	void closeloggingplugin(void);
	int logevents(std::vector<struct imevent> &imevents);
};

bool initloggingplugin(struct loggingplugininfo &loggingplugininfo,
	class Options &options, bool debugmode)
{
	std::string sqlitefile = options["sqlite_file"];

	if (sqlitefile.empty()) return false;

	localdebugmode = debugmode;

	loggingplugininfo.pluginname = PLUGIN_NAME;
	
	int rc = sqlite3_open(sqlitefile.c_str(), &db);
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't open DB, Error: %s", sqlite3_errmsg(db));
		return false;
	}

	rc = sqlite3_exec (db, CREATE_TABLE, NULL, NULL, NULL);
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't create table, Error: %s", sqlite3_errmsg(db));
		return false;
	}
	
	rc = sqlite3_prepare(db, INSERT_STATEMENT, -1, &insertstatement, 0);	
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_preapre(), Error: %s", sqlite3_errmsg(db));
		return false;
	}
	
	if (sqlite3_bind_parameter_count(insertstatement) != NO_FIELDS)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_bind_parameter_count(), Error: invalid paramter count");
		return false;
	}
	
	return true;
}

void closeloggingplugin(void)
{
	sqlite3_close(db);

	return;
}

/* The main plugin function. See loggingplugin.cpp. */
int logevents(std::vector<struct imevent> &imevents)
{
	for (std::vector<struct imevent>::iterator i = imevents.begin();
		i != imevents.end(); i++)
	{
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Logging one event");

		if (sqlite3_bind_int(insertstatement, 1, (*i).timestamp) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_text(insertstatement, 2, (*i).clientaddress.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_text(insertstatement, 3, (*i).protocolname.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_int(insertstatement, 4, (*i).outgoing) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_int(insertstatement, 5, (*i).type) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_text(insertstatement, 6, (*i).localid.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_text(insertstatement, 7, (*i).remoteid.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_int(insertstatement, 8, (*i).filtered) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_text(insertstatement, 9, (*i).categories.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
			return 1;
		if (sqlite3_bind_text(insertstatement, 10, (*i).eventdata.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
			return 1;

		while (sqlite3_step(insertstatement) == SQLITE_DONE);
		if (sqlite3_reset(insertstatement) != SQLITE_OK) return 2;
	}

	return 0;
}
