#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <iiimp-data.h>
#include <iiimp-opcode.h>

#include "iiimp-dataP.h"

IIIMP_open_reply *
iiimp_open_reply_new(
    IIIMP_data_s *	        data_s,
    IIIMP_card32                ns_id,
    IIIMP_card32                ns_errno)
{
    IIIMP_open_reply *	data;

    data = (IIIMP_open_reply *) malloc(sizeof(IIIMP_open_reply));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_read_reply *
iiimp_read_reply_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        size,
    IIIMP_card32        ns_errno,
    IIIMP_card32 *      object)
{
    IIIMP_read_reply *        data;
 
    data = (IIIMP_read_reply *) malloc(sizeof(IIIMP_read_reply));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4 + size + PAD(size);
    data->ns_id = ns_id;
    data->size = size;
    data->ns_errno = ns_errno;
    data->object = object;

    return data;
}

IIIMP_write_reply *
iiimp_write_reply_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        size,
    IIIMP_card32        ns_errno)
{
    IIIMP_write_reply *	data;

    data = (IIIMP_write_reply *)malloc(sizeof (IIIMP_write_reply));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->size = size;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_close_reply *
iiimp_close_reply_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_close_reply *	data;

    data = (IIIMP_close_reply *)malloc(sizeof (IIIMP_close_reply));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_seek_reply *
iiimp_seek_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    IIIMP_card32        offset)
{
    IIIMP_seek_reply *     data;

    data = (IIIMP_seek_reply *)malloc(sizeof (IIIMP_seek_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->offset = offset;

    return data;
}

IIIMP_opendir_reply *
iiimp_opendir_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    IIIMP_card32        nitems,
    IIIMP_utf8string *  d_name)
{
    IIIMP_opendir_reply *     data;
    IIIMP_utf8string *        pu8;
    size_t                    nbyte;

    data = (IIIMP_opendir_reply *)malloc(sizeof (IIIMP_opendir_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    if (d_name) {
	for (nbyte = 0, pu8 = d_name; NULL != pu8; pu8 = pu8->next) {
	    nbyte += pu8->nbyte;
	}
    }
    data->nbyte = 4 + 4 + 4 + 4 + nbyte;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->nitems = nitems;
    data->d_name = d_name;

    return data;
}

IIIMP_closedir_reply *
iiimp_closedir_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_closedir_reply *     data;
    data = (IIIMP_closedir_reply *)malloc(sizeof (IIIMP_closedir_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_mkdir_reply *
iiimp_mkdir_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_mkdir_reply *     data;

    data = (IIIMP_mkdir_reply *)malloc(sizeof (IIIMP_mkdir_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_rmdir_reply *
iiimp_rmdir_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_rmdir_reply *     data;

    data = (IIIMP_rmdir_reply *)malloc(sizeof (IIIMP_rmdir_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_symlink_reply *
iiimp_symlink_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_symlink_reply *     data;

    data = (IIIMP_symlink_reply *)malloc(sizeof (IIIMP_symlink_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_stat_reply *
iiimp_stat_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    iiimp_stat_struct * stat_buf)
{
    IIIMP_stat_reply *     data;

    data = (IIIMP_stat_reply *)malloc(sizeof (IIIMP_stat_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + 4 + sizeof(iiimp_stat_struct);
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->stat_buf = stat_buf;

    return data;
}

IIIMP_lstat_reply *
iiimp_lstat_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    iiimp_stat_struct * stat_buf)
{
    IIIMP_lstat_reply *     data;

    data = (IIIMP_lstat_reply *)malloc(sizeof (IIIMP_lstat_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + 4 + sizeof(iiimp_stat_struct);
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->stat_buf = stat_buf;

    return data;
}

IIIMP_fstat_reply *
iiimp_fstat_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    iiimp_stat_struct * stat_buf)
{
    IIIMP_fstat_reply *     data;

    data = (IIIMP_fstat_reply *)malloc(sizeof (IIIMP_fstat_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + sizeof(iiimp_stat_struct);
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->stat_buf = stat_buf;

    return data;
}

IIIMP_creat_reply *
iiimp_creat_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_creat_reply *     data;

    data = (IIIMP_creat_reply *)malloc(sizeof (IIIMP_creat_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_lseek_reply *
iiimp_lseek_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    IIIMP_card32        offset)
{
    IIIMP_lseek_reply *     data;

    data = (IIIMP_lseek_reply *)malloc(sizeof (IIIMP_lseek_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->offset = offset;

    return data;
}

IIIMP_unlink_reply *
iiimp_unlink_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_unlink_reply *     data;

    data = (IIIMP_unlink_reply *)malloc(sizeof (IIIMP_unlink_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_rename_reply *
iiimp_rename_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_rename_reply *     data;

    data = (IIIMP_rename_reply *)malloc(sizeof (IIIMP_rename_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_fcntl_reply *
iiimp_fcntl_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno,
    IIIMP_card32        return_value)
{
    IIIMP_fcntl_reply *     data;

    data = (IIIMP_fcntl_reply *)malloc(sizeof (IIIMP_fcntl_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;
    data->return_value = return_value;

    return data;
}

IIIMP_truncate_reply *
iiimp_truncate_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_truncate_reply *     data;

    data = (IIIMP_truncate_reply *)malloc(sizeof (IIIMP_truncate_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

IIIMP_ftruncate_reply *
iiimp_ftruncate_reply_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        ns_errno)
{
    IIIMP_ftruncate_reply *     data;
    data = (IIIMP_ftruncate_reply *)malloc(sizeof (IIIMP_ftruncate_reply));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->ns_errno = ns_errno;

    return data;
}

void
iiimp_open_reply_delete(IIIMP_data_s * data_s, IIIMP_open_reply *m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_read_reply_delete(IIIMP_data_s * data_s, IIIMP_read_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_write_reply_delete(IIIMP_data_s * data_s, IIIMP_write_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_close_reply_delete(IIIMP_data_s * data_s, IIIMP_close_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_seek_reply_delete(IIIMP_data_s * data_s, IIIMP_seek_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_opendir_reply_delete(IIIMP_data_s * data_s, IIIMP_opendir_reply * m)
{
    if (NULL == m) return;
    iiimp_utf8string_list_delete(data_s, m->d_name);
    free(m);
    return;
}

void
iiimp_closedir_reply_delete(IIIMP_data_s * data_s, IIIMP_closedir_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_mkdir_reply_delete(IIIMP_data_s * data_s, IIIMP_mkdir_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_rmdir_reply_delete(IIIMP_data_s * data_s, IIIMP_rmdir_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_symlink_reply_delete(IIIMP_data_s * data_s, IIIMP_symlink_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_stat_reply_delete(IIIMP_data_s * data_s, IIIMP_stat_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_fstat_reply_delete(IIIMP_data_s * data_s, IIIMP_fstat_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}


void
iiimp_lstat_reply_delete(IIIMP_data_s * data_s, IIIMP_lstat_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_creat_reply_delete(IIIMP_data_s * data_s, IIIMP_creat_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_lseek_reply_delete(IIIMP_data_s * data_s, IIIMP_lseek_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_unlink_reply_delete(IIIMP_data_s * data_s, IIIMP_unlink_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_rename_reply_delete(IIIMP_data_s * data_s, IIIMP_rename_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_fcntl_reply_delete(IIIMP_data_s * data_s, IIIMP_fcntl_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_truncate_reply_delete(IIIMP_data_s * data_s, IIIMP_truncate_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_ftruncate_reply_delete(IIIMP_data_s * data_s, IIIMP_ftruncate_reply * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_open_reply_pack(
    IIIMP_data_s *	data_s,
    IIIMP_open_reply *        m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_read_reply_pack(
    IIIMP_data_s *	data_s,
    IIIMP_read_reply *  m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;
    int                 i, count, object_size;
    IIIMP_card32 *      buf;

    object_size = m->size + PAD((m->size) % 4);
    count = (object_size / 4);

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->size, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    buf = (IIIMP_card32 *)m->object;
    for (i=0; i<count; i++) {
	PUTU32(*(buf + i), rest, p, data_s->byte_swap); 
    }

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_write_reply_pack(
    IIIMP_data_s *	data_s,
    IIIMP_write_reply *       m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->size, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_close_reply_pack(
    IIIMP_data_s *	data_s,
    IIIMP_close_reply *       m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_seek_reply_pack(
    IIIMP_data_s *      data_s,
    IIIMP_seek_reply *  m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);
    PUTU32(m->offset, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_opendir_reply_pack(
    IIIMP_data_s *         data_s,
    IIIMP_opendir_reply *  m,
    size_t *               nbyte,
    uchar_t **             ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);
    PUTU32(m->nitems, rest, p, data_s->byte_swap);
    iiimp_utf8string_list_pack(data_s, m->d_name, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_closedir_reply_pack(
    IIIMP_data_s *          data_s,
    IIIMP_closedir_reply *  m,
    size_t *                nbyte,
    uchar_t **              ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_mkdir_reply_pack(
    IIIMP_data_s *        data_s,
    IIIMP_mkdir_reply *   m,
    size_t *              nbyte,
    uchar_t **            ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_rmdir_reply_pack(
    IIIMP_data_s *       data_s,
    IIIMP_rmdir_reply *  m,
    size_t *             nbyte,
    uchar_t **           ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_symlink_reply_pack(
    IIIMP_data_s *         data_s,
    IIIMP_symlink_reply *  m,
    size_t *               nbyte,
    uchar_t **             ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_stat_reply_pack(
    IIIMP_data_s *       data_s,
    IIIMP_stat_reply *   m,
    size_t *             nbyte,
    uchar_t **           ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    if (m->ns_id > 0) {
	PUTU32(m->stat_buf->s_mode, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_nlink, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_uid, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_gid, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_atime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_mtime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_ctime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_blksize, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_attr, rest, p, data_s->byte_swap);

	PUTU64(m->stat_buf->s_dev, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_rdev, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_ino, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_size, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_blocks, rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_lstat_reply_pack(
    IIIMP_data_s *        data_s,
    IIIMP_lstat_reply *   m,
    size_t *              nbyte,
    uchar_t **            ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    if (m->ns_id > 0) {
	PUTU32(m->stat_buf->s_mode, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_nlink, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_uid, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_gid, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_atime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_mtime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_ctime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_blksize, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_attr, rest, p, data_s->byte_swap);

	PUTU64(m->stat_buf->s_dev, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_rdev, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_ino, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_size, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_blocks, rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_fstat_reply_pack(
    IIIMP_data_s *        data_s,
    IIIMP_fstat_reply *   m,
    size_t *              nbyte,
    uchar_t **            ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    if (m->ns_id > 0) {
	PUTU32(m->stat_buf->s_mode, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_nlink, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_uid, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_gid, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_atime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_mtime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_ctime, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_blksize, rest, p, data_s->byte_swap);
	PUTU32(m->stat_buf->s_attr, rest, p, data_s->byte_swap);

	PUTU64(m->stat_buf->s_dev, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_rdev, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_ino, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_size, rest, p, data_s->byte_swap);
	PUTU64(m->stat_buf->s_blocks, rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_creat_reply_pack(
    IIIMP_data_s *        data_s,
    IIIMP_creat_reply *   m,
    size_t *              nbyte,
    uchar_t **            ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_lseek_reply_pack(
    IIIMP_data_s *       data_s,
    IIIMP_lseek_reply *  m,
    size_t *             nbyte,
    uchar_t **           ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);
    PUTU32(m->offset, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_unlink_reply_pack(
    IIIMP_data_s *        data_s,
    IIIMP_unlink_reply *  m,
    size_t *              nbyte,
    uchar_t **            ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_rename_reply_pack(
    IIIMP_data_s *        data_s,
    IIIMP_rename_reply *  m,
    size_t *              nbyte,
    uchar_t **            ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_fcntl_reply_pack(
    IIIMP_data_s *      data_s,
    IIIMP_fcntl_reply * m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);
    PUTU32(m->return_value, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_truncate_reply_pack(
    IIIMP_data_s *          data_s,
    IIIMP_truncate_reply *  m,
    size_t *                nbyte,
    uchar_t **              ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

void
iiimp_ftruncate_reply_pack(
    IIIMP_data_s *           data_s,
    IIIMP_ftruncate_reply *  m,
    size_t *                 nbyte,
    uchar_t **               ptr)
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUT32(m->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;
    
    return;
}   

IIIMP_open_reply *
iiimp_open_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_open_reply *  data;
    size_t              rest;
    const uchar_t *     p;
  
    rest = nbyte_max;
    p = *ptr;
    
    if ((*nbyte < rest) || (rest < 4)) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }
    
    data = (IIIMP_open_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_read_reply *
iiimp_read_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_read_reply *  data;
    size_t              rest;
    const uchar_t *     p;
    int                 i, count, object_size;
    IIIMP_card32 *      buf;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    object_size =  (rest - 4 - 4 - 4 -4);
    count = (object_size / 4);

    data = (IIIMP_read_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->object = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->size, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    buf = (IIIMP_card32 *)malloc(data->size);
    if (NULL == buf) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    for (i=0; i<count; i++) {
	GETU32(*(buf + i), rest, p, data_s->byte_swap);
    }
    data->object = (void *)buf;

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_write_reply *
iiimp_write_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_write_reply * data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_write_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->size, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_close_reply *
iiimp_close_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_close_reply *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_close_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_seek_reply *
iiimp_seek_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_seek_reply *  data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_seek_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);
    GETU32(data->offset, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_opendir_reply *
iiimp_opendir_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_opendir_reply *  data;
    size_t                 rest;
    const uchar_t *        p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_opendir_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);
    GETU32(data->nitems, rest, p, data_s->byte_swap);

    if (data->nitems > 0) {
	data->d_name = iiimp_utf8string_list_unpack(data_s, &rest, &p, rest);
	if (NULL == data->d_name) {
	    iiimp_opendir_reply_delete(data_s, data);
	    return NULL;
	}

    }

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_closedir_reply *
iiimp_closedir_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_closedir_reply *  data;
    size_t                  rest;
    const uchar_t *         p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_closedir_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_mkdir_reply *
iiimp_mkdir_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_mkdir_reply * data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_mkdir_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_rmdir_reply *
iiimp_rmdir_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_rmdir_reply * data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_rmdir_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_symlink_reply *
iiimp_symlink_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_symlink_reply * data;
    size_t                rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < 4)) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_symlink_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_stat_reply *
iiimp_stat_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_stat_reply *  data;
    size_t              rest;
    const uchar_t *     p;
    iiimp_stat_struct * im_stat;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_stat_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    if (data->ns_id <= 0) {
	data->stat_buf = (iiimp_stat_struct *)NULL;
    } else {
	im_stat = (iiimp_stat_struct *) calloc(1, sizeof(iiimp_stat_struct));
	GETU32(im_stat->s_mode, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_nlink, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_uid, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_gid, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_atime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_mtime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_ctime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_blksize, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_attr, rest, p, data_s->byte_swap);
    
	GETU64(im_stat->s_dev, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_rdev, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_ino, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_size, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_blocks, rest, p, data_s->byte_swap);
	data->stat_buf = im_stat;
    }

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_lstat_reply *
iiimp_lstat_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_lstat_reply * data;
    size_t              rest;
    const uchar_t *     p;
    iiimp_stat_struct * im_stat;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_lstat_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    if (data->ns_id <= 0) {
	data->stat_buf = (iiimp_stat_struct *)NULL;
    } else {
	im_stat = (iiimp_stat_struct *) calloc(1, sizeof(iiimp_stat_struct));
	GETU32(im_stat->s_mode, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_nlink, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_uid, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_gid, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_atime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_mtime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_ctime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_blksize, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_attr, rest, p, data_s->byte_swap);
    
	GETU64(im_stat->s_dev, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_rdev, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_ino, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_size, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_blocks, rest, p, data_s->byte_swap);
	data->stat_buf = im_stat;
    }

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_fstat_reply *
iiimp_fstat_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_fstat_reply * data;
    size_t              rest;
    const uchar_t *     p;
    iiimp_stat_struct * im_stat;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_fstat_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    if (data->ns_id <= 0) {
	data->stat_buf = (iiimp_stat_struct *)NULL;
    } else {
	im_stat = (iiimp_stat_struct *) calloc(1, sizeof(iiimp_stat_struct));
	GETU32(im_stat->s_mode, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_nlink, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_uid, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_gid, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_atime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_mtime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_ctime, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_blksize, rest, p, data_s->byte_swap);
	GETU32(im_stat->s_attr, rest, p, data_s->byte_swap);
    
	GETU64(im_stat->s_dev, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_rdev, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_ino, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_size, rest, p, data_s->byte_swap);
	GETU64(im_stat->s_blocks, rest, p, data_s->byte_swap);
	data->stat_buf = im_stat;
    }

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_creat_reply *
iiimp_creat_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_creat_reply * data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_creat_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_lseek_reply *
iiimp_lseek_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_lseek_reply * data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_lseek_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);
    GETU32(data->offset, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_unlink_reply *
iiimp_unlink_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_unlink_reply * data;
    size_t               rest;
    const uchar_t *      p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_unlink_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_rename_reply *
iiimp_rename_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_rename_reply * data;
    size_t               rest;
    const uchar_t *      p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_rename_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_fcntl_reply *
iiimp_fcntl_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_fcntl_reply *  data;
    size_t               rest;
    const uchar_t *      p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_fcntl_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);
    GETU32(data->return_value, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_truncate_reply *
iiimp_truncate_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_truncate_reply * data;
    size_t                 rest;
    const uchar_t *        p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_truncate_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_ftruncate_reply *
iiimp_ftruncate_reply_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_ftruncate_reply * data;
    size_t                  rest;
    const uchar_t *         p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_ftruncate_reply *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GET32(data->ns_errno, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

void
iiimp_open_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_open_reply *  m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_read_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_read_reply *  m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tsize = %d\n", m->size);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_write_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_write_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tsize = %d\n", m->size);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_close_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_close_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_seek_reply_print(
    IIIMP_data_s *     data_s,
    IIIMP_seek_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fprintf(data_s->print_fp, "\toffset = %d\n", m->offset);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_opendir_reply_print(
    IIIMP_data_s *        data_s,
    IIIMP_opendir_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fprintf(data_s->print_fp, "\tnitems = %d\n", m->nitems);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\td_name = ");
    iiimp_utf8string_list_print(data_s, m->d_name);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_closedir_reply_print(
    IIIMP_data_s *         data_s,
    IIIMP_closedir_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_mkdir_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_mkdir_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_rmdir_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_rmdir_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_symlink_reply_print(
    IIIMP_data_s *        data_s,
    IIIMP_symlink_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_stat_reply_print(
    IIIMP_data_s *     data_s,
    IIIMP_stat_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_lstat_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_lstat_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_fstat_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_fstat_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_creat_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_creat_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_lseek_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_lseek_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fprintf(data_s->print_fp, "\toffset = %d\n", m->offset);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_unlink_reply_print(
    IIIMP_data_s *       data_s,
    IIIMP_unlink_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_rename_reply_print(
    IIIMP_data_s *       data_s,
    IIIMP_rename_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_fcntl_reply_print(
    IIIMP_data_s *      data_s,
    IIIMP_fcntl_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fprintf(data_s->print_fp, "\treturn_value = %d\n", m->return_value);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_truncate_reply_print(
    IIIMP_data_s *         data_s,
    IIIMP_truncate_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_ftruncate_reply_print(
    IIIMP_data_s *          data_s,
    IIIMP_ftruncate_reply * m
)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tns_errno = %d\n", m->ns_errno);
    (void)fputc('\n', data_s->print_fp);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
