#include <config.h>
#include <stdio.h>
#include "IMSvrCfgFile.hh"
#include "IMLog.hh"

static const int config_file_line_max = 4096;
static const char *config_delimiter = " \t";

void
IMSvrCfgFile::initialize()
{
    adddir("ListenAddress", &IMSvrCfgFile::add_listen_address, 1);
    adddir("DefaultPermission", &IMSvrCfgFile::set_default_permission, 1);
    adddir("AccessControl", &IMSvrCfgFile::set_access_control, 2);
    adddir("AllowSystemUser", &IMSvrCfgFile::set_allow_system_user, 1);
    adddir("UserPermission", &IMSvrCfgFile::set_user_permission, -1);
}

bool
IMSvrCfgFile::
add_listen_address(DirArgs &args)
{
    string address, service;
    string::size_type fpos;
    DirArgs::iterator it1 = args.begin();

    fpos = it1->rfind(':');
    if (fpos == string::npos) {
	address = *it1;
	service = get_strval(IMSvrCfg::PORT);
    } else {
	address = it1->substr(0, fpos);
	service = it1->substr(fpos + 1);
    }
    if (address.size() == 0) {
	LOG_ERROR("Invalid AddListenAddress:%s", it1->c_str());
	return false;
    }

    listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::INET, address, service));

    LOG_DEBUG("AddListenAddress:%s", it1->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_access_control(DirArgs &args)
{
    DirArgs::iterator it1 = args.begin();
    DirArgs::iterator it2 = it1;
    it2++;
    IMAuth::access_type at = get_access_type(*it1);

    if (at == IMAuth::UNKNOWN) return false;
    get_usermgr(ptarget)->set_entry(it2->c_str(), at);

    LOG_DEBUG("AccessControl(%s):%s", it1->c_str(), it2->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_default_permission(DirArgs &args)
{
    DirArgs::iterator it1 = args.begin();
    IMAuth::access_type at = get_access_type(*it1);

    if (at == IMAuth::UNKNOWN) return false;
    get_usermgr(ptarget)->set_default_entry(at);

    LOG_DEBUG("SetDefaultPermission(%s)", it1->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_allow_system_user(DirArgs &args)
{
    IMAuth::access_type at = get_access_type(*(args.begin()));
    if (at == IMAuth::UNKNOWN) return false;
    get_usermgr(ptarget)->set_systemuser_permission(at);
    LOG_DEBUG("AllowSystemUser(%s)", args.begin()->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_user_permission(DirArgs &args)
{
    DirArgs::iterator it;
    IMAuth::access_type at;
    string::size_type pos;
    string username, password;

    it = args.begin();
    if (it == args.end()) return false;
    at = get_access_type(*it);
    if (at == IMAuth::UNKNOWN) return false;
    LOG_DEBUG("UserPermission(%s)", it->c_str());
    it++;
    if (it == args.end()) return false;
    for (; it != args.end(); it++) {
	username = *it;
	LOG_DEBUG("  add user:%s", username.c_str());
	pos = username.find(':');
	if (pos == string::npos) {
	    get_usermgr(ptarget)->add_user(username.c_str(),
					   NULL, at);
	} else {
	    password = username.substr(pos + 1);
	    username = username.substr(0, pos);
	    get_usermgr(ptarget)->add_user(username.c_str(),
					   password.c_str(), at);
	}
    }
    return true;
}

IMAuth::access_type
IMSvrCfgFile::
get_access_type(
    string &key
)
{
    if (key.compare("deny") == 0)
	return IMAuth::DENY;
    else if (key.compare("permit") == 0)
	return IMAuth::PERMIT;
    else if (key.compare("checkuser") == 0)
	return IMAuth::CHECKUSER;
    else if (key.compare("password") == 0)
	return IMAuth::PASSWORD;
    else
	return IMAuth::UNKNOWN;
}

bool
IMSvrCfgFile::
readconfig()
{
    FILE *fp;
    char buf[config_file_line_max];
    char token[config_file_line_max];
    char *p;
    int i, argsnum, lineno;
    bool flag = true;
    DirectiveMap::iterator it;

    fp = fopen(filename.c_str(), "rw");
    if (!fp) return true;
    lineno = 0;
    while (fgets(buf, sizeof(buf), fp))	{
	lineno++;
	if (p = strchr(buf, '\n')) *p = '\0';
	if (buf[0] == '#') continue;
	p = strtok(buf, config_delimiter);
	if (!p) continue;

	it = dirmap.find(p);
	if (it == dirmap.end()) {
	    LOG_ERROR("Unknown directive (%d):%s\n", lineno, p);
	    flag = false;
	    break;
	}
	argsnum = it->second.argsnum;
	DirArgs args;
	if (argsnum >= 0) {
	    for (i = 0; i < argsnum; i++) {
		p = strtok(NULL, config_delimiter);
		if (!p) {
		    LOG_ERROR("too few arguments in directive (%d):%s\n",
			      lineno, it->first);
		}
		args.push_back(string(p));
	    }
	} else {
	    while (p = strtok(NULL, config_delimiter)) {
		args.push_back(string(p));
	    }
	}
	DirHandler h = it->second.h;
	if (!(this->*h)(args)) {
	    LOG_ERROR("Directive error (%d):%s", lineno, it->first);
	    flag = false;
	    break;
	}
    }

    fclose(fp);
    return flag;
}

void
IMSvrCfgFile::
adddir(
    const char *dir,
    DirHandler h,
    int argsnum
)
{
    Directive d;
    d.argsnum = argsnum;
    d.h = h;
    dirmap[dir] = d;
}

bool
IMSvrCfgFile::configure(
    IMSvr *pimsvr
)
{
    ptarget = pimsvr;
    /* ...configuration ... */
    if (!readconfig()) return false;
    if (!listenaddrvec.empty()) {
	config_listenaddress(pimsvr, listenaddrvec);
	setstr(IMSvrCfg::HOSTNAME, "");
    }

    return true;
}

IMSvrCfgFile::IMSvrCfgFile(
    IMSvrCfg *pbase,
    const char *x_filename
) :
    IMSvrCfg(*pbase)
{
    filename = x_filename;
    initialize();
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
