# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module defines the values of reference.

This module contains the following classes and function:
- MobilMeshModel
- MobilMeshTestCase
"""

#-------------------------------------------------------------------------------
# Library modules import
#-------------------------------------------------------------------------------

import sys, unittest

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.Common import *
import Base.Toolbox as Tool
from Base.XMLvariables import Variables, Model
from Base.XMLmodel import ModelTest
from Pages.OutputControlModel import OutputControlModel

#-------------------------------------------------------------------------------
# Mobil Mesh model class
#-------------------------------------------------------------------------------

class MobilMeshModel(Model):
    """
    Manage the input/output markups in the xml doc about mobil mesh
    """
    def __init__(self, case):
        """
        Constructor.
        """
        self.case = case

        self.node_models = self.case.xmlGetNode('thermophysical_models')
        self.node_ale    = self.node_models.xmlInitChildNode('ale_method', 'status')
        
        self.out = OutputControlModel(self.case)


    def _defaultInitialValues(self):
        """
        Return in a dictionnary which contains default values.
        """
        default = {}
        default['nalinf']  = 0
        default['nalimx']  = 1
        default['epalim']  = 1e-05
        default['iortvm']  = 'isotrop'
        default['ale_method']  = 'off'
    
        return default

    def _setVariablesandProperties(self):
        """
        Set variables and properties if ALE method is activated.
        """
        self.node_ale.xmlInitChildNode('variable', name='mesh_velocity_U', label='mesh_u')
        self.node_ale.xmlInitChildNode('variable', name='mesh_velocity_V', label='mesh_v')
        self.node_ale.xmlInitChildNode('variable', name='mesh_velocity_W', label='mesh_w')
        self.node_ale.xmlInitChildNode('property', name='mesh_viscosity_1', label='mesh_vi1')
        self._updateNodeViscosity()
        
        
    def _updateNodeViscosity(self):
        """
        Update properties beyond mesh visosity is isotrope or not.
        """
        if self.getViscosity() == 'orthotrop':
            self.node_ale.xmlInitChildNode('property', name='mesh_viscosity_2', label='mesh_vi2')
            self.node_ale.xmlInitChildNode('property', name='mesh_viscosity_3', label='mesh_vi3')
        else:
            node1 = self.node_ale.xmlGetChildNode('property', name='mesh_viscosity_2')
            node2 = self.node_ale.xmlGetChildNode('property', name='mesh_viscosity_3')
            if node1:
                node1.xmlRemoveNode()
            if node2:
                node2.xmlRemoveNode()


##    def setMethod(self, answer):
##        """
##        Set method of activation of ALE into xml file.
##        """
##        self.isOnOff(answer)
##        typ = ''
##        typ = self.out.getTypePostMeshes()
##        self.node_ale['status'] = answer
##       if answer == 'on':
##            if typ not in ('10', '11', '12'):
##                typ = '10'
##            self._setVariablesandProperties()
##        else:
##            if typ not in ('0', '1', '2'):
##                typ = '0'
##        self.out.setTypePostMeshes(typ) 
##
##
##    def getMethod(self):
##        """
##        Get method of activation of ALE from xml file.
##        """
##        if self.node_ale['status'] == '':
##            status = 'off'
##            self.setMethod(status)
##        else:
##            status = self.node_ale['status']
##        return status

    def getMethod(self):
        """
        Get status on balise "ALE" from xml file
        """
        status = self.node_ale['status']
        if not status:
            status = self._defaultInitialValues()['ale_method']
            self.setMethod(status)
        return status


    def setMethod(self, status):
        """
        Put status on balise "ALE" in xml file
        """
        self.isOnOff(status)
        typ = ''
        typ = self.out.getTypePostMeshes()
        self.node_ale['status'] = status    
        if status == 'on':
            if typ not in ('10', '11', '12'):
                typ = '10'
            self._setVariablesandProperties()
        else:
            if typ not in ('0', '1', '2'):
                typ = '0'
        self.out.setTypePostMeshes(typ)

    def setSubIterations(self, value):
        """
        Set value of fluid initialization sub iterations into xml file.
        """ 
        self.isInt(value)
        self.isGreaterOrEqual(value, 0)
        self.node_ale.xmlSetData('fluid_initialization_sub_iterations', value)


    def getSubIterations(self):
        """
        Get value of fluid initialization sub iterations from xml file.
        """ 
        nalinf = self.node_ale.xmlGetInt('fluid_initialization_sub_iterations')
        if not nalinf:
            nalinf = self._defaultInitialValues()['nalinf']
            self.setSubIterations(nalinf)
        return nalinf


    def setMaxIterations(self, value):
        """
        Set value of maximum of iteration if implicitation into xml file.
        """ 
        self.isInt(value)
        self.isGreaterOrEqual(value, 1)
        self.node_ale.xmlSetData('max_iterations_implicitation', value)


    def getMaxIterations(self):
        """
        Get value of maximum of iteration if implicitation from xml file.
        """ 
        nalimx = self.node_ale.xmlGetInt('max_iterations_implicitation')
        if not nalimx:
            nalimx = self._defaultInitialValues()['nalimx']
            self.setMaxIterations(nalimx )
        return nalimx


    def setPrecision(self, value):
        """
        Set value of precision of implicitation into xml file.
        """ 
        self.isGreater(value, 0.0)
        self.node_ale.xmlSetData('implicitation_precision', value)


    def getPrecision(self):
        """
        Get value of precision of implicitation from xml file.
        """ 
        epalim = self.node_ale.xmlGetDouble('implicitation_precision')
        if not epalim:
            epalim = self._defaultInitialValues()['epalim']
            self.setPrecision(epalim)
        return epalim


    def setViscosity(self, value):
        """
        Set value of mesh viscosity into xml file.
        """ 
        self.isInList(value, ['isotrop', 'orthotrop'])
        node = self.node_ale.xmlInitChildNode('mesh_viscosity')
        node['type'] = value
        self._updateNodeViscosity()


    def getViscosity(self):
        """
        Get value of mesh viscosity from xml file.
        """
        iortvm = self._defaultInitialValues()['iortvm']
        node = self.node_ale.xmlGetChildNode('mesh_viscosity')
        if node :
            iortvm = node['type']
        else:
            self.setViscosity(iortvm)

        return iortvm


#-------------------------------------------------------------------------------
# MobilMesh Model test case
#-------------------------------------------------------------------------------


class MobilMeshTestCase(ModelTest):
    """
    """
    def checkMobilMeshInstantiation(self):
        """Check whether the TurbulenceModel class could be instantiated"""
        model = None
        model = MobilMeshModel(self.case)
        assert model != None, 'Could not instantiate MobilMeshModel'
        
    def checkGetandSetMethod(self):
        """Check whether the MobilMeshModel class could be set and get method"""
        mdl = MobilMeshModel(self.case)
        mdl.setMethod('on')
        doc = """<ale_method status="on">
                    <variable label="mesh_u" name="mesh_velocity_U"/>
                    <variable label="mesh_v" name="mesh_velocity_V"/>
                    <variable label="mesh_w" name="mesh_velocity_W"/>
                    <property label="mesh_vi1" name="mesh_viscosity_1"/>
                    <mesh_viscosity type="isotrop"/>
                 </ale_method>"""
        assert mdl.node_ale == self.xmlNodeFromString(doc),\
            'Could not set mobil mesh model method'
        assert mdl.getMethod() == 'on',\
            'Could not get mobil mesh model method'

    def checkGetandSetSubIterations(self):
        """Check whether the MobilMeshModel class could be set and get sub iterations"""
        mdl = MobilMeshModel(self.case)
        mdl.setMethod('on')
        mdl.setSubIterations(12)
##        
        doc = """<ale_method status="on">
                    <variable label="mesh_u" name="mesh_velocity_U"/>
                    <variable label="mesh_v" name="mesh_velocity_V"/>
                    <variable label="mesh_w" name="mesh_velocity_W"/>
                    <property label="mesh_vi1" name="mesh_viscosity_1"/>
                    <mesh_viscosity type="isotrop"/>
                    <fluid_initialization_sub_iterations>12</fluid_initialization_sub_iterations>
                </ale_method>"""
        assert mdl.node_ale == self.xmlNodeFromString(doc),\
            'Could not set mobil mesh model sub iterations'
        assert mdl.getSubIterations() == 12,\
            'Could not get mobil mesh model sub iteration'


    def checkGetandSetMaxIterations(self):
        """Check whether the MobilMeshModel class could be set and get max iterations"""
        mdl = MobilMeshModel(self.case)
        mdl.setMethod('on')
        mdl.setMaxIterations(99)

        doc = """<ale_method status="on">
                    <variable label="mesh_u" name="mesh_velocity_U"/>
                    <variable label="mesh_v" name="mesh_velocity_V"/>
                    <variable label="mesh_w" name="mesh_velocity_W"/>
                    <property label="mesh_vi1" name="mesh_viscosity_1"/>
                    <mesh_viscosity type="isotrop"/>
                    <max_iterations_implicitation>99</max_iterations_implicitation>
                </ale_method>"""
        assert mdl.node_ale == self.xmlNodeFromString(doc),\
            'Could not set mobil mesh model max iterations'
        assert mdl.getMaxIterations() == 99,\
            'Could not get mobil mesh model max iteration'

    def checkGetandSetPrecision(self):
        """Check whether the MobilMeshModel class could be set and get precision"""
        mdl = MobilMeshModel(self.case)
        mdl.setMethod('on')
        mdl.setPrecision(0.001)
        
        doc = """<ale_method status="on">
                    <variable label="mesh_u" name="mesh_velocity_U"/>
                    <variable label="mesh_v" name="mesh_velocity_V"/>
                    <variable label="mesh_w" name="mesh_velocity_W"/>
                    <property label="mesh_vi1" name="mesh_viscosity_1"/>
                    <mesh_viscosity type="isotrop"/>
                    <implicitation_precision>0.001</implicitation_precision>
                </ale_method>"""
        assert mdl.node_ale == self.xmlNodeFromString(doc),\
            'Could not set mobil mesh model precision'
        assert mdl.getPrecision() == 0.001,\
            'Could not get mobil mesh model precision'

    def checkGetandSetViscosity(self):
        """Check whether the MobilMeshModel class could be set and get viscosity"""
        mdl = MobilMeshModel(self.case)
        mdl.setMethod('on')
        mdl.setViscosity('orthotrop')

        doc = """<ale_method status="on">
                    <variable label="mesh_u" name="mesh_velocity_U"/>
                    <variable label="mesh_v" name="mesh_velocity_V"/>
                    <variable label="mesh_w" name="mesh_velocity_W"/>
                    <property label="mesh_vi1" name="mesh_viscosity_1"/>
                    <mesh_viscosity type="orthotrop"/>
                    <property label="mesh_vi2" name="mesh_viscosity_2"/>
                    <property label="mesh_vi3" name="mesh_viscosity_3"/>
                </ale_method>"""
        assert mdl.node_ale == self.xmlNodeFromString(doc),\
            'Could not set mobil mesh model visocity type'
        assert mdl.getViscosity() == 'orthotrop',\
            'Could not get mobil mesh model viscosity type'

def suite():
    testSuite = unittest.makeSuite(MobilMeshTestCase, "check")
    return testSuite

def runTest():
    print "MobilMeshTestCase"
    runner = unittest.TextTestRunner()
    runner.run(suite())

#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
