# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module defines the lagrangian two phase flow modelling management.

This module contains the following classes and function:
- LagrangianModel
- LagrangianTestCase
"""

#-------------------------------------------------------------------------------
# Library modules import
#-------------------------------------------------------------------------------

import sys, unittest

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.Common import *
import Base.Toolbox as Tool
from Base.XMLvariables import Model

#-------------------------------------------------------------------------------
# lagrangian model class
#-------------------------------------------------------------------------------

class LagrangianModel(Model):
    """
    """
    def __init__(self, case):
        """
        Constructor.
        """
        self.case = case

        nModels         = self.case.xmlGetNode('thermophysical_models')
        self.node_gas   = nModels.xmlGetNode('gas_combustion',  'model')
        self.node_coal  = nModels.xmlGetNode('pulverized_coal', 'model')
        self.node_joule = nModels.xmlGetNode('joule_effect',    'model')
        self.node_therm = nModels.xmlGetNode('thermal_scalar',  'model')

        self.node_lagr = self.case.xmlGetNode('lagrangian')

        self.lagrangianModel = ('off',
                                'one_way',
                                'two_way',
                                'frozen')

#    def getTurbulenceModel(self):
#        """
#	Return the turbulence model. Warning, this method doesn't handle
#	the initial default value of the turbulence model. Therefore, in
#	the XMLmodel class, the order of the call TurbulenceModel, and
#        LagrangianModel should not be change.
#	"""
#	turb = nModels.xmlGetNode('turbulence',  'model')
#	if not turb: exit(1)
#	model = turb['model']
#	if not model: exit(1)
#	return model
 

    def defaultLagrangianValues(self):
        """
        Return in a dictionnary which contains default values.
        """
        default = {}
        default['model'] = "off"

        return default


    def getAllLagrangianModels(self):
        """
        Return all defined lagrangian models in a tuple.
        """
        return self.lagrangianModel


    def lagrangianModelsList(self):
        """
        Create a tuple with the lagrangian models allowed
        by the calculation features.
        """
        from TurbulenceModel import TurbulenceModel
        model = TurbulenceModel(self.case).getTurbulenceModel()
        del TurbulenceModel
        if model not in ('off',
                         'k-epsilon',
                         'k-epsilon-PL',
                         'Rij-epsilon',
                         'Rij-SSG',
                         'v2f-phi',
                         'k-omega-SST'):
            return ('off',)
        else:
            return self.lagrangianModel
        


    def setLagrangianModel(self, model):
        """
        Update the lagrangian model markup from the XML document.
        """
        self.isInList(model, self.lagrangianModelsList())
        self.node_lagr['model'] = model
        
        if model == 'off':
            import CoalCombustionModel
            coal = CoalCombustionModel.CoalCombustionModel(self.case).getCoalCombustionModel()
            if coal == 'coal_lagr':
                CoalCombustionModel.CoalCombustionModel(self.case).setCoalCombustion('off')
            del CoalCombustionModel


    def getLagrangianModel(self):
        """
        Return the current lagrangian model.
        """
        model = self.node_lagr['model']
        if model not in self.lagrangianModelsList():
            model = self.defaultLagrangianValues()['model']
            self.setLagrangianModel(model)

        return model


#-------------------------------------------------------------------------------
# Lagrangian test case
#-------------------------------------------------------------------------------


class LagrangianTestCase(unittest.TestCase):
    """
    """
    def setUp(self):
        """
        This method is executed before all "check" methods.
        """
        from Base.XMLengine import Case
        from Base.XMLinitialize import XMLinit
        self.case = Case()
        XMLinit(self.case)


    def tearDown(self):
        """
        This method is executed after all "check" methods.
        """
        del self.case


    def checkLagrangianInstantiation(self):
        """
        Check whether the LagrangianModel class could be instantiated
        """
        model = None
        model = LagrangianModel(self.case)
        
        assert model != None, 'Could not instantiate LagrangianModel'


    def checkLagrangianModelsList(self):
        """
        Check whether the LagrangianModelsList could be get
        """
        mdl = LagrangianModel(self.case)
        self.case.xmlGetNode('turbulence')['model'] = 'LES_Smagorinsky'
        assert mdl.lagrangianModelsList() == ('off',),\
               'Could not use the lagrangianModelsList method'


def suite():
    testSuite = unittest.makeSuite(LagrangianTestCase, "check")
    return testSuite

    
def runTest():
    print "LagrangianTestCase A COMPLETER *********."
    runner = unittest.TextTestRunner()
    runner.run(suite())


#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
