# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007 Canonical Limited.
#   Author: Robert Collins.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""System wide hooks to trigger dbus events from bzr activity."""


from bzrlib.branch import Branch
from bzrlib.smart.server import SmartTCPServer

import activity


def install_hooks():
    """Install the dbus hooks into bzrlib."""
    Branch.hooks.install_named_hook(
        'post_change_branch_tip', on_post_change_branch_tip,
        'Announcing on branch change on D-Bus')
    SmartTCPServer.hooks.install_named_hook(
        'server_started', on_server_start,
        'Registering server URL mapping')
    SmartTCPServer.hooks.install_named_hook(
        'server_stopped', on_server_stop,
        'Unregistering server mapping')


def on_post_change_branch_tip(params):
    """Announce the new head revision of the branch to dbus."""
    activity.Activity().advertise_branch(params.branch)


def on_server_start(local_urls, public_url):
    """Add the servers local and public urls to the session Broadcaster."""
    for local_url in local_urls:
        activity.Activity().add_url_map(local_url, public_url)


def on_server_stop(local_urls, public_url):
    """The server has shutdown, so remove the servers local and public urls."""
    for local_url in local_urls:
        activity.Activity().remove_url_map(local_url, public_url)
