# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Rick Spencer rick.spencer@canonical.com
#This program is free software: you can redistribute it and/or modify it 
#under the terms of the GNU General Public License version 3, as published 
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but 
#WITHOUT ANY WARRANTY; without even the implied warranties of 
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along 
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import sys
import os
import gtk
from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import Record

from bughugger.bughuggerconfig import getdatapath

class PreferencesBughuggerDialog(gtk.Dialog):
    __gtype_name__ = "PreferencesBughuggerDialog"
    prefernces = {}

    def __init__(self):
        """__init__ - This function is typically not called directly.
        Creation of a PreferencesBughuggerDialog requires redeading the associated ui
        file and parsing the ui definition extrenally,
        and then calling PreferencesBughuggerDialog.finish_initializing().

        Use the convenience function NewPreferencesBughuggerDialog to create
        NewAboutBughuggerDialog objects.
        """

        pass

    def finish_initializing(self, builder):
        """finish_initalizing should be called after parsing the ui definition
        and creating a AboutBughuggerDialog object with it in order to finish
        initializing the start of the new AboutBughuggerDialog instance.
        """

        #get a reference to the builder and set up the signals
        self.builder = builder
        self.builder.connect_signals(self)

        #set up couchdb and the preference info
        self.__db_name = "bughugger"
        self.__database = CouchDatabase(self.__db_name, create=True)
        self.__preferences = None
        self.__key = None

        #set the record type and then initalize the preferences
        self.__record_type = "http://wiki.ubuntu.com/Quickly/RecordTypes/Bughugger/Preferences"
        self.__preferences = self.get_preferences()

        for pref in self.__preferences.keys():
            if not pref.startswith("_") and pref != "record_type":
                chk_name = "checkbutton_" + pref
                pref_val = self.__preferences[pref]
                self.builder.get_object(chk_name).set_active(pref_val)

    def get_preferences(self):
        """get_preferences  -returns a dictionary object that contain
        preferences for bughugger. Creates a couchdb record if
        necessary.
        """

        if self.__preferences == None: #the dialog is initializing
            self.__load_preferences()
            
        #if there were no saved preference, this 
        return self.__preferences

    def __load_preferences(self):
        #TODO: add prefernces to the self.__preferences dict
        #default preferences that will be overwritten if some are saved
        self.__preferences = {"record_type":self.__record_type,
                              "use_production": True,
                              "auto_connect":True
                             }
        
        results = self.__database.get_records(record_type=self.__record_type, create_view=True)
       
        if len(results.rows) == 0:
            #no preferences have ever been saved
            #save them before returning
            self.__key = self.__database.put_record(Record(self.__preferences))
        else:
            self.__preferences = results.rows[0].value
            self.__key = results.rows[0].value["_id"]
        
    def __save_preferences(self):
        self.__database.update_fields(self.__key, self.__preferences)

    def ok(self, widget, data=None):
        """ok - The user has elected to save the changes.
        Called before the dialog returns gtk.RESONSE_OK from run().
        """

        #make any updates to self.__preferences here
        #self.__preferences["preference1"] = "value2"
        for pref in self.__preferences.keys():
            if not pref.startswith("_") and pref != "record_type":
                chk_name = "checkbutton_" + pref
                pref_val = self.builder.get_object(chk_name).get_active()
                self.__preferences[pref] = pref_val

        self.__save_preferences()

    def cancel(self, widget, data=None):
        """cancel - The user has elected cancel changes.
        Called before the dialog returns gtk.RESPONSE_CANCEL for run()
        """

        #restore any changes to self.__preferences here
        pass

def NewPreferencesBughuggerDialog():
    """NewPreferencesBughuggerDialog - returns a fully instantiated
    PreferencesBughuggerDialog object. Use this function rather than
    creating a PreferencesBughuggerDialog instance directly.
    """

    #look for the ui file that describes the ui
    ui_filename = os.path.join(getdatapath(), 'ui', 'PreferencesBughuggerDialog.ui')
    if not os.path.exists(ui_filename):
        ui_filename = None

    builder = gtk.Builder()
    builder.add_from_file(ui_filename)
    dialog = builder.get_object("preferences_bughugger_dialog")
    dialog.finish_initializing(builder)
    return dialog

if __name__ == "__main__":
    dialog = NewPreferencesBughuggerDialog()
    dialog.show()
    gtk.main()

