#!/usr/bin/env python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Rick Spencer rick.spencer@canonical.com
#This program is free software: you can redistribute it and/or modify it 
#under the terms of the GNU General Public License version 3, as published 
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but 
#WITHOUT ANY WARRANTY; without even the implied warranties of 
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along 
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import pygtk
pygtk.require('2.0')
import gtk
import os
import gobject
from quickly.widgets.asynch_task_progressbox import AsynchTaskProgressBox
import LaunchpadUtils

from bughugger.bughuggerconfig import getdatapath


class FindDialog():
 """ FindDialog: Base class for dialogs that search Launchpad.
 After the dialog returns, check get_selected to retrieve the selcted object

 """
 def __init__(self, title, label, find_task):
  """Create a FindDialog

  arguments:
  title -- string to display as dialog title
  label -- string to display as label  for search field
  find_task -- function to call when user clicks the search button, runs async

  """
  self.find_task = find_task

  self.builder = gtk.Builder()

  ui_filename = os.path.join(getdatapath(), 'ui', 'FindDialog.ui')
  self.builder.add_from_file(ui_filename)
  self.dialog = self.builder.get_object("dialog")
  self.dialog.set_title(title)
  self.dialog.connect("delete-event",self.on_delete)
  self.builder.get_object("search_label").set_text(label)
  self.builder.connect_signals(self)
  self.dialog.show()

  self.dialog.set_response_sensitive(1, False)

  scroll = gtk.ScrolledWindow()
  scroll.show()

  self.treeview = gtk.TreeView()
  self.treeview.show()
  empty_model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_PYOBJECT)
  self.treeview.set_model(empty_model)
  self.treeview.set_size_request(400,250)
  self.treeview.connect("cursor_changed", self.__cursor_changed)
  self.treeview.connect("row_activated", self.__row_clicked)
  scroll.add(self.treeview)


  self.builder.get_object("vbox1").pack_start(scroll)

  rend = gtk.CellRendererText()
  rend.set_property("editable",False)

  col = gtk.TreeViewColumn("Results", rend, text=0)
  col.set_resizable(False)
  col.set_sort_column_id(0)
  self.treeview.append_column(col)

 #Following are the functions needed to make this look like a
 #gtk.Dialog to callers

 def on_delete(self, widget, data=None):
  self.dialog.response(-6)
  return True

 def connect(self, signal, func):
  self.dialog.connect(signal,func)

 def show(self):
  self.dialog.show()

 def hide(self):
  self.dialog.hide()

 def run(self):
  self.dialog.run()


 def text_changed(self, widget, data = None):
  """text_changed: signal handler for the search text field.
  This handler is used to change the search button sensitivity based on
  whether the user had entered text to search for.

  """
  text_entry = self.builder.get_object("text_entry")
  can_search = False
  if len(text_entry.get_text().strip()) > 0:
   can_search = True
 
  self.builder.get_object("search_button").set_sensitive(can_search)
  

 def __cursor_changed(self, widget, data = None):
  """__cursor_changed: signal handler for cursor changes in the treeview.
     Toggles the Ok button sensitivity based on whether there is a slections.

  """
  if len(self.treeview.get_selection().get_selected_rows()) > 0:
     self.dialog.set_response_sensitive(1, True)
  else:
   self.dialog.set_response_sensitive(1, False)
  
 def find(self, widget, data = None):
  """find: starts progress UI and executes the search supplied in
  the find_task used to instantiate the class.

  """
  text_entry = self.builder.get_object("text_entry")
  search_button = self.builder.get_object("search_button")

  text = text_entry.get_text()
  text = text.strip()
  if len(text) > 0:
   params = {"text":text}
   find_asynch =  AsynchTaskProgressBox(self.find_task, params, False)
   find_asynch.connect("complete",self.__find_complete)
   self.builder.get_object("vbox1").pack_end(find_asynch, False, False)
   find_asynch.show()
   find_asynch.start("Searching for " + text)
   text_entry.set_sensitive(False)
   search_button.set_sensitive(False)

 def __find_complete(self, widget, results = None):
  """find_complete: called when the search task completes.
  Sets UI sensitivity and populates the treeview with the results.

  """
  gtk.gdk.threads_enter() #block main ui thread for updating
  #turn off the progress UI
  self.builder.get_object("vbox1").remove(widget)

  text_entry = self.builder.get_object("text_entry")
  search_button = self.builder.get_object("search_button")

  #set button sensitivity appropriately
  text_entry.set_sensitive(True)
  search_button.set_sensitive(True)

  #create the treestore from the returned list of objects
  #the first column will be displayed,
  #the second column is the found object stored to be returned by get_selected
  tree_store = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_PYOBJECT)
  for r in results:
   try:
    name = "%s (%s)" % (r.display_name, r.name)
   except:
    name = r.display_name
   tree_store.append([name,r])

  #put the results into the treeview
  self.treeview.set_model(tree_store)

  #set button sensitity appropriately
  if len(tree_store) < 1:
   self.dialog.set_response_sensitive(1, False)

  gtk.gdk.threads_leave()

 def get_selected(self):
  """get_selected: returns the object stored in the selected row"""
  iter =  self.treeview.get_selection().get_selected()[1]
  return self.treeview.get_model().get_value(iter, 1)
  
 def __row_clicked(self, widget, iter, data = None):
  """__row_clicked: signal handler to interprate the user double clicking
  a row as asking to use the object in that row.

  """
  self.builder.get_object("dialog").response(gtk.RESPONSE_OK)

if __name__ == "__main__":
 w = FindDialog("test",None,None)
 print w.run()
 gtk.main()
