# Written by Henrik Nilsen Omma
# (C) Canonical, Ltd. Licensed under the GPL
""" Keep in mind some filtering happens in launchpadbugs/buglist_filter.py
    It does not just occur in bughelper / bugnumbers
"""

from optparse import OptionParser

try:
    import utils
except:
    import launchpadBugs.utils as utils

def status_importance_case_callback(option, opt_str, value, parser):
    result = list()
    values = value.split(",")
    for v in values:
        if v.lower().endswith("w/") or v.lower().endswith("w/o"):
            result.append(v.lstrip().capitalize())
        else:
            x = v.split()
            result.append(" ".join([i.capitalize() for i in x]))
    setattr(parser.values, option.dest, ",".join(result))

class commandLine:
    def __init__(self, prog=None):
        parser = OptionParser(version=utils.find_version_number())
        if parser.get_prog_name() == "bughelper" or prog == "bughelper":
            parser.set_defaults(attachments=False, sourcepackage=None, 
                                url="", minbug=None, verbose=0,
                                case_sensitive=False,file=None)
            parser.set_usage(usage="""%prog --version
       %prog -p <package> 
       %prog -l <Launchpad-URL>
       %prog -b <bugnumber>
       %prog --help""")

            parser.add_option("-p", "--package", type="string", 
                                    dest="sourcepackage", 
                                    metavar="PACKAGE", 
                                    help="package with bugs to search")
            parser.add_option("-l", "--url", type="string", dest="url", 
                                    metavar="URL", help="bug list URL to search")
            parser.add_option("-b", "--bugnr", type="int", dest="bugnr", 
                                    metavar="BUGNUMBER", help="bug number to try clue with")
            parser.add_option("-m", "--minbug", type="int",
                                    dest="minbug", metavar="BUGNUMBER",
                                    help="only search bugs with bug numbers higher than <number>")
            parser.add_option("-f", "--filterbug", type="string", 
                                    dest="filterbug", metavar="BUGNUMBER[,BUGNUMBER,...]",
                                    help="don't show provided bug number(s)")
            parser.add_option("--status", action="callback", callback=status_importance_case_callback,
                                    dest="status", default="", metavar="STATUS",
                                    type="string", help="filter on provided status")
            parser.add_option("--importance", action="callback", callback=status_importance_case_callback,
                                    dest="importance", default="", metavar="IMPORTANCE",
                                    type="string", help="filter on provided importance")
            parser.add_option("--tag", type="string", dest="tag",
                                    default="", metavar="TAG[,TAG,...]",
                                    help="filter on bug tag(s)")
            parser.add_option("-C", "--closed", action="store_true", 
                                    dest="closed_bugs",
                                    help="also search closed bugs")
            parser.add_option("-d", "--dups", action="store_true", 
                                    dest="duplicates", metavar="duplicates", 
                                    help="also search duplicate bugs")
            parser.add_option("-U", "--upstream", action="store_true", 
                                    dest="upstream",
                                    help="also consider upstream bugs")
            parser.add_option("-A", "--attachments", action="store_true", 
                                    dest="attachments",
                                    help="search in attachments also")
            parser.add_option("-c", "--case-sensitive", action="store_true", 
                                    dest="case_sensitive",
                                    help="case sensitive search - default is off")
            parser.add_option("-v", "--verbose", type="int", dest="verbose",
                                    metavar="<INT>",
                                    help="use inherited clues to a depth of <number>")
            parser.add_option("-t", "--try", type="string", nargs=3, 
                                    dest="try_clue", metavar="<package> <condition> <clue>",
                                    help="search for <condition> and display <clue> when found (in addition to stored clues)")
            parser.add_option("-T", "--try-only", type="string", nargs=3,
                                    dest="try_only_clue", metavar="<package> <condition> <clue>",
                                    help="search for <condition> and display <clue> when found (disregarding stored clues)")
            parser.add_option("-H", "--heuristic", action="store_true", 
                                    dest="heuristic",
                                    help="use all clue files, if a bug has no source \
package assigned or no clue file for the package exists")
            parser.add_option("--format", type="string", dest="format", 
                                    default="plain", metavar="(plain|wiki|html)", 
                                    help="format output type - default is plain")
            parser.add_option("--footer", type="string", dest="footer",
                              default="", metavar="(s|t|st)",
                              help="show (s)tatistical or date/(t)ime information in the footer")
            parser.add_option("--file", type="string", dest="file",
                              default="", help="print to file")
            parser.add_option("--gen-config", action="store_true", 
                              dest="gen_config",
                              help="generate default config file")
            parser.add_option("--lc", type="string", dest="lastcomment",
                              default="", metavar="COMMENT_SPEC",
                              help='filter on author ("u:brian-murray) and / or date ("d:YYYY-MM-DD") of last comment i.e. --lc "u:brian-murray&d:2007-12-04"')
            parser.add_option("--reporter", type="string", dest="reporter",
                              default="", metavar="LAUNCHPAD-LOGIN",
                              help="filter on the reporter")
            parser.add_option("--cookie", type="string", dest="cookie",
                              default="", help="valid cookie-file for authentication")
            parser.add_option("--debug", action="store_true", dest="debug",
                              help="print debug information to stderr")
            parser.add_option("--ignore_conflicts", action="store_true",
                                    dest="ignore_conflicts", 
                                    help="ignore conflicts in filter options")
            parser.add_option("--parsemode", type="string", dest="parsemode",
                              default="text", help="set parse method ('html' or 'text') (default: 'text')")

        if parser.get_prog_name() == "bugnumbers":
            parser.set_defaults(package=None, url="", minbug=None,file=None)
            parser.set_usage(usage="""%prog --version
       %prog -P <project>
       %prog -D <distro>
       %prog -p <package>
       %prog -l <Launchpad-URL>
       %prog --help""")
           
            parser.add_option("-P", "--project", type="string", 
                                    dest="product", metavar="PROJECT",
                                    default="", help="project with bugs to search")
            parser.add_option("-D", "--distro", type="string", 
                                    dest="distro", metavar="DISTRIBUTION",
                                    default="", help="distribution with bugs to search")
            parser.add_option("-p", "--package", type="string", 
                                    dest="sourcepackage", 
                                    metavar="PACKAGE", help="package with bugs to search")
            parser.add_option("-l", "--url", type="string", dest="url", 
                                    metavar="URL", help="bug list URL to search")
            parser.add_option("-m", "--minbug", type="int", dest="minbug",
                                    metavar="BUGNUMBER",
                                    help="only search bugs with numbers higher than <bugnumber>")
            parser.add_option("-f", "--filterbug", type="string", 
                                    dest="filterbug",
                                    metavar="BUGNUMBER[,BUGNUMBER,...]",
                                    help="don't show provided bug number(s)")
            parser.add_option("-U", "--upstream", action="store_true", 
                                    dest="upstream", metavar="upstream", 
                                    help="also consider upstream bugs")
            parser.add_option("-C", "--closed", action="store_true", 
                                    dest="closed_bugs",
                                    help="also search closed bugs")
            parser.add_option("-d", "--dups", action="store_true", 
                                    dest="duplicates",
                                    help="also search duplicates")
            parser.add_option("--count", action="store_true",
                                    dest="count", 
                                    help="count the number of comments") 
            parser.add_option("--status", action="callback", callback=status_importance_case_callback,
                                    dest="status", default="", metavar="STATUS",
                                    type="string", help="filter on provided status")
            parser.add_option("--importance", action="callback", callback=status_importance_case_callback,
                                    dest="importance", default="", metavar="IMPORTANCE",
                                    type="string", help="filter on provided importance")
            parser.add_option("--component", type="string", dest="component",
                                    default="", metavar="COMPONENT",
                                    help="filter on provided component")
            parser.add_option("--tag", type="string", dest="tag",
                                    default="", metavar="TAG[,TAG,...]",
                                    help="filter on bug tag(s)")
            parser.add_option("--reporter", type="string", dest="reporter",
                                    default="", metavar="LAUNCHPAD-LOGIN",
                                    help="filter on the reporter")
            parser.add_option("--subscriber", type="string", dest="subscriber",
                                    default="", metavar="LAUNCHPAD-LOGIN",
                                    help="filter on a subscriber")
            parser.add_option("--commenter", type="string", dest="commenter",
                                    default="", metavar="LAUNCHPAD-LOGIN",
                                    help="filter on a commenter")
            parser.add_option("--supervisor", type="string", dest="supervisor",
                                    default="", metavar="LAUNCHPAD-LOGIN",
                                    help="filter on the bug supervisor")
            parser.add_option("--nd", type="string", dest="no_duplicates",
                                    default="", metavar='"(<|>)<INT>"',
                                    help="filter on the number of duplicates")
            parser.add_option("--ns", type="string", dest="no_subscribers",
                                    default="", metavar='"(<|>)<INT>"',
                                    help="filter on the number of direct subscribers")
            parser.add_option("--nc", type="string", dest="no_comments",
                                    default="", metavar='"(<|>)<INT>"',
                                    help="filter on the number of comments")
            parser.add_option("--na", type="string", dest="no_attachments",
                                    default="", metavar='"(<|>)<INT>"',
                                    help="filter on the number of attachments")
            parser.add_option("--lc", type="string", dest="lastcomment",
                                    default="", metavar="COMMENT_SPEC",
                                    help='''filter on the last comment's author (u:LAUNCHPAD-LOGIN) and/or date (d:YYYY-MM-DD) of last comment e.g. --lc "u:brian-murray&d:2007-12-04"''')
            parser.add_option("--dc", type="string", dest="date_created",
                                    default="", metavar="DATE_SPEC",
                                    help='filter on the date a bug task was created')
            parser.add_option("--du", type="string", dest="date_updated",
                                    default="", metavar="DATE_SPEC",
                                    help='filter on the date a bug was updated')
            parser.add_option("--patch", action="store_true", 
                                    dest="patch",
                                    help="filter for bugs containing a patch attached")
            parser.add_option("--branch", action="store_true",
                                    dest="branch",
                                    help="filter for bugs containing a related bzr branch")
            parser.add_option("--needs-forwarding", action="store_true",
                                    dest="needs_forwarding",
                                    help="filter for bugs needing forwarding upstream")
            parser.add_option("--fixed-upstream", action="store_true",
                                    dest="fixed_upstream",
                                    help="filter for bugs that are resolved upstream")
            parser.add_option("--sort", type="string", dest="sort",
                                    default="", metavar="(-)(nr|status|importance)",
                                    help="sort output by bugnumber, status or importance; add '-' (e.g. '-nr') for reverse order")
            parser.add_option("--stats", action="store_true",
                                    dest="stats", 
                                    help="show statistical information (status and importance of bugs)")
            parser.add_option("--format", type="string", dest="format", 
                                    default="plain", metavar="(plain|numbers|url|wiki|bugday|csv|html)", 
                                    help="format output - default is plain")
            parser.add_option("--footer", type="string", dest="footer",
                                    default="", metavar="(s|t|st)",
                                    help="print (s)tatistical or date/(t)ime information in the footer")
            parser.add_option("--file", type="string", dest="file",
                              default="", help="print to file")
            parser.add_option("--cookie", type="string", dest="cookie",
                              default="", help="valid cookie-file for authentication")
            parser.add_option("--debug", action="store_true", dest="debug",
                              help="print debug information to stderr")
            parser.add_option("--ignore_conflicts", action="store_true",
                                    dest="ignore_conflicts", 
                                    help="ignore conflicts in filter options")
            parser.add_option("--parsemode", type="string", dest="parsemode",
                              default="text", help="set parse method ('html' or 'text') (default: 'text')")


        if parser.get_prog_name() == "bugxml":
            parser.set_defaults(text='', cluefile='', add='')
            parser.set_usage(usage="""%prog --version
       %prog -e <text>
       %prog -v <clue file>
       %prog -a <package> <condition> <clue>""")

            parser.add_option("-e", type="string", dest="text", metavar="text", 
                                    help="escape provided string for use in XML files")
            parser.add_option("-v", "--validate", type="string", dest="cluefile", 
                                    metavar="clue", help="validate clue file syntax")
            parser.add_option("-a", "--simple-add", type="string", nargs=3, 
                                     dest="add", metavar="<package> <condition> <clue>", 
                                     help="add clue to package name")
            parser.add_option("-d", "--dont-list", type="string", 
                                    dest="dontlist", metavar="BUGNUMBER[,BUGNUMBER,...]", 
                                    help="add a dontlist tag to XML for provided bug number(s)")

        if parser.get_prog_name() == "buginfo":
            parser.set_defaults()
            parser.set_usage(usage="""%prog --version
       %prog [options] <Launchpad-Bug-Number>
       %prog --help""")
            parser.add_option("-v", "--verbose", action="store_true", dest="verbose",
                                    help="verbose")
            parser.add_option("--title", action="store_true",
                                    dest="title", 
                                    help="show title")
            parser.add_option("--reporter", action="store_true",
                                    dest="reporter",
                                    help="show launchpad-login of the reporter")
            parser.add_option("--tags", action="store_true",
                                    dest="tags",
                                    help="show all bug tags")
            parser.add_option("--isduplicate", action="store_true",
                                    dest="isduplicate",
                                    help="show what the bug is a duplicate of")
            parser.add_option("--comments", action="store_true",
                                    dest="comments",
                                    help="show number and properties of comments")
            parser.add_option("--countcomments", action="store_true",
                                    dest="countcomments", 
                                    help="show the quantity of comments")
            parser.add_option("--countcommenters", action="store_true",
                                    dest="countcommenters", 
                                    help="show the quantity of unique commenters")
            parser.add_option("-c", "--comment", type="int", dest="comment",
                                    metavar="COMMENTNUMBER", help="show properties of a comment")
            parser.add_option("--author", action="store_true",
                                    dest="author", 
                                    help="show launchpad-login of the author of a comment")
            parser.add_option("--date", action="store_true",
                                    dest="date", 
                                    help="show date of a comment")
            parser.add_option("--lastcomment", action="store_true",
                                    dest="lastcomment", 
                                    help="show properties of the last comment")
            parser.add_option("--debug", action="store_true", dest="debug",
                              help="print debug information to stderr")
            parser.add_option("--cookie", type="string", dest="cookie",
                              default="", help="valid cookie-file for authentication")
            parser.add_option("--parsemode", type="string", dest="parsemode",
                              default="text", help="set parse method ('html' or 'text') (default: 'text')")
        # options with arguments
        self.parser = parser
        if not prog:
            (self.options, self.args) = parser.parse_args()

