/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BsC (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FtpCopier.h"
#include "Shared.h"
#include "InfoField.h"
#include "Config.h"
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qprogressbar.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qfileinfo.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const int     FtpCopier::ProgressMinSize  = 300;
const QString FtpCopier::BoxInfoTitle     = tr( "Source and destination" );
const QString FtpCopier::SrcLabel         = tr( "Source:" );
const QString FtpCopier::DstLabel         = tr( "Destination:" );
const QString FtpCopier::BoxProgressTitle = tr( "Progress" );
const QString FtpCopier::FtpCopyCaption   = tr( "FTP copy processing" );
const QString FtpCopier::OpenReadError    = tr( "Can't open the file for reading:\n%1" );
const QString FtpCopier::OpenWriteError   = tr( "Can't open the file for writing:\n%1" );
const QString FtpCopier::CantCopyDir      = tr( "\"%1\" is a directory.\nCan't copy a directory." );


//*******************************************************************
// FtpCopier                                             CONSTRUCTOR
//*******************************************************************
FtpCopier::FtpCopier(	QWidget* const in_parent,
								const FtpServerInfo& in_fsi, 
								const ViewTable::SelectedItems& in_items,
								const QString& in_dir,
								const bool in_src_is_ftp )
: QDialog      ( in_parent )
, d_status     ( new InfoField( this ))
, d_fsi        ( in_fsi )
, d_dir        ( in_dir )
, d_items      ( in_items )
, d_src_is_ftp ( in_src_is_ftp )
, d_fm         ( font() )
, d_ftp        ( this )
, d_idx        ( 0 )
, d_ftp_cmd    ( FTP_CMD_NONE )
{
	setCaption( tr(FtpCopyCaption) );
	setFont( Config::instance()->lfs_default_font()  );

	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin );
	main_layout->setSpacing( Shared::LayoutSpacing );
	//..............................................
	main_layout->addWidget( make_info_area() );
	main_layout->addWidget( make_progress_area() );
	main_layout->addLayout( make_button_area() );
	main_layout->addWidget( d_status );
	//..............................................
	connect( &d_ftp, SIGNAL( stateChanged         (int)      ),
				this  , SLOT  ( state_changed        (int)      ));
	connect( &d_ftp, SIGNAL( dataTransferProgress (int,int)  ),
				this  , SLOT  ( transfer_progress    (int, int) ));
	connect( &d_ftp, SIGNAL( done                 (bool)     ),
				this  , SLOT  ( done                 (bool)     ));
}
// end of FtpCopier

//*******************************************************************
// make_info_area                                            PRIVATE
//*******************************************************************
QWidget* FtpCopier::make_info_area()
{
	QGroupBox* const gbox = new QGroupBox( 2, Qt::Horizontal, tr(BoxInfoTitle), this );
	QFrame* const frame = new QFrame( gbox );
	QGridLayout* const grid_layout = new QGridLayout( frame, 2, 2, 4 );
	grid_layout->setColStretch( 1, Shared::OverStretch );
	//..............................................
	grid_layout->addWidget( new QLabel( tr(SrcLabel), frame ), 0, 0, Qt::AlignRight );
	grid_layout->addWidget( d_src_info = new InfoField( frame ), 0, 1 );
	grid_layout->addWidget( new QLabel( tr(DstLabel), frame ), 1, 0, Qt::AlignRight );
	grid_layout->addWidget( d_dst_info = new InfoField( frame ), 1, 1 );
	//..............................................
	return gbox;
}
// end of make_info_area

//*******************************************************************
// make_progress_area                                        PRIVATE
//*******************************************************************
QWidget* FtpCopier::make_progress_area()
{
	QGroupBox* const gbox = new QGroupBox( 1, Qt::Vertical, tr(BoxProgressTitle), this );
	//..............................................
	d_progress_bar = new QProgressBar( gbox );
	d_progress_bar->setCenterIndicator( TRUE );
	//..............................................
	return gbox;
}
// end of make_progress_area

//*******************************************************************
// make_button_area                                          PRIVATE
//*******************************************************************
QLayout* FtpCopier::make_button_area()
{
	QHBoxLayout* const layout = new QHBoxLayout;
	layout->setMargin( Shared::LayoutMargin );
	layout->setSpacing( Shared::LayoutSpacing );
	//..............................................
	layout->addStretch( Shared::OverStretch );
	layout->addWidget( d_left_btn  = new QPushButton( Shared::RunBtnLabel, this ));
	layout->addWidget( d_right_btn = new QPushButton( Shared::CloseBtnLabel , this ));
	//.............................................
	connect( d_left_btn , SIGNAL( clicked() ), this, SLOT( slot_left_btn()  ));
	connect( d_right_btn, SIGNAL( clicked() ), this, SLOT( slot_right_btn() ));
	
	return layout;
}
// end of make_button_area

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void FtpCopier::polish()
{
	Shared::polish_width( this, 40 );
	state_changed( QFtp::Unconnected );
}
// end of polish

//*******************************************************************
// get_ui_info                                               PRIVATE
//*******************************************************************
/*
bool FtpCopier::get_ui_info( const QString& in_name, int& in_size, bool& in_isdir )
{
	bool retval = FALSE;
	
	vector<QUrlInfo>::const_iterator it = d_ui.begin();
	while( it != d_ui.end() ) {
		if( in_name == (*it).name() ) {
			in_size  = (*it).size();
			in_isdir = (*it).isDir();
			retval   = TRUE;
			break;
		}
		++it;
	}

	return retval;
}
*/
// end of get_ui_size

//*******************************************************************
// get_lfs_info                                              PRIVATE
//*******************************************************************
bool FtpCopier::get_lfs_info( const QString& in_name, int& in_size, bool& in_isdir )
{
	int retval = FALSE;
	
	QFileInfo fi( in_name );
	if( fi.exists() ) {
		in_size  = fi.size();
		in_isdir = fi.isDir();
		retval   = TRUE;
	}
	return retval;
}
// end of get_lfs_size

//*******************************************************************
// next_file                                                 PRIVATE
//*******************************************************************
bool FtpCopier::next_file( QString& out_src, QString& out_dst )
{
	bool retval = FALSE;
	
	d_progress_bar->reset();
	if( FALSE == d_items.empty() ) {
		if( d_idx < d_items.size() ) {
			const ViewTableItem* const item = d_items[d_idx];
					
			// *** FTP -> LFS ***
			if( d_src_is_ftp ) {
				out_src = item->name();
				if( item->is_dir() ) {
					QMessageBox::information( this, tr(FtpCopyCaption), tr(CantCopyDir).arg( out_src ) );
				}
				else {
					out_dst = Shared::make_path( d_dir, item->name() );
					d_progress_bar->setTotalSteps( item->size() );
					display_info( Shared::make_path( d_fsi.dir(), out_src ), out_dst );
					retval = TRUE;
				}
			}
			// *** LFS -> FTP
			else {
				out_src = item->path();
				if( item->is_dir() ) {
					QMessageBox::information( this, tr(FtpCopyCaption), tr(CantCopyDir).arg( out_src ) );
				}
				else {
					out_dst = item->name();
					d_progress_bar->setTotalSteps( item->size() );
					display_info( out_src, Shared::make_path( d_fsi.dir(), out_dst ));
					retval = TRUE;
				}
			}
		}
		else {
			d_ftp_cmd = FTP_CMD_CLOSE;
			d_ftp.close();
		}
	}
	
	return retval;
}
// end of display_next

//*******************************************************************
// display                                                   PRIVATE
//*******************************************************************
void FtpCopier::display_info( const QString& in_src_path, const QString& in_dst_path )
{
	const int MAXLEN = d_src_info->width();

	QString src_path = in_src_path;
	if( d_fm.width( src_path ) > MAXLEN ) {
		Shared::clip_path( d_fm, d_src_info->width(), src_path );
	}
	d_src_info->setText( src_path );
	
	QString dst_path = in_dst_path;
	if( d_fm.width( dst_path ) > MAXLEN ) {
		Shared::clip_path( d_fm, d_dst_info->width(), dst_path );
	}
	d_dst_info->setText( dst_path );

	Shared::idle();
}
// end of display

//*******************************************************************
// slot_left_btn                                        PRIVATE slot
//*******************************************************************
void FtpCopier::slot_left_btn()
{
	d_left_btn->hide();
	d_right_btn->setText( Shared::BreakBtnLabel );
	ftp_connect();
}
// end of slot_left_btn

//*******************************************************************
// slot_right_btn                                       PRIVATE slot
//*******************************************************************
void FtpCopier::slot_right_btn()
{
	( QFtp::Unconnected == d_ftp.state() ) ? reject() : ftp_abort();
}
// end of slot_right_btn

//*******************************************************************
// ftp_connect                                               PRIVATE
//*******************************************************************
void FtpCopier::ftp_connect()
{
	d_ftp_cmd = FTP_CMD_LOGIN;	
	d_ftp.connectToHost( d_fsi.addr() );
	d_ftp.login( d_fsi.user(), d_fsi.pass() );
}
// end of ftp_connect

//*******************************************************************
// ftp_cd                                                    PRIVATE
//*******************************************************************
void FtpCopier::ftp_cd( const QString& in_path )
{
	d_ftp_cmd = FTP_CMD_CD;
	d_ftp.cd( in_path );
}
// end of ftp_cd

//*******************************************************************
// ftp_get                                                   PRIVATE
//*******************************************************************
void FtpCopier::ftp_get()
{
	d_ftp_cmd = FTP_CMD_GET;
	QString src, dst;	

	if( next_file( src, dst ) ) {
		d_current_file.setName( dst );
		if( FALSE == d_current_file.open( IO_WriteOnly ) ) {
			QMessageBox::information( this, tr(FtpCopyCaption), tr(OpenWriteError).arg( dst ) );
			done( FALSE );
		}
		else {
			d_ftp.get( src, &d_current_file );
		}
	}
	else {
		done( FALSE );
	}
}
// end of ftp_get

//*******************************************************************
// ftp_put                                                   PRIVATE
//*******************************************************************
void FtpCopier::ftp_put()
{
	d_ftp_cmd = FTP_CMD_PUT;
	QString src, dst;
	
	if( next_file( src, dst ) ) {
		d_current_file.setName( src );
		if( FALSE == d_current_file.open( IO_ReadOnly ) ) {
			QMessageBox::information( this, tr(FtpCopyCaption), tr(OpenReadError).arg( src ) );
			done( FALSE );
		}
		else {
			d_ftp.put( &d_current_file, dst );
		}
	}
	else {
		done( FALSE );
	}
}
// end of ftp_put

//*******************************************************************
// ftp_abort                                                 PRIVATE
//*******************************************************************
void FtpCopier::ftp_abort()
{
	d_ftp_cmd = FTP_CMD_ABORT;
	d_ftp.abort();
}
// end of ftp_abort

//*******************************************************************
// ftp_close                                                PRIVATE
//*******************************************************************
void FtpCopier::ftp_close()
{
	d_ftp_cmd = FTP_CMD_CLOSE;
	d_ftp.close();
}
// end of ftp_close

//*******************************************************************
// state_changed                                        PRIVATE slot
//*******************************************************************
void FtpCopier::state_changed( int in_state )
{
	switch( in_state ) {
		case QFtp::Unconnected:
			d_status->setText( tr(Shared::FTP_Unconnected) );
			break;
		case QFtp::HostLookup:
			d_status->setText( tr(Shared::FTP_HostLookup) );
			break;
		case QFtp::Connecting:
			d_status->setText( tr(Shared::FTP_Connecting) );
			break;
		case QFtp::Connected:
			d_status->setText( tr(Shared::FTP_Connected) );
			break;
		case QFtp::LoggedIn:
			d_status->setText( tr(Shared::FTP_LoggedIn) );
			break;
		case QFtp::Closing:
			d_status->setText( tr(Shared::FTP_Closing) );
			break;
		default:
			d_status->setText( tr(Shared::FTP_Unknown) );
	}
	Shared::idle();
}
// end of state_changed

//*******************************************************************
// transfer_progress                                    PRIVATE slot
//*******************************************************************
void FtpCopier::transfer_progress( int in_done, int )
{
	d_progress_bar->setProgress( in_done );
	Shared::idle();
}
// end of transfer_progress

//*******************************************************************
// done                                                 PRIVATE slot
//*******************************************************************
void FtpCopier::done( bool in_error )
{
	if( in_error ) {
		if( FTP_CMD_ABORT == d_ftp_cmd ) {
			ftp_close();
		}
		else {
			if( d_ftp_cmd != FTP_CMD_CLOSE ) {
				QMessageBox::critical( this, tr(FtpCopyCaption), d_ftp.errorString() );
				d_ftp.close();
			}
			reject();
		}
	}
	else {
		switch( d_ftp_cmd ) {
			case FTP_CMD_LOGIN:
				ftp_cd( d_fsi.dir() );
				break;
			case FTP_CMD_ABORT:
				ftp_close();
				break;
			case FTP_CMD_CLOSE:
				d_current_file.close();
				reject();
				break;
			case FTP_CMD_CD:
				d_src_is_ftp ? ftp_get() : ftp_put();
				break;
			case FTP_CMD_PUT:
				d_current_file.close();
				++d_idx;
				ftp_put();
				break;
			case FTP_CMD_GET:
				d_current_file.close();
				++d_idx;
				ftp_get();
				break;
			default:
				d_ftp_cmd = FTP_CMD_NONE;
		}
	}
}
// end of done
