
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2009
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Bandwidth limited oscillator code. This is called to generate the waveforms
 * for bandwidth limited complex waves into the global buffer space that can
 * then be referenced by the diverse oscillators used throughout the engine.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "bristolblo.h"

static int init = 1;
int bloflags = 0;

float blosine[BRISTOL_BLO_SIZE];
float blocosine[BRISTOL_BLO_SIZE];
float blosquare[BRISTOL_BLO_SIZE];
float bloramp[BRISTOL_BLO_SIZE];
float blosaw[BRISTOL_BLO_SIZE];
float blotriangle[BRISTOL_BLO_SIZE];

/*
 * Generate the waveforms to the given harmonic reference size. The code could
 * be optimised however it is really only likely to ever be called once at
 * startup or 'intermittently' whilst programmming a synth (depending on the
 * oscillator implementation - most use a private cache stuffed at init time).
 */
void
generateBLOwaveforms(int harmonics, float gn)
{
	int i, j, k;
	float gain;

	printf("generate bandwidth limited waveforms(%i, %2.0f)\n", harmonics, gn);

	if (harmonics == 0) {
		bloflags &= ~BRISTOL_BLO;
		return;
	} else
		bloflags |= BRISTOL_BLO;

	/*
	 * See if we need to stuff our base waves.
	 */
	if (init) {
		init = 0;

		for (i = 0; i < BRISTOL_BLO_SIZE; i++)
		{
			blosine[i] = sinf(2 * M_PI * ((float) i) / BRISTOL_BLO_SIZE) * gn;
			blocosine[i] = cosf(2 * M_PI * ((float) i) / BRISTOL_BLO_SIZE) * gn;
		}
	}

	for (i = 0; i < BRISTOL_BLO_SIZE; i++)
		blosquare[i] = bloramp[i] = blotriangle[i] = 0;

	/*
	 * The square wave:
	 *	sum all odd sine harmonics, gain is reciprocal of harmonic.
	 */
	for (j = 1; j <= harmonics; j++)
	{
		if ((j & 0x01) == 0)
			continue;

		k = 0; gain = 1 / ((float) j);

		for (i = 0; i < BRISTOL_BLO_SIZE; i++)
		{
			blosquare[i] += blosine[k] * gain;
			if ((k += j) >= BRISTOL_BLO_SIZE)
				k -= BRISTOL_BLO_SIZE;
		}
	}

	/*
	 * Ramp and saw, they will be inverted but will only be noticable when
	 * used as LFO. This gives audible crackle at low harmonic counts:
	 *
	 *	sum all odd and subtract all even harmonics, gain is reciprocal of h
	 */
	for (j = 1; j <= harmonics; j++)
	{
		k = 0; gain = 1 / ((float) j);

		if (j & 0x01)
		{
			for (i = 0; i < BRISTOL_BLO_SIZE; i++)
			{
				blosaw[i] += blosine[k] * gain;
				if ((k += j) >= BRISTOL_BLO_SIZE)
					k -= BRISTOL_BLO_SIZE;
			}
		} else {
			for (i = 0; i < BRISTOL_BLO_SIZE; i++)
			{
				blosaw[i] -= blosine[k] * gain;
				if ((k += j) >= BRISTOL_BLO_SIZE)
					k -= BRISTOL_BLO_SIZE;
			}
		}
	}
	/* Invert this for the ramp */
 	for (i = 0; i < BRISTOL_BLO_SIZE; i++)
		bloramp[i] = -blosaw[i];

	/*
	 * Tri:
	 * Sum odd cosine harmonics, gain is reciprocal of square of harmonic
	 */
	k = 0; gain = 1.0;
	for (i = 0; i < BRISTOL_BLO_SIZE; i++)
		blotriangle[i] = blosine[i];

	for (j = 3; j <= harmonics; j+=2)
	{
		k = 0;
		gain = 1.0 / ((float) (j * j));

		for (i = 0; i < BRISTOL_BLO_SIZE; i++)
		{
			blotriangle[i] += blocosine[k] * gain;
			if ((k += j) >= BRISTOL_BLO_SIZE)
				k -= BRISTOL_BLO_SIZE;
		}
	}
}

