
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2009
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Emulator code for the C64 SID audio chip. All the register numbers and bit
 * masks are defined here, the sid.c code will accept the register addressess,
 * parse the register bitmaps into some internal tables and then allow IO to
 * generate all the signals. The SID chip was partly digital (OSC/Env/noise)
 * and partly analogue (filter). This is emulated as integer code and floating
 * point code respectively.
 */

#ifndef _B_SID_DEFS
#define _B_SID_DEFS

#define B_SID_VOICE_1	0
#define B_SID_VOICE_2	1
#define B_SID_VOICE_3	2

#define B_SID_COUNT			32

/* Registers */
#define B_SID_REGISTERS		32
#define B_SID_V1_FREQ_LO	0x00
#define B_SID_V1_FREQ_HI	0x01
#define B_SID_V1_PW_LO		0x02
#define B_SID_V1_PW_HI		0x03
#define B_SID_V1_CONTROL	0x04
#define B_SID_V1_ATT_DEC	0x05
#define B_SID_V1_SUS_REL	0x06

#define B_SID_V2_FREQ_LO	0x07
#define B_SID_V2_FREQ_HI	0x08
#define B_SID_V2_PW_LO		0x09
#define B_SID_V2_PW_HI		0x0a
#define B_SID_V2_CONTROL	0x0b
#define B_SID_V2_ATT_DEC	0x0c
#define B_SID_V2_SUS_REL	0x0d

#define B_SID_V3_FREQ_LO	0x0e
#define B_SID_V3_FREQ_HI	0x0f
#define B_SID_V3_PW_LO		0x10
#define B_SID_V3_PW_HI		0x11
#define B_SID_V3_CONTROL	0x12
#define B_SID_V3_ATT_DEC	0x13
#define B_SID_V3_SUS_REL	0x14

#define B_SID_FILT_LO		0x15
#define B_SID_FILT_HI		0x16
#define B_SID_FILT_RES_F	0x17
#define B_SID_FILT_M_VOL	0x18

/* X/Y Pots can be reassigned as the chip emulator has no support for them */
#define B_SID_X_ANALOGUE	0x19
#define B_SID_Y_ANALOGUE	0x1a

#define B_SID_OSC_3_OUT		0x1b
#define B_SID_ENV_3_OUT		0x1c

/*
 * This register was not defined as a part of SID. Control is needed to emulate
 * like the reset line and give filter type and oscillator mixing options.
 */
#define B_SID_CONTROL		0x1d

/* Bits in different registers - Voice */
#define B_SID_V_NOISE		0x80
#define B_SID_V_SQUARE		0x40
#define B_SID_V_RAMP		0x20
#define B_SID_V_TRI			0x10
#define B_SID_V_TEST		0x08
#define B_SID_V_RINGMOD		0x04
#define B_SID_V_SYNC		0x02
#define B_SID_V_GATE		0x01

/*
 * Bits in different registers - B_SID_FILT_RES_F
 */
#define B_SID_F_RESMASK		0xf0
#define B_SID_F_MIX_EXT		0x08
#define B_SID_F_MIX_V_3		0x04
#define B_SID_F_MIX_V_2		0x02
#define B_SID_F_MIX_V_1		0x01

/*
 * Bits in different registers - B_SID_FILT_M_VOL
 */
#define B_SID_F_3_OFF		0x80
#define B_SID_F_HP			0x40
#define B_SID_F_BP			0x20
#define B_SID_F_LP			0x10
#define B_SID_F_VOLMASK		0x0f

/* Bits in different registers - bristol control */
#define B_SID_C_RESET		0x80
#define B_SID_C_LPF			0x40
#define B_SID_C_MULTI_V		0x20 /* Mix waveforms rather than 'AND' them */
#define B_SID_C_DEBUG		0x10
#define B_SID_C_DEBUG_D		0x10 /* An alias, digital vs analogue debug */ 
#define B_SID_C_DEBUG_A		0x08
#define B_SID_C_PRINT		0x04

/*
 * B_SID diverse analogue (ie, float) functions - emulate the package 
 * characteristics such as oscillator leakage, S/N ratio and analogue IO
 * lines
 */
#define B_SID_IO			9
#define B_SID_INIT			0x00 /* INIT, takes samplerate, returns ID */ 
#define B_SID_DESTROY		0x01
#define B_SID_ANALOGUE_IO	0x02
#define B_SID_GAIN			0x03
#define B_SID_DETUNE		0x04 /* Depth of phase detuning with Osc mixing */ 
#define B_SID_SN_RATIO		0x05 /* Filter noise */
#define B_SID_SN_LEAKAGE	0x06 /* Envelope leakage */
#define B_SID_DC_BIAS		0x07 /* Output DC signal bias */
#define B_SID_OBERHEIM		0x08 /* Feed forword level of filter poles */

/*
 * These were millisecond rates for the envelope attack. Decay and release
 * were specified at 3 times all these values however I think that came more
 * from the scaled (exponential) deltas than the actual value, ie, decay times
 * varied by time since the attack was linear and the decay was somewhat
 * arbitrarily exponential which needs to be studied.
 */
float bSidEnvRates[16] = {
	0.002,
	0.006,
	0.016,
	0.024,
	0.038,
	0.056,
	0.068,
	0.080,
	0.100,
	0.250,
	0.500,
	0.800,
	1.000,
	3.000,
	5.000,
	8.000
};

/*
 * These are frequency table multipliers to convert the phase offsets to and
 * from actual frequencies. The following can be used to define the Oscillator
 * phase accumulator for a given Frequency, hence we can define at least a 
 * midi key to phase accumulation mapping table. This calculation is based on
 * the nominal 1Mhz clock from the original.
 *
 *	Fphase = Fout / 0.059604645
#define B_SID_FREQ_CALC 16777216
 */
#define B_SID_FREQ_DIV 0.059604645
/* Evaluate 1/B_SID_FREQ_DIV since mult is faster than div */
#define B_SID_FREQ_MULT 16.777216

/*
 * This is to scale the 12 bit counter back to 0..1 float
 */
#define B_S_F_SCALER 0.000488520

/*
 * The following table is actually based on a 2MHz divider circuit, it is not
 * actually equally tempered.
 */
unsigned short freqmap[128] = {
	8.175879 / B_SID_FREQ_DIV,
	8.661983 / B_SID_FREQ_DIV,
	9.177091 / B_SID_FREQ_DIV,
	9.722803 / B_SID_FREQ_DIV,
	10.300889 / B_SID_FREQ_DIV,
	10.913456 / B_SID_FREQ_DIV,
	11.562431 / B_SID_FREQ_DIV,
	12.249948 / B_SID_FREQ_DIV,
	12.978416 / B_SID_FREQ_DIV,
	13.750051 / B_SID_FREQ_DIV,
	14.567703 / B_SID_FREQ_DIV,
	15.434004 / B_SID_FREQ_DIV,
	16.351892 / B_SID_FREQ_DIV,
	17.324118 / B_SID_FREQ_DIV,
	18.354349 / B_SID_FREQ_DIV,
	19.445795 / B_SID_FREQ_DIV,
	20.601990 / B_SID_FREQ_DIV,
	21.826912 / B_SID_FREQ_DIV,
	23.125130 / B_SID_FREQ_DIV,
	24.500196 / B_SID_FREQ_DIV,
	25.957170 / B_SID_FREQ_DIV,
	27.500481 / B_SID_FREQ_DIV,
	29.135832 / B_SID_FREQ_DIV,
	30.868008 / B_SID_FREQ_DIV,
	32.704319 / B_SID_FREQ_DIV,
	34.648834 / B_SID_FREQ_DIV,
	36.709373 / B_SID_FREQ_DIV,
	38.892345 / B_SID_FREQ_DIV,
	41.204830 / B_SID_FREQ_DIV,
	43.654778 / B_SID_FREQ_DIV,
	46.251331 / B_SID_FREQ_DIV,
	49.000393 / B_SID_FREQ_DIV,
	51.915688 / B_SID_FREQ_DIV,
	55.002476 / B_SID_FREQ_DIV,
	58.271664 / B_SID_FREQ_DIV,
	61.736015 / B_SID_FREQ_DIV,
	65.410782 / B_SID_FREQ_DIV,
	69.300072 / B_SID_FREQ_DIV,
	73.421440 / B_SID_FREQ_DIV,
	77.784691 / B_SID_FREQ_DIV,
	82.413055 / B_SID_FREQ_DIV,
	87.313370 / B_SID_FREQ_DIV,
	92.506935 / B_SID_FREQ_DIV,
	98.000786 / B_SID_FREQ_DIV,
	103.831375 / B_SID_FREQ_DIV,
	110.011002 / B_SID_FREQ_DIV,
	116.550117 / B_SID_FREQ_DIV,
	123.472031 / B_SID_FREQ_DIV,
	130.821564 / B_SID_FREQ_DIV,
	138.600143 / B_SID_FREQ_DIV,
	146.842880 / B_SID_FREQ_DIV,
	155.569382 / B_SID_FREQ_DIV,
	164.826111 / B_SID_FREQ_DIV,
	174.641983 / B_SID_FREQ_DIV,
	185.013870 / B_SID_FREQ_DIV,
	196.001572 / B_SID_FREQ_DIV,
	207.684326 / B_SID_FREQ_DIV,
	220.022003 / B_SID_FREQ_DIV,
	233.100235 / B_SID_FREQ_DIV,
	246.974564 / B_SID_FREQ_DIV,
	261.643127 / B_SID_FREQ_DIV,
	277.238708 / B_SID_FREQ_DIV,
	293.685760 / B_SID_FREQ_DIV,
	311.138763 / B_SID_FREQ_DIV,
	329.706573 / B_SID_FREQ_DIV,
	349.283966 / B_SID_FREQ_DIV,
	370.096222 / B_SID_FREQ_DIV,
	392.003143 / B_SID_FREQ_DIV,
	415.454926 / B_SID_FREQ_DIV,
	440.140839 / B_SID_FREQ_DIV,
	466.200470 / B_SID_FREQ_DIV,
	494.071136 / B_SID_FREQ_DIV,
	523.286255 / B_SID_FREQ_DIV,
	554.631165 / B_SID_FREQ_DIV,
	587.544067 / B_SID_FREQ_DIV,
	622.277527 / B_SID_FREQ_DIV,
	659.630615 / B_SID_FREQ_DIV,
	698.812012 / B_SID_FREQ_DIV,
	740.192444 / B_SID_FREQ_DIV,
	784.313721 / B_SID_FREQ_DIV,
	831.255188 / B_SID_FREQ_DIV,
	880.281677 / B_SID_FREQ_DIV,
	932.835815 / B_SID_FREQ_DIV,
	988.142273 / B_SID_FREQ_DIV,
	1047.120361 / B_SID_FREQ_DIV,
	1109.877930 / B_SID_FREQ_DIV,
	1175.088135 / B_SID_FREQ_DIV,
	1245.329956 / B_SID_FREQ_DIV,
	1319.261230 / B_SID_FREQ_DIV,
	1398.601440 / B_SID_FREQ_DIV,
	1481.481445 / B_SID_FREQ_DIV,
	1569.858765 / B_SID_FREQ_DIV,
	1663.893555 / B_SID_FREQ_DIV,
	1760.563354 / B_SID_FREQ_DIV,
	1865.671631 / B_SID_FREQ_DIV,
	1976.284546 / B_SID_FREQ_DIV,
	2096.436035 / B_SID_FREQ_DIV,
	2222.222168 / B_SID_FREQ_DIV,
	2352.941162 / B_SID_FREQ_DIV,
	2493.765625 / B_SID_FREQ_DIV,
	2638.522461 / B_SID_FREQ_DIV,
	2801.120361 / B_SID_FREQ_DIV,
	2967.359131 / B_SID_FREQ_DIV,
	3144.654053 / B_SID_FREQ_DIV,
	3333.333252 / B_SID_FREQ_DIV,
	3521.126709 / B_SID_FREQ_DIV,
	/* This was the maximum frequency, the rest use octave foldback */
	3731.343262 / B_SID_FREQ_DIV,

	1976.284546 / B_SID_FREQ_DIV,
	2096.436035 / B_SID_FREQ_DIV,
	2222.222168 / B_SID_FREQ_DIV,
	2352.941162 / B_SID_FREQ_DIV,
	2493.765625 / B_SID_FREQ_DIV,
	2638.522461 / B_SID_FREQ_DIV,
	2801.120361 / B_SID_FREQ_DIV,
	2967.359131 / B_SID_FREQ_DIV,
	3144.654053 / B_SID_FREQ_DIV,
	3333.333252 / B_SID_FREQ_DIV,
	3521.126709 / B_SID_FREQ_DIV,
	3731.343262 / B_SID_FREQ_DIV,

	1976.284546 / B_SID_FREQ_DIV,
	2096.436035 / B_SID_FREQ_DIV,
	2222.222168 / B_SID_FREQ_DIV,
	2352.941162 / B_SID_FREQ_DIV,
	2493.765625 / B_SID_FREQ_DIV,
	2638.522461 / B_SID_FREQ_DIV,
	2801.120361 / B_SID_FREQ_DIV,
	2967.359131 / B_SID_FREQ_DIV,
	3144.654053 / B_SID_FREQ_DIV,
};

/*
 * Using the following definitions we can use calls such as the following:
 *
 *	sid_register(id, B_SID_V1_FREQ_LO, phasemap[69][FREQ_LO]);
 *	sid_register(id, B_SID_V1_FREQ_HI, phasemap[69][FREQ_HI]);
 *
 * Which would set voice zero oscillation to A_440, ie, for MIDI key 69.
 */
#define FREQ_LO 0
#define FREQ_HI 1

typedef unsigned char SIDPhase[2];
SIDPhase *phasemap = (SIDPhase *) freqmap;

/*
 * Finally, these are the two single access methods to the chip emulation, one
 * will write a value to one of its registers, the other will set some emulator
 * equivalent analogue parameters in the floating point domain. 
 *
 * Details of the register settings are in the C64 programmer manual excepting
 * the control register which is specific to the emulator.
 */

int sid_register(int, unsigned char, unsigned char); /* ID, register, value */

float sid_IO(int, int, float); /* ID, command, param */

#endif /* _B_SID_DEFS */

