/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/csocket.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Mon Jun 29 18:18:45 1998                          */
/*    Last change :  Sat Oct 22 15:04:58 2005 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Scheme sockets                                                   */
/*    -------------------------------------------------------------    */
/*    This file is based on a contribution of                          */
/*    David Tolpin (dvd@pizza.msk.su)                                  */
/*                                                                     */
/*    Bugs correction (conversion between host and network byte order) */
/*    by Marc Furrer (Marc.Furrer@di.epfl.ch)                          */
/*                                                                     */
/*    Reworked  by Erick Gallesio for 2.2 release.                     */
/*    Some additions and simplifications (I hope).                     */
/*=====================================================================*/
#if defined( _MSC_VER) || defined( _MINGW_VER )
#  define _BGL_WIN32_VER
#endif

#include <bigloo_config.h>
#include <sys/types.h>
#ifndef _BGL_WIN32_VER
#   include <sys/socket.h>
#   include <netinet/in.h>
#   include <arpa/inet.h>
#   include <netdb.h>
#   include <time.h>
#   if( BGL_HAVE_SELECT )
#     include <sys/time.h>
#     include <sys/types.h>
#     include <unistd.h>
#   endif
#else
#   if defined( _MINGW_VER )
#      include "windows.h"
#   endif
#   include <winsock2.h>
#   include <mswsock.h>
#   include <io.h>
#endif
#include <fcntl.h>
#include <memory.h>
#include <errno.h>
#include <bigloo.h>

#define socklen_t void

#ifndef _BGL_WIN32_VER
#   define BAD_SOCKET(s) ((s) < 0)
#else
#   define BAD_SOCKET(s) ((s) == INVALID_SOCKET)
#endif

#define SOCKET_IO_BUFSIZE 1024

/*---------------------------------------------------------------------*/
/*    Imports ...                                                      */
/*---------------------------------------------------------------------*/
extern long default_io_bufsiz;
extern obj_t close_input_port( obj_t );
extern obj_t close_output_port( obj_t );
extern obj_t file_to_buffered_input_port( FILE *, long );

#ifndef _BGL_WIN32_VER
extern int dup( int );
extern int close( int );
#endif

extern long bgl_pipe_fread( char *, long, long, FILE * );

/*---------------------------------------------------------------------*/
/*    Symbol mutex                                                     */
/*---------------------------------------------------------------------*/
static obj_t socket_mutex = BUNSPEC;
static obj_t host_mutex = BUNSPEC;
DEFINE_STRING( socket_mutex_name, _1, "socket-mutex", 12 );
DEFINE_STRING( host_mutex_name, _2, "host-mutex", 10 );

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    socket_error ...                                                 */
/*---------------------------------------------------------------------*/
static void
socket_error( char *who, char *message, obj_t object ) {
   C_SYSTEM_FAILURE( BGL_IO_ERROR, who, message, object );
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    system_client_error ...                                          */
/*---------------------------------------------------------------------*/
static void
system_client_error( obj_t hostname, int port ) {
   char buffer1[ 512 ];
   char buffer2[ 512 ];

   sprintf( buffer1, "%s (%d)", strerror( errno ), errno );
   sprintf( buffer2, "%s:%d", BSTRING_TO_STRING( hostname ), port );

   socket_error( "make-client-socket", buffer1, string_to_bstring( buffer2 ) );
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    system_error ...                                                 */
/*---------------------------------------------------------------------*/
static void
system_error( char *who, obj_t val ) {
   char buffer[ 512 ];

   sprintf( buffer, "%s (%d)", strerror( errno ), errno );

   socket_error( who, buffer, val );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_host ...                                                     */
/*---------------------------------------------------------------------*/
obj_t
bgl_host( obj_t hostname ) {
   struct hostent *hp;
   struct sockaddr_in host;
   obj_t res;

   if( host_mutex == BUNSPEC ) {
      host_mutex = bgl_make_mutex( host_mutex_name) ;
   }

   bgl_mutex_lock( host_mutex );

   if( (hp = gethostbyname(BSTRING_TO_STRING(hostname))) == NULL ) {
      bgl_mutex_unlock( host_mutex );
      C_SYSTEM_FAILURE( BGL_IO_UNKNOWN_HOST_ERROR,
			"host", "unknown or misspelled host name", hostname );
   }

   memcpy( (char *) (&host.sin_addr), hp->h_addr, hp->h_length );
   res = string_to_bstring( inet_ntoa( host.sin_addr ) );

   bgl_mutex_unlock( host_mutex );

   return res;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    socket_startup ...                                               */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF void
socket_startup() {
#ifdef _BGL_WIN32_VER
   WSADATA data;
   int result = 0;
   WORD version_requested = MAKEWORD(2, 2);
   DWORD val = SO_SYNCHRONOUS_NONALERT;

   result = WSAStartup( version_requested, &data );
   if( 0 != result ) {
	socket_error( "socket_init",
		      "Failed to Initialize socket library", BNIL );
   }

   result = setsockopt( INVALID_SOCKET,
			SOL_SOCKET, SO_OPENTYPE,
			(const char *)&val,
			sizeof( val ) );
   if( 0 != result ) {
      socket_error( "make_server_socket",
		    "cannot set socket options", BUNSPEC);
   }
#endif

   socket_mutex = bgl_make_mutex( socket_mutex_name );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    socket_cleanup ...                                               */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF void
socket_cleanup() {
#ifdef _BGL_WIN32_VER
   WSACleanup();
#endif
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    set_socket_io_ports ...                                          */
/*---------------------------------------------------------------------*/
static void
set_socket_io_ports( int s, obj_t sock, char *who, char bufferedp ) {
   int t, len, port;
   obj_t hostname;
   char *fname;
   FILE *fs, *ft = 0;
   char buffer[ 200 ];

   /* if on windows obtain a C run-time compatable file descriptor */
#ifdef _BGL_WIN32_VER
   s = _open_osfhandle( s, _O_RDWR );
#endif

   /* duplicate handles so that we are able to access one */
   /* socket channel via two scheme ports.                */
   t = dup( s );

   if( t == -1 ) {
      sprintf( buffer, "%s: cannot duplicate io port", who );
      socket_error( "set_socket_io_ports", buffer, BUNSPEC );
   }

   if( !((fs = fdopen( s, "r" )) && (ft = fdopen( t, "w" ))) ) {
      sprintf( buffer, "%s: cannot create socket io ports", who );
      socket_error( "set_socket_io_ports", buffer, sock );
   }

   port = SOCKET( sock ).portnum;
   hostname = SOCKET( sock ).hostname;
   len = STRING_LENGTH( hostname ) + 20;
   fname = alloca( len );
   sprintf( fname, "%s:%d", BSTRING_TO_STRING( hostname ), port );

   /* Create input port */
   SOCKET( sock ).input = file_to_buffered_input_port( fs, bufferedp ? SOCKET_IO_BUFSIZE : 1 );
   SOCKET( sock ).input->input_port_t.kindof = KINDOF_SOCKET;
   SOCKET( sock ).input->input_port_t.name = string_to_bstring( fname );
   SOCKET( sock ).input->input_port_t.sysread = bgl_pipe_fread;

   /* Create output port */
   SOCKET(sock).output = make_output_port( fname, ft, KINDOF_FILE );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    set_socket_blocking ...                                          */
/*---------------------------------------------------------------------*/
#if( BGL_HAVE_FCNTL )
void
set_socket_blocking( int fd, int bool ) {
   int val;

   if( (val = fcntl( fd, F_GETFL, 0) ) < 0 ) {
      socket_error( "make_client_socket",
		    "cannot get socket control",
		    BUNSPEC);
   }

   if( bool ) {
      val |= O_NONBLOCK;
   } else {
      val &= ~O_NONBLOCK;
   }

   if( fcntl( fd, F_SETFL, val ) < 0) {
      socket_error( "make_client_socket",
		    "cannot set socket control",
		    BUNSPEC);
   }
}
#endif

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_client_socket ...                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
make_client_socket( obj_t hostname, int port, bool_t bufp, int timeo ) {
   struct hostent *hp;
   struct sockaddr_in server;
   int s;
   obj_t a_socket;

   /* acquire the global socket lock */
   bgl_mutex_lock( socket_mutex );

   /* Locate the host IP address */
   if( (hp = gethostbyname( BSTRING_TO_STRING( hostname ) )) == NULL ) {
      bgl_mutex_unlock( socket_mutex );
      C_SYSTEM_FAILURE( BGL_IO_UNKNOWN_HOST_ERROR,
			"make-client-socket",
			"unknown or misspelled host name",
			hostname );
   }

   /* Get a socket */
   if( BAD_SOCKET( s = (int)socket(AF_INET, SOCK_STREAM, 0) ) ) {
      bgl_mutex_unlock( socket_mutex );
      socket_error( "make-client-socket", "cannot create socket", hostname );
   }

   /* Setup a connect address */
   memset( &server, 0, sizeof( server ) );
   memcpy( (char *)&server.sin_addr, hp->h_addr, hp->h_length );
   server.sin_family = AF_INET;
   server.sin_port = htons( port );

   /* release the socket lock */
   bgl_mutex_unlock( socket_mutex );

#if( BGL_HAVE_SELECT && BGL_HAVE_FCNTL )
   if( timeo > 0 ) set_socket_blocking( s, 1 );
#endif

   /* Try to connect */
   if( connect( s, (struct sockaddr *) &server, sizeof(server)) != 0 ) {
#if( BGL_HAVE_SELECT && defined( EINPROGRESS ) ) 
      if( errno == EINPROGRESS ) {
	 fd_set readfds;
	 fd_set writefds;
	 struct timeval timeout;

	 FD_ZERO( &readfds );
	 FD_ZERO( &writefds );
	 FD_SET( s, &readfds );
	 FD_SET( s, &writefds );
	 timeout.tv_sec = 0;
	 timeout.tv_usec = timeo;

	 if( select( s + 1, &readfds, &writefds, NULL, &timeout ) <= 0 ) {
	    close( s );
	    socket_error( "make-client-socket",
			  "Connection time out",
			  hostname );
	    system_client_error( hostname, port );
	 } else {
	    int err;
	    int len = sizeof( int );
	    
	    getsockopt( s, SOL_SOCKET, SO_ERROR, (char *)&err, &len );

	    if( err != 0 ) {
	       close( s );
	       socket_error( "make-client-socket",
			     strerror( err ),
			     hostname );
	       system_client_error( hostname, port );
	    }
	 }
	 set_socket_blocking( s, 0 );	 
      } else {
	 close( s );
	 system_client_error( hostname, port );
      }
#else
      close( s );
      system_client_error( hostname, port );
#endif
   }

   /* Create a new Scheme socket object */
   a_socket = GC_MALLOC( SOCKET_SIZE );
   a_socket->socket_t.header = MAKE_HEADER( SOCKET_TYPE, 0 );
   a_socket->socket_t.portnum = ntohs( server.sin_port );
   a_socket->socket_t.hostname = string_to_bstring( hp->h_name );
   a_socket->socket_t.hostip = string_to_bstring( inet_ntoa( server.sin_addr ) );
   a_socket->socket_t.fd = s;
   a_socket->socket_t.input = BFALSE;
   a_socket->socket_t.output = BFALSE;
   a_socket->socket_t.stype = SOCKET_CLIENT;
   a_socket->socket_t.userdata = BUNSPEC;

   set_socket_io_ports( s, a_socket, "make-client-socket", bufp );
   return BREF( a_socket );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_server_socket ...                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
make_server_socket( int port ) {
   char msg[] = "make-server-socket";
   struct sockaddr_in sin;
   int s, portnum, len;
   obj_t a_socket;

   /* Determine port to use */
   portnum = port;
   if( portnum < 0 )
      socket_error( "make-server-socket", "bad port number", BINT( port ) );

   /* Create a socket */
   if( BAD_SOCKET(s = (int) socket(AF_INET, SOCK_STREAM, 0)) )
      socket_error( "make-server-socket", "Cannot create socket", BUNSPEC );

   /* Bind the socket to a name */
   sin.sin_family = AF_INET;
   sin.sin_port = htons( portnum );
   sin.sin_addr.s_addr = INADDR_ANY;

   /* OPTIONAL */
   {
      int optval;
      optval = 1;

      /* set the reuse flag */
      if( setsockopt(s, SOL_SOCKET, SO_REUSEADDR,
		     (char *) &optval, sizeof(optval)) < 0 ) {
	 system_error( msg, BINT( port ) );
      }
   }

   if( bind(s, (struct sockaddr *) &sin, sizeof(sin)) < 0 ) {
      close( s );
      system_error( msg, BINT( port ) );
   }

   /* Query the socket name, permits to get the true socket number */
   /* if 0 was given                                               */
   len = sizeof( sin );
   if( getsockname( s, (struct sockaddr *) &sin, (socklen_t *) & len ) < 0 ) {
      close( s );
      system_error( msg, BINT( port ) );
   }

   /* Indicate that we are ready to listen */
   if( listen( s, 5 ) < 0 ) {
      close( s );
      system_error( msg, BINT( port ) );
   }

   /* Now we can create the socket object */
   a_socket = GC_MALLOC(SOCKET_SIZE);
   a_socket->socket_t.header = MAKE_HEADER( SOCKET_TYPE, 0 );
   a_socket->socket_t.portnum = ntohs( sin.sin_port );
   a_socket->socket_t.hostname = BFALSE;
   a_socket->socket_t.hostip = BFALSE;
   a_socket->socket_t.fd = s;
   a_socket->socket_t.input = BFALSE;
   a_socket->socket_t.output = BFALSE;
   a_socket->socket_t.stype = SOCKET_SERVER;
   a_socket->socket_t.userdata = BUNSPEC;

   return BREF( a_socket );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_local_addr ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
socket_local_addr( obj_t sock ) {
   struct sockaddr_in sin;
   int len = sizeof( sin );

   if( SOCKET( sock ).stype == SOCKET_SERVER ) {
      return string_to_bstring( "0.0.0.0" );
   }

   if( getsockname( SOCKET( sock ).fd,
		    (struct sockaddr *) &sin, (socklen_t *) & len) )
      socket_error( "socket-local-address", strerror( errno ), sock );

   return string_to_bstring( (char *)inet_ntoa( sin.sin_addr ) );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_shutdown ...                                              */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
socket_shutdown( obj_t sock, int close_socket ) {
   int fd = SOCKET(sock).fd;
   obj_t chook = SOCKET_CHOOK(sock);

   if( fd > 0 ) {
      if( close_socket ) {
	 shutdown( fd, 2 );
	 close( fd );
	 SOCKET( sock ).fd = -1;
      } else {
	 close( fd );
      }
   }

   if( PROCEDUREP( chook ) ) {
      if( PROCEDURE_ARITY(chook) == 1 ) {
	 PROCEDURE_ENTRY( chook )( chook, sock, BEOA );
      } else {
	 C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			   "socket-shutdown",
			   "illegal close hook arity",
			   chook );
      }
   }

   /* Warning: input and output can have already be garbaged :if the   */
   /* socket is no more used, the input and output are not marked as   */
   /* used and can (eventually) be released before the call to         */
   /* shutdown (through free_socket) be done. One way could be to just */
   /* set SOCKET(sock).{in|out}put to #t and wait that next GC frees   */
   /* the ports if not already down. However, this will really         */
   /* disconnect the peer when the GC occurs rather than when the call */
   /* to shutdown is done. This is not important if this function is   */
   /* called by the GC, but could be annoying when it is called by the */
   /* user                                                             */
   if( INPUT_PORTP(SOCKET(sock).input) ) {
      close_input_port( SOCKET( sock ).input );
      SOCKET( sock ).input = BFALSE;
   }
   
   if( OUTPUT_PORTP( SOCKET( sock ).output ) ) {
      close_output_port( SOCKET( sock ).output );
      SOCKET(sock).output = BFALSE;
   }

   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_accept ...                                                */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
socket_accept( obj_t serv, char bufp, int errp ) {
   char *s;
   struct sockaddr_in sin;
   struct hostent *host;
   int len = sizeof( sin );
   int new_s;
   obj_t a_socket;
   int new_fd;

   while( BAD_SOCKET( new_s = (int) accept( SOCKET( serv ).fd,
					    (struct sockaddr *)&sin,
					    (socklen_t *)&len)) ) {
      if( errno == EINTR )
	 continue;

      if( errp )
	 system_error( "socket-accept", serv );
      else
	 return BFALSE;
   }

   /* Set the client info (if possible its name, otherwise its IP number) */
   host = gethostbyaddr( (char *)&sin.sin_addr,
			 sizeof( sin.sin_addr ),
			 AF_INET );
   s = (char *)inet_ntoa( sin.sin_addr );

   /* allocate and fill the new socket client for this connection */
   a_socket = GC_MALLOC( SOCKET_SIZE );
   a_socket->socket_t.header = MAKE_HEADER( SOCKET_TYPE, 0 );
   a_socket->socket_t.portnum = ntohs( sin.sin_port );
   a_socket->socket_t.hostname = string_to_bstring( host ? host->h_name : s );
   a_socket->socket_t.hostip = string_to_bstring( s );
   a_socket->socket_t.fd = new_s;
   a_socket->socket_t.stype = SOCKET_CLIENT;
   a_socket->socket_t.userdata = BUNSPEC;

   set_socket_io_ports( new_s, BREF( a_socket ), "socket-accept", bufp );

   return BREF( a_socket );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_close ...                                                 */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
socket_close( obj_t sock ) {
   int fd = SOCKET( sock ).fd;
   obj_t chook = SOCKET_CHOOK( sock );

   if( fd > 0 ) {
      close( fd );
      SOCKET( sock ).fd = -1;
   }

   if( PROCEDUREP( chook ) ) {
      if( PROCEDURE_ARITY(chook) == 1 ) {
	 PROCEDURE_ENTRY( chook )( chook, sock, BEOA );
      } else {
	 C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			   "socket-close",
			   "Illegal close hook arity",
			   chook );
      }
   }

   if( INPUT_PORTP( SOCKET( sock ).input ) ) {
      close_input_port( SOCKET( sock ).input );
      SOCKET( sock ).input = BFALSE;
   }
   
   if( OUTPUT_PORTP( SOCKET( sock ).output ) ) {
      close_output_port( SOCKET( sock ).output );
      SOCKET( sock ).output = BFALSE;
   }

   return BUNSPEC;
}
