;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/blib/bdb.scm                     */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Wed Jul 28 07:10:43 1999                          */
;*    Last change :  Thu Aug  8 17:37:28 2002 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The Bdb library                                                  */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __bdb
   
   (extern (macro bdb-printf::int (::string ::int) "printf")
	   (macro tmpnam::string (::long) "tmpnam")
	   (bdb-heap-size::int () "bdb_heap_size")
	   (bdb-alloc-gc::int () "bdb_alloc_gc")
	   (bdb-gc-number::int () "bdb_gc_number")
	   (bdb-gc-explore::int (::int ::obj ::int) "bdb_gc_explore")
	   (bdb-gc-explore-offset::int (::int ::long ::int) "bdb_gc_explore_offset")
	   (bdb-gc-dump::int (::int ::string ::int ::int ::int ::int) "bdb_gc_dump")
	   (bdb-gc-stat::int (::int ::int) "bdb_gc_stat")
	   (bdb-init-allocation-hooks!::int () "bdb_init_allocation_hooks")
	   (macro GC-gcollect::int () "GC_gcollect")
	   (GC-heap-info-available::bool "GC_heap_info_available")
	   
	   (bdb-gc-gather-stat::obj (::bool) "bdb_gc_gather_stat")
	   (bdb-set-lock!::int () "bdb_set_lock")
	   (bdb-release-lock!::int () "bdb_release_lock")

	   (macro bdb-live-producer-reset::int () "bdb_live_producer_reset")
	   (macro bdb-alloc-producer-reset::int () "bdb_allocated_producer_reset")

	   (export bdb-close-client! "bdb_close_client")
	   (export mangle "bdb_mangle")
	   (export mangle-for-funcall "bdb_mangle_for_funcall")
	   (export mangle2 "bdb_mangle2")
	   (export demangle "bdb_demangle")
	   (export demangle2 "bdb_demangle2")
	   (export output-value "bdb_output_value")
	   (export output-classes "bdb_output_classes")
	   (export dump-classes "bdb_dump_classes")
	   (export heap-info "bdb_heap_info")
	   (export heap-explore "bdb_heap_explore")
	   (export heap-explore-offset "bdb_heap_explore_offset")
	   (export heap-explore-send "bdb_heap_explore_send")
	   (export heap-dump "bdb_heap_dump")
	   (export heap-stat "bdb_heap_stat")
	   (export heap-dump-send "bdb_heap_dump_send")
	   (export bdb-print "bdb_print")
	   (export bdb-whatis "bdb_whatis")
	   (export bdb-funcall "bdb_funcall")
	   (export bdb-bint "bdb_bint")
	   (export bdb-bchar "bdb_bchar")
	   (export bdb-true "bdb_true")
	   (export bdb-false "bdb_false")
	   (export bdb-nil "bdb_nil")
	   (export bdb-find-type "bdb_find_type")
	   
	   (export bdb-heap-info-available? "bdb_heap_info_available")
	   (export bdb-get-heap-statistics "bdb_get_heap_statistics")
	   (export gc-auto-stat "GC_auto_stat")
	   (export gc-print-auto-stat "GC_print_auto_stat")

	   (export bdb-clean-heap-statistics "bdb_clean_heap_statistics"))

   (import __bdb_env)
   
   (export (bdb-initial-breakpoint)
	   (bdb-close-client!)
	   (mangle::int ::int ::string)
	   (mangle-for-funcall::int ::int ::string)
	   (mangle2::int ::int ::string ::string)
	   (demangle::int ::int ::string)
	   (demangle2::int ::int ::string ::string)
	   (output-value::int ::int ::obj ::bool)
	   (output-classes::int)
	   (dump-classes::int ::int)
	   (bdb-print::string ::obj ::bool)
	   (bdb-whatis::int ::int ::obj)
	   (bdb-funcall::obj ::procedure ::pair-nil)
	   (bdb-bint ::bint)
	   (bdb-bchar ::bchar)
	   (bdb-true)
	   (bdb-false)
	   (bdb-nil::obj)
	   (bdb-find-type::string ::obj)
	   
	   ;; heap inspection
	   (heap-info::int ::int)
	   (heap-explore::int ::int ::obj ::int)
	   (heap-explore-offset::int ::int ::long ::int)
	   (heap-explore-send::int ::int ::int ::int ::int ::string ::int ::string ::pair-nil ::int ::string)
	   (heap-dump::int ::int ::string ::int ::int ::int ::int)
	   (heap-stat::int ::int ::int)
	   (heap-dump-send::int ::int ::obj)
	   (bdb-heap-info-available?::int ::int)

	   (bdb-clean-heap-statistics ::int)
	   (bdb-get-heap-statistics::int ::int)
	   (gc-auto-stat)
	   (gc-print-auto-stat)))

;*---------------------------------------------------------------------*/
;*    bdb-initial-breakpoint ...                                       */
;*---------------------------------------------------------------------*/
(define (bdb-initial-breakpoint)
   #unspecified)
 
;*---------------------------------------------------------------------*/
;*    *client* ...                                                     */
;*---------------------------------------------------------------------*/
(define *client* #unspecified)

;*---------------------------------------------------------------------*/
;*    bdb:init-client! ...                                             */
;*    -------------------------------------------------------------    */
;*    This function is called by the server each time the server       */
;*    wants to communicate. There is no way to improve that because    */
;*    there is no way for the server to know that the client is        */
;*    already initialized. In consequence, we have to take care        */
;*    here not to initalize several time the same client.              */
;*---------------------------------------------------------------------*/
(define (bdb:init-client! port-number)
   (let ()
      (c-push-trace 'bdb:init-client!)
      ;; if we have already opened a client but from a different port-number
      ;; we start closing the previous client
      (if (and (socket? *client*)
	       (not (=fx (socket-port-number *client*) port-number)))
	  (bdb-close-client!))
      (if (not (socket? *client*))
	  (begin
	     ;; we setup the socket client
	     (set! *client* (make-client-socket "localhost" port-number))
	     (if (not (socket? *client*))
		 (error "bdb:init-client" "Can't setup client" *client*))))
      (c-pop-trace)
      0))
       
;*---------------------------------------------------------------------*/
;*    bdb-close-client! ...                                            */
;*---------------------------------------------------------------------*/
(define (bdb-close-client!)
   (if (socket? *client*)
       (begin
	  (socket-shutdown *client* #f)
 	  (set! *client* #unspecified))))

;*---------------------------------------------------------------------*/
;*    bdb-invoke ...                                                   */
;*---------------------------------------------------------------------*/
(define (bdb-invoke port-number kind obj)
   (let ()
      (c-push-trace 'bdb:invoke)
      (bdb:init-client! port-number)
      (let ((port (socket-output *client*)))
	 ;; the kind of connection
	 (write kind port)
	 (newline port)
	 ;; the value to be sent
	 (write obj port)
	 (newline port)
	 ;; we flush the all transfer
	 (flush-output-port port))
      (c-pop-trace)
      0))

;* {*---------------------------------------------------------------------*} */
;* {*    *mangling-cache* ...                                             *} */
;* {*---------------------------------------------------------------------*} */
;* (define *demangling-cache* #unspecified)                            */
;* (define *mangling-cache* #unspecified)                              */
;*                                                                     */
;* {*---------------------------------------------------------------------*} */
;* {*    reset-demangling-cache! ...                                      *} */
;* {*---------------------------------------------------------------------*} */
;* (define (reset-demangling-cache!)                                   */
;*    (let ()                                                          */
;*       (c-push-trace 'bdb:reset-demangling-cache!)                   */
;*       (set! *demangling-cache* (make-hashtable))                    */
;*       (set! *mangling-cache* (make-hashtable))                      */
;*       (c-pop-trace)))                                               */

;*---------------------------------------------------------------------*/
;*    *bdb-module-info* ...                                            */
;*---------------------------------------------------------------------*/
(define *bdb-module-info* '())

;*---------------------------------------------------------------------*/
;*    mangle ...                                                       */
;*    -------------------------------------------------------------    */
;*    Mangle the identifier and returns to bdb.                        */
;*---------------------------------------------------------------------*/
(define (mangle port-number::int bgl::string)
   (let ((c (bgl->c bgl)))
      (bdb-invoke port-number #t c))
   0)

;*---------------------------------------------------------------------*/
;*    mangle2 ...                                                      */
;*    -------------------------------------------------------------    */
;*    Mangle the identifier and returns to bdb.                        */
;*---------------------------------------------------------------------*/
(define (mangle2 port-number::int bgl::string loc::string)
   (let ((c (c-bgl->c bgl loc)))
      (bdb-invoke port-number #t c))
   0)

;*---------------------------------------------------------------------*/
;*    mangle-for-funcall ...                                           */
;*    -------------------------------------------------------------    */
;*    Mangle the identifier and returns to bdb.                        */
;*    The returned identifier is the identifier of the closure         */
;*    associated to the Bigloo identifier.                             */
;*---------------------------------------------------------------------*/
(define (mangle-for-funcall port-number::int bgl::string)
   (let ((c (bgl->c-funcall bgl)))
      (bdb-invoke port-number #t c))
   0)

;*---------------------------------------------------------------------*/
;*    demangle ...                                                     */
;*    -------------------------------------------------------------    */
;*    Demangles a C identifier.                                        */
;*---------------------------------------------------------------------*/
(define (demangle port-number::int c::string)
   (let ((c (c->bgl c)))
      (bdb-invoke port-number #t c))
   0)

;*---------------------------------------------------------------------*/
;*    demangle2 ...                                                    */
;*    -------------------------------------------------------------    */
;*    Demangles a C identifier.                                        */
;*---------------------------------------------------------------------*/
(define (demangle2 port-number::int c::string loc::string)
   (let ((c (c-c->bgl c loc)))
      (bdb-invoke port-number #t c))
   0)
   
;*---------------------------------------------------------------------*/
;*    output-value ...                                                 */
;*---------------------------------------------------------------------*/
(define (output-value port-number::int value circle?::bool)
   (let ()
      (c-push-trace 'bdb:output-value)
      (let ((port (open-output-string)))
	 (if circle?
	     (write-circle value port)
	     (let ((len (get-write-length)))
		(set-write-length! 10)
		(write value port)
		(set-write-length! len)))
	 (let ((c (cons (find-runtime-type value)
			(close-output-port port))))
	    (bdb-invoke port-number #t c)
	    (c-pop-trace)
	    0))))

;*---------------------------------------------------------------------*/
;*    output-classes ...                                               */
;*---------------------------------------------------------------------*/
(define (output-classes::int)
   (let ()
      (c-push-trace 'bdb:output-value)
      (print (bgl-get-classes))
      (c-pop-trace)
      0))

;*---------------------------------------------------------------------*/
;*    dump-classes ...                                                 */
;*---------------------------------------------------------------------*/
(define (dump-classes port-number::int)
   (bdb-invoke port-number #t (bgl-get-classes))
   0)

;*---------------------------------------------------------------------*/
;*    *info-chunk* ...                                                 */
;*    -------------------------------------------------------------    */
;*    We must avoid allocating object when an HEAP-INFO request        */
;*    is received in order to deliver revelant informations.           */
;*---------------------------------------------------------------------*/
(define *info-chunk* (list #unspecified
			   #unspecified
			   #unspecified))

;*---------------------------------------------------------------------*/
;*    heap-info ...                                                    */
;*    -------------------------------------------------------------    */
;*    This function deliver to the debugger informations relative      */
;*    to the GC state (e.g., the heap, the number of allocations, ...) */
;*---------------------------------------------------------------------*/
(define (heap-info port-number::int)
   ;; the gc number
   (set-car! *info-chunk* (bdb-gc-number))
   ;; the heap size
   (set-car! (cdr *info-chunk*) (bdb-heap-size))
   ;; the number of allocation since last gc
   (set-car! (cddr *info-chunk*) (bdb-alloc-gc))
   ;; the serialization
   (let ((c *info-chunk*))
      (bdb-invoke port-number #t c)
      0))

;*---------------------------------------------------------------------*/
;*    heap-explore ...                                                 */
;*    -------------------------------------------------------------    */
;*    Explore an object in the heap.                                   */
;*---------------------------------------------------------------------*/
(define (heap-explore port-number::int obj verbose)
   (bdb-gc-explore port-number obj verbose))

;*---------------------------------------------------------------------*/
;*    heap-explore-offset ...                                          */
;*    -------------------------------------------------------------    */
;*    Explore an object in the heap.                                   */
;*---------------------------------------------------------------------*/
(define (heap-explore-offset port-number::int offset verbose)
   (bdb-gc-explore-offset port-number offset verbose))

;*---------------------------------------------------------------------*/
;*    heap-explore-send ...                                            */
;*---------------------------------------------------------------------*/
(define (heap-explore-send port-number::int
			   kind::int gen::int size::int
			   fun::string age::int
			   type::string link::pair-nil
			   base::int
			   root-base::string)
   (let ()
      (c-push-trace 'bdb:explore-send)
      (let* ((source (bgl-source fun))
	     (c (list kind gen size fun age type link base source root-base)))
	 (bdb-invoke port-number #t c)
	 (c-pop-trace)
	 0)))

;*---------------------------------------------------------------------*/
;*    heap-dump ...                                                    */
;*    -------------------------------------------------------------    */
;*    Dump an object in the heap.                                      */
;*---------------------------------------------------------------------*/
(define (heap-dump port-number::int fname width depth mode hide)
   (let ()
      (c-push-trace 'bdb:heap-dump)
      (let ((res (bdb-gc-dump port-number fname width depth mode hide)))
	 (c-pop-trace)
	 res)))

;*---------------------------------------------------------------------*/
;*    heap-stat ...                                                    */
;*    -------------------------------------------------------------    */
;*    Computes statistics of the heap occupation.                      */
;*---------------------------------------------------------------------*/
(define (heap-stat port-number::int hide)
   (let ()
      (c-push-trace 'bdb:heap-stat)
      (let ((res (bdb-gc-stat port-number hide)))
	 (c-pop-trace)
	 res)))

;*---------------------------------------------------------------------*/
;*    heap-dump-send ...                                               */
;*---------------------------------------------------------------------*/
(define (heap-dump-send port-number::int obj)
   (let ()
      (c-push-trace 'bdb:heap-dump-send)
      (let ((c obj))
	 (bdb-invoke port-number #t c)
	 (c-pop-trace)
	 0)))

;*---------------------------------------------------------------------*/
;*    bdb-print ...                                                    */
;*    -------------------------------------------------------------    */
;*    This function returns a string that is the result of the         */
;*    printing. This function does not perform any output. This        */
;*    function is called when gdb wants a value. For instance, it is   */
;*    used by the PRINT and the DISPLAY commands.                      */
;*---------------------------------------------------------------------*/
(define (bdb-print::string value::obj circle?::bool)
   (let ()
      (c-push-trace 'bdb:print)
      (let ((port (open-output-string)))
	 (if circle?
	     (write-circle value port)
	     (let ((len (get-write-length)))
		(set-write-length! 10)
		(write value port)
		(set-write-length! len)))
	 (let ((res (close-output-port port)))
	    (c-pop-trace)
	    res))))

;*---------------------------------------------------------------------*/
;*    bdb-whatis ...                                                   */
;*    -------------------------------------------------------------    */
;*    Returns the dynamic type of a Scheme object.                     */
;*---------------------------------------------------------------------*/
(define (bdb-whatis::int port-number::int value::obj)
   (let ()
      (c-push-trace 'bdb:whatis)
      (let ((port (open-output-string)))
	 (display "type = " port)
	 (write (find-runtime-type value) port)
	 (bdb-invoke port-number #t (close-output-port port)))
      (c-pop-trace)
      0))
   
;*---------------------------------------------------------------------*/
;*    bdb-true ...                                                     */
;*---------------------------------------------------------------------*/
(define (bdb-true)
   #t)

;*---------------------------------------------------------------------*/
;*    bdb-false ...                                                    */
;*---------------------------------------------------------------------*/
(define (bdb-false)
   #f)

;*---------------------------------------------------------------------*/
;*    bdb-bchar ...                                                    */
;*---------------------------------------------------------------------*/
(define (bdb-bchar char)
   (pragma::obj "BCHAR( $1 )" (char->integer char)))
   
;*---------------------------------------------------------------------*/
;*    bdb-bint ...                                                     */
;*---------------------------------------------------------------------*/
(define (bdb-bint int)
   (pragma::obj "BINT( $1 )" int))
   
;*---------------------------------------------------------------------*/
;*    bdb-nil ...                                                      */
;*---------------------------------------------------------------------*/
(define (bdb-nil)
   '())

;*---------------------------------------------------------------------*/
;*    bdb-funcall ...                                                  */
;*---------------------------------------------------------------------*/
(define (bdb-funcall fun::procedure pair::pair-nil)
   (apply fun pair))

;*---------------------------------------------------------------------*/
;*    bdb-find-type ...                                                */
;*---------------------------------------------------------------------*/
(define (bdb-find-type obj)
   (find-runtime-type obj))

;*---------------------------------------------------------------------*/
;*    bdb-heap-info-available? ...                                     */
;*---------------------------------------------------------------------*/
(define (bdb-heap-info-available? port-number::int)
   (bdb-invoke port-number #t GC-heap-info-available))

;*---------------------------------------------------------------------*/
;*    *heap-statistics*                                                */
;*---------------------------------------------------------------------*/
(define *heap-statistics* #unspecified)

;*---------------------------------------------------------------------*/
;*    gc-print-auto-stat ...                                           */
;*---------------------------------------------------------------------*/
(define (gc-print-auto-stat)
   (let loop ((hs *heap-statistics*)
	      (i 0))
      (if (pair? hs)
	  (let ((s (car hs)))
	     (print "----------------------------------" i)
	     (for-each (lambda (x) (print "   " x)) s)
	     (loop (cdr hs) (+fx i 1))))))

;*---------------------------------------------------------------------*/
;*    *gc-auto-stack-lock*                                             */
;*---------------------------------------------------------------------*/
(define *gc-auto-stack-lock* #f)

;*---------------------------------------------------------------------*/
;*    gc-auto-stat ...                                                 */
;*    -------------------------------------------------------------    */
;*    The function that is called each time the collector executes     */
;*    a collection.                                                    */
;*    -------------------------------------------------------------    */
;*    It can't be permitted for this function to be re-entrant         */
;*    (otherwise it means that gc-auto-stat allocatates too much       */
;*    and that it will loop forever). We thus use a lock to avoid      */
;*    simultaneous invokations.                                        */
;*---------------------------------------------------------------------*/
(define *auto-stat-num* 0)
(define (gc-auto-stat)
   (if (not *gc-auto-stack-lock*)
       (begin
	  (set! *gc-auto-stack-lock* #t)
	  (let ()
	     (c-push-trace 'bdb:gc-auto-stat)
	     (bdb-set-lock!)
;* 	     (bdb-printf #"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n" 0) */
;* 	     (bdb-printf #"+++ gc-auto-stat...: %d\n" *auto-stat-num*)     */
	     (set! *auto-stat-num* (+fx 1 *auto-stat-num*))
	     (set! *heap-statistics*
		   (cons (bdb-gc-gather-stat #t)
			 (if (pair-or-null? *heap-statistics*)
			     *heap-statistics*
			     '())))
	     (bdb-release-lock!)
	     (c-pop-trace)
	     (set! *gc-auto-stack-lock* #f)
	     #unspecified))
       (begin
;* 	  (bdb-printf #"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n" 0) */
;* 	  (bdb-printf #"---- skiping gc-auto-stat...: %d\n" *auto-stat-num*) */
	  (set! *auto-stat-num* (+fx 1 *auto-stat-num*)))))
       


;*---------------------------------------------------------------------*/
;*    bdb-clean-heap-statistics ...                                    */
;*---------------------------------------------------------------------*/
(define (bdb-clean-heap-statistics port-number::int)
   (set! *heap-statistics* '())
   (bdb-live-producer-reset)
   (bdb-alloc-producer-reset)
   (bdb-invoke port-number #t #t))

;*---------------------------------------------------------------------*/
;*    bdb-get-heap-statistics ...                                      */
;*---------------------------------------------------------------------*/
(define (bdb-get-heap-statistics port-number::int)
   (let ()
      (c-push-trace 'bdb:get-heap-statistics)
      (bdb-set-lock!)
      ;; then we can send the result to bdb
      (cond
	 ((eq? *heap-statistics* #unspecified)
	  ;; no gc yet
	  (bdb-invoke port-number #t #f))
	 ((not (pair? *heap-statistics*))
	  ;; no change since last invokation
	  (bdb-invoke port-number #t '()))
	 (else
	  ;; changes since last invokation
	  ;; we have to force a GC
	  (GC-gcollect)
	  (let* ((stat (reverse! *heap-statistics*))
		 (fname (tmpnam 0))
		 (file (with-output-to-file fname
			  ;; we created file will be delete by Kbdb
			  (lambda ()
			     (write stat)))))
	     (bdb-invoke port-number 'file fname)
	     (set! *heap-statistics* '()))))
      (bdb-release-lock!)
      (c-pop-trace)
      0))

;*---------------------------------------------------------------------*/
;*    At init-time, we check if heap info are available. If they are   */
;*    we hook the GC to gather new statistics each time a GC is        */
;*    spawned.                                                         */
;*---------------------------------------------------------------------*/
(if GC-heap-info-available
    (begin
       ;; we set the allocation hooks
       (bdb-init-allocation-hooks!)
       ;; we register that each time the collector spawn a collection
       ;; we must gather statistics
       (pragma "GC_add_gc_hook( GC_auto_stat )")
       #unspecified))

