;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime/Eval/evcompile.scm           */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Mar 25 09:09:18 1994                          */
;*    Last change :  Wed Jun  9 10:02:08 2004 (serrano)                */
;*    -------------------------------------------------------------    */
;*    La pre-compilation des formes pour permettre l'interpretation    */
;*    rapide                                                           */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __evcompile
   (include "Eval/byte-code.sch")
   
   (import  __type
	    __error
	    __bigloo
	    __tvector
	    __structure
	    __tvector
	    __bexit
	    __os
	    __dsssl
	    __bit
	    
	    __r4_numbers_6_5
	    __r4_numbers_6_5_fixnum
	    __r4_numbers_6_5_flonum
	    __r4_characters_6_6
	    __r4_equivalence_6_2
	    __r4_booleans_6_1
	    __r4_symbols_6_4
	    __r4_strings_6_7
	    __r4_pairs_and_lists_6_3
	    __r4_input_6_10_2
	    __r4_control_features_6_9
	    __r4_vectors_6_8
	    __r4_ports_6_10_1
	    __r4_output_6_10_3

   	    __evenv
	    __eval
	    __expand)
   
   (export  (evcompile exp env Genv where named? tail loc)
	    (find-loc  exp default)))

;*---------------------------------------------------------------------*/
;*    find-loc ...                                                     */
;*---------------------------------------------------------------------*/
(define (find-loc exp default)
   (if (epair? exp)
       (cer exp)
       default))

;*---------------------------------------------------------------------*/
;*    untype-ident ...                                                 */
;*---------------------------------------------------------------------*/
(define (untype-ident id)
   (if (not (symbol? id))
       id
       (let* ((string (symbol->string id))
	      (len    (string-length string)))
	  (let loop ((walker  0))
	     (cond
		((=fx walker len)
		 id)
		((and (char=? (string-ref string walker) #\:)
		      (<fx walker (-fx len 1))
		      (char=? (string-ref string (+fx walker 1)) #\:))
		 (string->symbol (substring string 0 walker)))
		(else
		 (loop (+fx walker 1))))))))

;*---------------------------------------------------------------------*/
;*    untype-ident* ...                                                */
;*---------------------------------------------------------------------*/
(define (untype-ident* idents)
   (cond
      ((null? idents)
       '())
      ((pair? idents)
       (cons (untype-ident (car idents)) (untype-ident* (cdr idents))))
      (else
       (untype-ident idents))))

;*---------------------------------------------------------------------*/
;*    evcompile ...                                                    */
;*    -------------------------------------------------------------    */
;*    La phase d'expansion a genere une syntaxe correcte. On n'a donc  */
;*    plus du tout a la tester maintenant.                             */
;*---------------------------------------------------------------------*/
(define (evcompile exp env genv where named? tail loc)
   (match-case exp
      (()
       (evcompile-error loc "eval" "Illegal expression" '()))
      ((module ?- . ?decls)
       (module-declaration! decls)
       (unspecified))
      ((assert . ?-)
       (unspecified))
      ((atom ?atom)
       (cond
	  ((symbol? atom)
	   (evcompile-ref (variable loc atom env genv) loc))
	  ((or (vector? atom)
	       (struct? atom))
	   (evcompile-error loc
			    "eval"
			    "Illegal expression (should be quoted)"
			    exp))
	  (else
	   (evcompile-cnst atom loc))))
      ((quote ?cnst)
       (evcompile-cnst cnst (find-loc exp loc)))
      ((if ?si ?alors ?sinon)
       (let ((loc (find-loc exp loc)))
	  (evcompile-if (evcompile si env genv
				   where #f #f
				   (find-loc si loc))
			(evcompile alors env genv
				   where named? tail
				   (find-loc alors loc))
			(evcompile sinon env genv
				   where named? tail
				   (find-loc sinon loc))
			loc)))
      (((kwote or) . ?rest)
       (evcompile-or rest env genv where named? (find-loc exp loc)))
      (((kwote and) . ?rest)
       (evcompile-and rest env genv where named? (find-loc exp loc)))
      ((begin . ?rest)
       (evcompile-begin rest env genv where named? tail (find-loc exp loc)))
      ((or (define ?var (and (lambda . ?-) ?val))
	   (define ?var (begin (and (lambda . ?-) ?val))))
       (if (and (eq? where 'nowhere)
		(or (eq? genv (scheme-report-environment 5))
		    (eq? genv (null-environment 5))))
	   (evcompile-error loc
			    "eval"
			    "Illegal define form (sealed environment)"
			    exp)
	   (let ((loc (find-loc exp loc)))
	      (evcompile-define-lambda (untype-ident var)
				       (delay (evcompile val '()
							 genv var #t #t
							 (find-loc exp loc)))
				       loc))))
      ((define ?var ?val)
       (if (and (eq? where 'nowhere)
		(or (eq? genv (scheme-report-environment 5))
		    (eq? genv (null-environment 5))))
	   (evcompile-error loc
			    "eval"
			    "Illegal define form (sealed environment)"
			    exp)
	   (let ((loc (find-loc exp loc)))
	      (evcompile-define-value (untype-ident var)
				      (evcompile val '()
						 genv where named? #t
						 (find-loc val loc))
				      loc))))
      ((set! . ?-)
       (match-case exp
	  ((?- (and (? symbol?) ?var) ?val)
	   (let ((loc (find-loc exp loc)))
	      (evcompile-set (variable loc var env genv)
			     (evcompile val env
					genv var #t #f
					(find-loc val loc))
			     loc)))
	  (else
	   (error "set!" "Illegal form" exp))))
      ((bind-exit ?escape ?body)
       (let ((loc (find-loc exp loc)))
	  (evcompile-bind-exit (evcompile `(lambda ,escape ,body)
					  env
					  genv
					  escape
					  #t
					  #f
					  (find-loc body loc))
			       loc)))
      ((unwind-protect ?body . ?protect)
       (let ((loc (find-loc exp loc)))
	  (evcompile-unwind-protect (evcompile body env
					       genv where named? #f
					       (find-loc body loc))
				    (evcompile-begin protect env genv
						     where named? #f
						     (find-loc protect loc))
				    loc)))
      ((lambda ?formals ?body)
       (let* ((loc (find-loc exp loc))
	      (scm-formals (dsssl-formals->scheme-formals
			    formals
			    (lambda (proc msg obj)
			       (evcompile-error loc proc msg obj))))
	      (untyped-scm-formals (untype-ident* scm-formals)))
	  (evcompile-lambda untyped-scm-formals
			    (evcompile (expand
					(make-dsssl-function-prelude
					 exp
					 formals
					 body
					 (lambda (proc msg obj)
					    (evcompile-error loc
							     proc
							     msg
							     obj))))
				       (extend-env untyped-scm-formals env)
				       genv
				       where #f #t
				       (find-loc body loc))
			    where
			    named? loc)))
      ((let ?bindings ?body)
       (evcompile-let bindings body env
		      genv where named? tail
		      (find-loc exp loc)))
      ((let* ?bindings ?body)
       (evcompile-let* bindings body env
		       genv where named? tail
		       (find-loc exp loc)))
      ((letrec ?bindings ?body)
       (evcompile-letrec bindings body env
			 genv where named? tail
			 (find-loc exp loc)))
      (((atom ?fun) . ?args)
       (let* ((loc (find-loc exp loc))
	      (actuals (map (lambda (a)
			       (evcompile a env genv where #f #f loc))
			    args)))
	  (cond
	     ((symbol? fun)
	      (let ((proc (variable loc fun env genv)))
		 (evcompile-application fun
					(evcompile-ref proc loc)
					actuals
					tail
					loc)))
	     ((procedure? fun)
	      (evcompile-compiled-application fun actuals loc))
	     (else
	      (evcompile-error loc "eval" "Not a procedure" fun)
	      (evcode -2 loc (list "eval" "Not a procedure" fun))))))
      ((?fun . ?args)
       (let ((loc (find-loc exp loc))
	     (actuals (map (lambda (a)
			      (evcompile a env genv where #f #f loc))
			   args))
	     (proc (evcompile fun env genv where #f #f loc)))
	  (evcompile-application fun proc actuals tail loc)))
      (else
       (evcompile-error loc "eval" "Illegal form" exp))))

;*---------------------------------------------------------------------*/
;*    evcompile-cnst ...                                               */
;*---------------------------------------------------------------------*/
(define (evcompile-cnst cnst loc)
   (cond
      ((vector? cnst)
       (evcode -1 loc cnst))
      (else
       cnst)))

;*---------------------------------------------------------------------*/
;*    evcompile-ref ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-ref variable loc)
   (cond
      ((eval-global? variable)
       (evcode (if (eq? (eval-global-tag variable) 1) 5 6) loc variable))
      ((dynamic? variable)
       (evcode 7 loc (dynamic-name variable)))
      (else
       (case variable
	  ((0 1 2 3)
	   (evcode variable loc))
	  (else
	   (evcode 4 loc variable))))))

;*---------------------------------------------------------------------*/
;*    evcompile-set ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-set variable value loc)
   (cond
      ((eval-global? variable)
       (evcode 8 loc variable value))
      ((dynamic? variable)
       (evcode 9 loc (dynamic-name variable) value))
      (else
       (case variable
	  ((0 1 2 3)
	   (evcode (+fx 10 variable) loc value))
	  (else
	   (evcode 14 loc variable value))))))

;*---------------------------------------------------------------------*/
;*    evcompile-if ...                                                 */
;*---------------------------------------------------------------------*/
(define (evcompile-if si alors sinon loc)
   (evcode 15 loc si alors sinon))

;*---------------------------------------------------------------------*/
;*    evcompile-or ...                                                 */
;*---------------------------------------------------------------------*/
(define (evcompile-or body env genv where named? loc)
   (let ((as (map (lambda (x)
		     (evcompile x env genv where named? #f loc))
		  body)))
      (list->evcode 67 loc as)))

;*---------------------------------------------------------------------*/
;*    evcompile-and ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-and body env genv where named? loc)
   (let ((as (map (lambda (x)
		     (evcompile x env genv where named? #f loc))
		  body)))
      (list->evcode 68 loc as)))

;*---------------------------------------------------------------------*/
;*    evcompile-begin ...                                              */
;*---------------------------------------------------------------------*/
(define (evcompile-begin body env genv where named? tail loc)
   (cond
      ((null? body)
       (evcompile #unspecified env genv where named? tail loc))
      ((null? (cdr body))
       (evcompile (car body) env genv
		  where named? tail
		  (find-loc (car body) loc)))
      (else
       (let ((cbody (let loop ((rest body))
		       (cond
			  ((null? rest)
			   '())
			  ((null? (cdr rest))
			   (cons (evcompile (car rest) env
					    genv where named? tail
					    (find-loc (car rest) loc))
				 '()))
			  (else
			   (cons (evcompile (car rest) env genv where #f #f
					    (find-loc (car rest) loc))
				 (loop (cdr rest))))))))
	  (list->evcode 16 loc cbody)))))

;*---------------------------------------------------------------------*/
;*    evcompile-define-lambda ...                                      */
;*    -------------------------------------------------------------    */
;*    Le calcul de `val' a ete differe car on ne veut evcompiler la    */
;*    valeur liee d'un define qu'une fois que la variable a ete liee   */
;*    dans l'environment. Si on ne fait pas cela on se tape que des    */
;*    appels dynamics dans les definitions des fonctions               */
;*    auto-recursives !                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-define-lambda var val loc)
   (evcode 17 loc var val))

;*---------------------------------------------------------------------*/
;*    evcompile-define-value ...                                       */
;*---------------------------------------------------------------------*/
(define (evcompile-define-value var val loc)
   (evcode 63 loc var val))
    
;*---------------------------------------------------------------------*/
;*    evcompile-bind-exit ...                                          */
;*---------------------------------------------------------------------*/
(define (evcompile-bind-exit body loc)
   (evcode 18 loc body))

;*---------------------------------------------------------------------*/
;*    evcompile-unwind-protect ...                                     */
;*---------------------------------------------------------------------*/
(define (evcompile-unwind-protect body protect loc)
   (evcode 64 loc body protect))

;*---------------------------------------------------------------------*/
;*    evcompile-compiled-application ...                               */
;*---------------------------------------------------------------------*/
(define (evcompile-compiled-application proc args loc)
   (case (length args)
      ((0)
       (evcode 25 loc proc))
      ((1)
       (evcode 26 loc proc (car args)))
      ((2)
       (evcode 27 loc proc (car args) (cadr args)))
      ((3)
       (evcode 28 loc proc (car args) (cadr args) (caddr args)))
      ((4)
       (evcode 29 loc proc (car args) (cadr args) (caddr args) (cadddr args)))
      (else
       (evcode 30 loc proc args))))

;*---------------------------------------------------------------------*/
;*    evcompile-application ...                                        */
;*---------------------------------------------------------------------*/
(define (evcompile-application name proc args tail loc)
   (if tail
       (case (length args)
	  ((0)
	   (evcode 131 loc name proc tail))
	  ((1)
	   (evcode 132 loc name proc (car args) tail))
	  ((2)
	   (evcode 133 loc name proc (car args) (cadr args) tail))
	  ((3)
	   (evcode 134 loc name proc (car args) (cadr args) (caddr args) tail))
	  ((4)
	   (evcode 135 loc name proc (car args) (cadr args) (caddr args) (cadddr args) tail))
	  (else
	   (evcode 136 loc name proc args tail)))
       (case (length args)
	  ((0)
	   (evcode 31 loc name proc))
	  ((1)
	   (evcode 32 loc name proc (car args)))
	  ((2)
	   (evcode 33 loc name proc (car args) (cadr args)))
	  ((3)
	   (evcode 34 loc name proc (car args) (cadr args) (caddr args)))
	  ((4)
	   (evcode 35 loc name proc (car args) (cadr args) (caddr args) (cadddr args)))
	  (else
	   (evcode 36 loc name proc args)))))

;*---------------------------------------------------------------------*/
;*    evcompile-lambda ...                                             */
;*---------------------------------------------------------------------*/
(define (evcompile-lambda formals body where named? loc)
   (match-case formals
      ((or () (?-) (?- ?-) (?- ?- ?-) (?- ?- ?- ?-))
       (if named?
	   (evcode (+fx (length formals) 37) loc body where)
	   (evcode (+fx (length formals) 42) loc body)))
      ((atom ?-)
       (if named?
	   (evcode 47 loc body where)
	   (evcode 51 loc body)))
      (((atom ?-) . (atom ?-))
       (if named?
	   (evcode 48 loc body where)
	   (evcode 52 loc body)))
      (((atom ?-) (atom ?-) . (atom ?-))
       (if named?
	   (evcode 49 loc body where)
	   (evcode 53 loc body)))
      (((atom ?-) (atom ?-) (atom ?-) . (atom ?-))
       (if named?
	   (evcode 50 loc body where)
	   (evcode 54 loc body)))
      (else
       (if named?
	   (evcode 55 loc body where formals)
	   (evcode 56 loc body formals)))))

;*---------------------------------------------------------------------*/
;*    evcompile-let ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-let bindings body env genv where named? tail loc)
   (let* ((env2 (extend-env (map (lambda (i) (untype-ident (car i))) bindings)
			    env))
	  (b (evcompile body env2 genv where named? tail loc))
	  (as (map (lambda (a)
		      (evcompile (cadr a) env genv where named? #f loc))
		   bindings)))
      (evcode 65 loc b (reverse! as))))
   
;*---------------------------------------------------------------------*/
;*    evcompile-let* ...                                               */
;*---------------------------------------------------------------------*/
(define (evcompile-let* bindings body env genv where named? tail loc)
   (let loop ((bdgs bindings)
	      (as '())
	      (env3 env))
      (if (null? bdgs)
	  (let* ((env2 (extend-env
			(reverse! (map (lambda (i) (untype-ident (car i)))
				       bindings))
			env))
		 (bd (evcompile body env2 genv where named? tail loc)))
	     (evcode 66 loc bd (reverse! as)))
	  (let* ((b (car bdgs))
		 (a (evcompile (cadr b) env3 genv where named? #f loc)))
	     (loop (cdr bdgs)
		   (cons a as)
		   (extend-env (list (car b)) env3))))))
   
;*---------------------------------------------------------------------*/
;*    evcompile-letrec ...                                             */
;*---------------------------------------------------------------------*/
(define (evcompile-letrec bindings body env genv where named? tail loc)
   (let* ((env2 (extend-env (map (lambda (i) (untype-ident (car i))) bindings)
			    env))
	  (b (evcompile body env2 genv where named? tail loc))
	  (as (map (lambda (a)
		      (evcompile (cadr a) env2 genv (car a) #t #f loc))
		   bindings))) 
      (evcode 70 loc b as)))
   
;*---------------------------------------------------------------------*/
;*    variable ...                                                     */
;*---------------------------------------------------------------------*/
(define (variable loc symbol env genv)
   (if (not (symbol? symbol))
       (evcompile-error loc "eval" "Illegal `set!' expression" symbol)
       (let ((offset (let loop ((env   env)
				(count 0))
			(cond 
			   ((null? env)
			    #f)
			   ((eq? (car env) symbol)
			    count)
			   (else
			    (loop (cdr env) (+fx count 1)))))))
	  (if offset
	      offset
	      (let ((global (if (or (eq? genv (scheme-report-environment 5))
				    (eq? genv (interaction-environment)))
				(eval-lookup symbol)
				(evcompile-error loc
						 "eval"
						 "Unbound variable"
						 symbol))))
		 (if (not global)
		     (if (eq? genv (scheme-report-environment 5))
			 (evcompile-error loc
					  "eval"
					  "Unbound variable"
					  symbol)
			 (cons 'dynamic symbol))
		     global))))))

;*---------------------------------------------------------------------*/
;*    dynamic? ...                                                     */
;*---------------------------------------------------------------------*/
(define-inline (dynamic? variable)
   (and (pair? variable)
	(eq? (car variable) 'dynamic)))

;*---------------------------------------------------------------------*/
;*    dynamic-name ...                                                 */
;*---------------------------------------------------------------------*/
(define-inline (dynamic-name dynamic)
   (cdr dynamic))

;*---------------------------------------------------------------------*/
;*    extend-env ...                                                   */
;*---------------------------------------------------------------------*/
(define (extend-env extend old-env)
   (let _loop_ ((extend extend))
      (cond
	 ((null? extend)
	  old-env)
	 ((not (pair? extend))
	  (cons extend old-env))
	 (else
	  (cons (car extend) (_loop_ (cdr extend)))))))

;*---------------------------------------------------------------------*/
;*    evcompile-error ...                                              */
;*---------------------------------------------------------------------*/
(define (evcompile-error loc proc mes obj)
   (match-case loc
      ((at ?fname ?loc)
       (error/location proc mes obj fname loc))
      (else
       (error proc mes obj))))
