/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "xslfoendnote.h"
#include "usfmtools.h"
#include "constants.h"
#include "gwrappers.h"


XslFoEndnote::XslFoEndnote (const Usfm& usfm, bool show)
// Stores the properties for all the endnote related styles.
// If "show" is true, then printing of the endnotes is done when the 
// "print" property of the \fe style is true.
{
  // Store and initialize variables.
  myshow = show;
  NoteNumberingType note_numbering_type = nntNumerical;
  position_type = eptAfterEverything;
  ustring note_numbering_user_sequence;
  standardparagraph = NULL;
  extraparagraph = NULL;
  
  // Go through all the styles.
  for (unsigned int i = 0; i < usfm.styles.size(); i++) {
    if (usfm.styles[i].type == stFootEndNote) {
      // Check subtype.
      FootEndNoteType footnotetype = (FootEndNoteType) usfm.styles[i].subtype;
      switch (footnotetype) {
        case fentFootnote:
        {
          // Footnotes not done here.
          break;
        }
        case fentEndnote:
        {
          // Store data for the endnotes.
          opening_marker = usfm_get_full_opening_marker (usfm.styles[i].marker);
          closing_marker = usfm_get_full_closing_marker (usfm.styles[i].marker);
          anchor_fontpercentage = usfm.styles[i].fontpercentage;
          anchor_italic = usfm.styles[i].italic;
          anchor_bold = usfm.styles[i].bold;
          anchor_underline = usfm.styles[i].underline;
          anchor_smallcaps = usfm.styles[i].smallcaps;
          anchor_superscript = usfm.styles[i].superscript;
          note_numbering_type = (NoteNumberingType) usfm.styles[i].userint1;
          note_numbering_user_sequence = usfm.styles[i].userstring1;
          position_type = (EndnotePositionType) usfm.styles[i].userint2;
          dump_at_marker = usfm.styles[i].userstring2;  
          if (myshow) myshow = usfm.styles[i].print;
          break;
        }
        case fentStandardContent:
        {
          // Standard content.
          if (!standardparagraph)
            standardparagraph = new XslFoFootnoteParagraph (
              usfm.styles[i].marker, usfm.styles[i].fontsize, 
              usfm.styles[i].italic, usfm.styles[i].bold, usfm.styles[i].underline, usfm.styles[i].smallcaps,
              usfm.styles[i].justification, 
              usfm.styles[i].spacebefore, usfm.styles[i].spaceafter, 
              usfm.styles[i].leftmargin, usfm.styles[i].rightmargin, usfm.styles[i].firstlineindent, 
              usfm.styles[i].userbool1);
          endnote_markers.insert (usfm.styles[i].marker);
          break;
        }
        case fentContent:
        case fentContentWithEndmarker:
        {
          // Store data for the endnote body.
          content_marker.push_back (usfm_get_full_opening_marker (usfm.styles[i].marker));
          content_apocrypha.push_back (usfm.styles[i].userbool1);
          endnote_markers.insert (usfm.styles[i].marker);
          break;
        }
        case fentParagraph:
        {
          // Store relevant data for an extra paragraph marker.
          if (!extraparagraph)
            extraparagraph = new XslFoFootnoteParagraph (
              usfm.styles[i].marker, usfm.styles[i].fontsize, 
              usfm.styles[i].italic, usfm.styles[i].bold, usfm.styles[i].underline, usfm.styles[i].smallcaps,
              usfm.styles[i].justification, 
              usfm.styles[i].spacebefore, usfm.styles[i].spaceafter, 
              usfm.styles[i].leftmargin, usfm.styles[i].rightmargin, usfm.styles[i].firstlineindent, 
              usfm.styles[i].userbool1);
          endnote_markers.insert (usfm.styles[i].marker);
          break;
        }
      }
    }
  }
  // Ensure that both of the paragraph styles are there.
  if (!standardparagraph)
    standardparagraph = new XslFoFootnoteParagraph ("ft", 11, OFF, OFF, OFF, OFF, JUSTIFIED, 0, 0, 3, 0, 0, false);
  if (!extraparagraph)
    extraparagraph =    new XslFoFootnoteParagraph ("fp", 11, OFF, OFF, OFF, OFF, JUSTIFIED, 0, 0, 3, 0, 3, false);
  // Create endnote caller object.
  endnotecaller = new NoteCaller (note_numbering_type, note_numbering_user_sequence, false);
}


XslFoEndnote::~XslFoEndnote ()
{
  delete endnotecaller;
  delete standardparagraph;
  delete extraparagraph;
}


void XslFoEndnote::transform (XmlFoBlock * xmlfoblock, ustring& line, UsfmInlineMarkers * inline_markers)
// Replace all endnote related content with corresponding xslfo code.
{
  // If no opening marker in stylesheet, bail out.
  if (opening_marker.empty())
    return;
  
  // Variables.
  size_t opening_position;
  
  // Look for footnotes, but only deal with them if they have the endmarker too.
  opening_position = line.find (opening_marker);
  while (opening_position != string::npos) {
    
    // Look for the endmarker.
    size_t closing_position;
    closing_position = line.find (closing_marker, opening_position);    
    if (closing_position == string::npos) {
      gw_warning ("Missing endmarker: " + line);
      return;
    }

    // Take out this bit of the line, transform it, and insert it again.
    ustring footnote;
    footnote = line.substr (opening_position + opening_marker.length(), closing_position - opening_position - closing_marker.length());
    line.erase (opening_position, closing_position - opening_position + closing_marker.length());
    if (myshow) {
      footnote = transform_main_parts (xmlfoblock, footnote, inline_markers);
      line.insert (opening_position, footnote);
    }
    
    // Search for another footnote.
    opening_position = line.find (opening_marker, opening_position);
  }
}


ustring XslFoEndnote::transform_main_parts (XmlFoBlock * xmlfoblock, const ustring& line, UsfmInlineMarkers * inline_markers)
{
  // Variables.
  ustring xslfo_code_text;
  ustring xslfo_code_note;
  
  // Work on a copy.
  ustring footnote (line);
  
  // Extract the endnote caller.    
  ustring caller;
  if (footnote.length() > 0) {
    caller = footnote.substr (0, 1);
    caller = trim (caller);
    if (caller == "+") {
      caller = endnotecaller->get_caller();
    } else if (caller == "-") {
      caller.clear();
    }
    footnote.erase (0, 1);
    footnote = trim (footnote);
  }
  // Insert the xslfo code.
  xslfo_code_text.append (XmlFoInlineText (caller, xmlfoblock, anchor_fontpercentage, anchor_italic, anchor_bold, anchor_underline, anchor_smallcaps, anchor_superscript, 0));

  // We now come to the endnote body.
  xslfo_code_note.append ("<fo:block>");
  xslfo_code_note.append ("<fo:list-block provisional-label-separation=\"0pt\" provisional-distance-between-starts=\"18pt\">");
  xslfo_code_note.append ("<fo:list-item>");
  
  // Insert the caller in the footnote body too.
  xslfo_code_note.append ("<fo:list-item-label end-indent=\"label-end()\">");
  vector<ustring> lines;
  {
    XmlFoBlock block (&lines, standardparagraph->fontsize, 100,
                      standardparagraph->italic, standardparagraph->bold,
                      standardparagraph->underline, standardparagraph->smallcaps,
                      standardparagraph->justification, 
                      standardparagraph->spacebefore, standardparagraph->spaceafter, 
                      0, 0, 0, false, false);
    if (!caller.empty()) {
      lines.push_back (XmlFoInlineText (caller, &block, anchor_fontpercentage, anchor_italic, anchor_bold, anchor_underline, anchor_smallcaps, false, 0));
    }
  }
  xslfo_code_note.append ("\n");
  for (unsigned int i = 0; i < lines.size(); i++) {
    xslfo_code_note.append (lines[i]);
    xslfo_code_note.append ("\n");
  }
  xslfo_code_note.append ("</fo:list-item-label>");

  // Insert some code for this endnote into the body.
  xslfo_code_note.append ("<fo:list-item-body start-indent=\"body-start()\">");
  
  // Divide the endnote into paragraphs, if there are more than one.
  vector<ustring> paragraphs;
  size_t pos;
  pos = footnote.find (extraparagraph->marker_open);
  while (pos != string::npos) {
    paragraphs.push_back (footnote.substr (0, pos));
    footnote.erase (0, pos + extraparagraph->marker_open.length());
    pos = footnote.find (extraparagraph->marker_open);
  }
  paragraphs.push_back (footnote);

  // Deal with each paragraph.
  for (unsigned int i = 0; i < paragraphs.size(); i++) {
    // Choose the right paragraph.
    XslFoFootnoteParagraph * paragraph;
    if (i == 0)
      paragraph = standardparagraph;
    else
      paragraph = extraparagraph;
    // Format actual text, and comply with footnote nesting, see USFM standard.
    paragraphs[i] = usfm_notes_handle_nesting (paragraphs[i], 
      standardparagraph->marker_open, standardparagraph->marker_close, endnote_markers);
    // Insert the paragraph as another block.
    lines.clear();
    {
      XmlFoBlock block (&lines, paragraph->fontsize, 100,
                        paragraph->italic, paragraph->bold,
                        paragraph->underline, paragraph->smallcaps,
                        paragraph->justification, 
                        paragraph->spacebefore, paragraph->spaceafter, 
                        paragraph->leftmargin, paragraph->rightmargin, 
                        paragraph->firstlineindent, false, false);
      lines.push_back (paragraphs[i]);
    }
    xslfo_code_note.append ("\n");
    for (unsigned int i = 0; i < lines.size(); i++) {
      xslfo_code_note.append (lines[i]);
      xslfo_code_note.append ("\n");
    }
  }    
  
  // Close footnote and foot notebody.
  xslfo_code_note.append ("</fo:list-item-body>");
  xslfo_code_note.append ("</fo:list-item>");
  xslfo_code_note.append ("</fo:list-block>");
  xslfo_code_note.append ("</fo:block>");

  // Transform inline markers in the footnote.
  if (inline_markers) {
    usfm_handle_inline_text (xslfo_code_note, inline_markers, NULL, imXslFo, NULL);
  }

  // Store the code for the note, and return the code to be put in the text.
  notes.push_back (xslfo_code_note);
  return xslfo_code_text;  
}


void XslFoEndnote::end_book (vector<ustring>& xslfo_lines)
/*
This function is supposed to be called when the printing routine reaches
the end of a book.
If the settings call for it, it will print and flush all endnotes collected
thus far.
*/
{
  if (position_type != eptAfterBook) return;
  dump (xslfo_lines);
}


void XslFoEndnote::end_project (vector<ustring>& xslfo_lines)
/*
This function is supposed to be called when the printing routine reaches
the end of the project.
If the settings call for it, it will print and flush all endnotes collected
thus far.
*/
{
  if (position_type != eptAfterEverything) return;
  dump (xslfo_lines);
}


bool XslFoEndnote::new_marker (const ustring& marker)
/*
This function is supposed to be called when the printing routine finds a new
marker.
If the settings call for it, it will return true to signal that the endnotes
should be printed and flushed.
*/
{
  if (position_type != eptAtMarker) return false;
  if (marker != dump_at_marker) return false;
  return true;
}


void XslFoEndnote::dump (vector<ustring>& xslfo_lines)
/*
This dumps all footnotes collected thus far.
*/
{
  for (unsigned int i = 0; i < notes.size(); i++) {
    xslfo_lines.push_back (notes[i]);
  }
  notes.clear();
}
