/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "libraries.h"
#include "utilities.h"
#include "references.h"
#include "directories.h"
#include "constants.h"
#include "referenceutils.h"
#include "bible.h"


References::References (GtkListStore * liststore, GtkWidget * treeview, GtkTreeViewColumn * treecolumn)
{
  myliststore = liststore;
  mytreeview = treeview;
  mytreecolumn = treecolumn;
}


References::~References ()
{
}


void References::load (const ustring & filename)
{
  // Loads references from a file.
  // If no filename is given, the default name is used.
  ustring file (filename);
  if (file.length () == 0)
    file = references_default_filename ();
  try
  {
    ReadText rt (file, true);
    // Pick out the references and leave the rest.
    ustring book;
    ustring chapter;
    ustring verse;
    for (unsigned int i = 0; i < rt.lines.size (); i++)
      if (reference_discover ("", "", "", rt.lines[i], book, chapter, verse))
        references.push_back (book + " " + chapter + ":" + verse);
    sort_references (references);
    load_searchwords (file);
  }
  catch (exception & ex)
  {
    cerr << "Loading references: " << ex.what () << endl;
  }
  comments.clear();
  for (unsigned int i = 0; i < references.size(); i++)
    comments.push_back ("Loaded from file");
}


void References::fill_store ()
{
  /*
   * Fill the list store with the data.
   */
  // Clear the store first.
  gtk_list_store_clear (myliststore);
  // Now add new data.
  GtkTreeIter iter;
  for (unsigned int i = 0; i < references.size (); i++)
  {
    gtk_list_store_append (myliststore, &iter);
    gtk_list_store_set (myliststore, &iter, 0, references[i].c_str (), 1, comments[i].c_str(), -1);
  }
  // Save references to default location.
  save ("");
  // Show number of references in the title of the column
  set_header ();
}


void References::set_header ()
{
  // Sets the header in the treeview to the number of references.
  string s = TEXT_REFERENCES;
  s.append (" - ");
  s.append (convert_to_string (int (references.size ())));
  gtk_tree_view_column_set_title (mytreecolumn, s.c_str ());
}


void References::save (const ustring & filename)
{
  // Saves references to a file.
  // If no filename is given, the default name is used.
  ustring file (filename);
  if (file.length () == 0)
    file = references_default_filename ();
  try
  {
    write_lines (file, references);
  }
  catch (exception & ex)
  {
    cerr << "Saving references: " << ex.what () << endl;
  }
  save_searchwords (file);
}


void References::goto_next ()
{
  /* 
   * This selects the next reference, if there is any.
   * If no item has been selected it selects the first, if it's there.
   */
  goto_next_previous_internal (true);
}


void References::goto_previous ()
{
  /* 
   * This goes to the previous reference, if there is any.
   * If no item has been selected it chooses the first, if it's there.
   */
  goto_next_previous_internal (false);
}


void References::goto_next_previous_internal (bool next)
{
  // Continue only when references are available.
  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (myliststore), &iterator)) {
    // See which is selected.
    GtkTreeSelection *selection;
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (mytreeview));
    // See how many are selected.
    gint number_selected;
    number_selected = gtk_tree_selection_count_selected_rows (selection);
    if (number_selected > 0) {
      // One or more selected: Store pointer to last one selected, and select next.
      gtk_tree_selection_selected_foreach (selection, References::goto_foreach_function, gpointer (&iterator));
      if (next) {
        if (gtk_tree_model_iter_next (GTK_TREE_MODEL (myliststore), &iterator)) {
          gtk_tree_selection_unselect_all (selection);
          gtk_tree_selection_select_iter (selection, &iterator);
        }
      } 
      else {
        GtkTreePath *path;
        path = gtk_tree_model_get_path (GTK_TREE_MODEL (myliststore), &iterator);
        if (gtk_tree_path_prev (path)) {
          gtk_tree_selection_unselect_all (selection);
          gtk_tree_model_get_iter (GTK_TREE_MODEL (myliststore), &iterator, path);
          gtk_tree_selection_select_iter (selection, &iterator);
        }
      }
    }
    else {
      // None selected: select the first one.
      gtk_tree_selection_select_iter (selection, &iterator);
    }
    // Scroll, if needed, to make the new selection visible.
    if (gtk_list_store_iter_is_valid (myliststore, &iterator)) {
      GtkTreePath *path;
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (myliststore), &iterator);
      if (next)
        gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (mytreeview), path, NULL, true, 0.9, 0);
      else
        gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (mytreeview), path, NULL, true, 0.1, 0);
      gtk_tree_path_free (path);
    }
  }
}


void References::goto_foreach_function (GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
  GtkTreeIter *iterator;
  iterator = (GtkTreeIter *) data;
  *iterator = *iter;
}


void References::get_loaded ()
{
  // This gets the references that are loaded in the editor.
  references.clear ();
  gtk_tree_model_foreach (GTK_TREE_MODEL (myliststore), GtkTreeModelForeachFunc (loaded_get_foreach_function), gpointer (&references));
}


gboolean References::loaded_get_foreach_function (GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
  gchar * reference;
  gtk_tree_model_get (model, iter, 0, &reference, -1);
  ((vector < string > *)data)->push_back (reference);
  g_free (reference);
  return false;
}


#define REFERENCES_SEARCHWORD "searchword: "
#define REFERENCES_CASE "casesensitive: "
#define REFERENCES_GLOB "globbing: "
#define REFERENCES_MATCH_BEGIN "matchbegin: "
#define REFERENCES_MATCH_END "matchend: "


void References::load_searchwords (const ustring & filename)
/*
  Read the file with the search words, and pick out those
  that belong to variabele filename. See the save method
  for the format of the file
*/
{
  try
  {
    searchwords.clear ();
    casesensitives.clear ();
    globs.clear ();
    matchbeginnings.clear ();
    matchendings.clear ();
    ustring searchword;
    bool casesensitive = false;
    bool globbing = false;
    bool matchbeginning = false;
    bool matchending = false;
    bool take_this = false;
    int take_what = 0;
    vector <ustring> lines;    
    ReadText rt (references_searchwords_filename (), true);
    for (unsigned int i = 0; i < rt.lines.size (); i++) {
      ustring s = trim (rt.lines[i]);
      ustring first;
      ustring last;
      if (s.length () > 0) {
        first = s[0];
        last = s[s.length () - 1];
      }
      if ((first == "[") && (last == "]"))
        take_this = false;
      if (s.find (filename) != string::npos)
        take_this = true;
      if (take_this) {
        take_what++;
      }
      else {
        take_what = 0;
      }
      if (take_what > 1) {
        if (s.find (REFERENCES_SEARCHWORD) == 0) {
          s.erase (0, strlen (REFERENCES_SEARCHWORD));
          searchword = s;
        }
        if (s.find (REFERENCES_CASE) == 0) {
          s.erase (0, strlen (REFERENCES_CASE));
          casesensitive = convert_to_bool (s);
        }
        if (s.find (REFERENCES_GLOB) == 0) {
          s.erase (0, strlen (REFERENCES_GLOB));
          globbing = convert_to_bool (s);
        }
        if (s.find (REFERENCES_MATCH_BEGIN) == 0) {
          s.erase (0, strlen (REFERENCES_MATCH_BEGIN));
          matchbeginning = convert_to_bool (s);
        }
        if (s.find (REFERENCES_MATCH_END) == 0) {
          s.erase (0, strlen (REFERENCES_MATCH_END));
          matchending = convert_to_bool (s);
          // As this is the last one in the list, store everything at once.
          searchwords.push_back (searchword);
          casesensitives.push_back (casesensitive);
          globs.push_back (globbing);
          matchbeginnings.push_back (matchbeginning);
          matchendings.push_back (matchending);
        }
      }
    }
  }
  catch (exception & ex)
  {
    cerr << "Reading searchwords: " << ex.what () << endl;
  }
}


void References::save_searchwords (const ustring& filename)
{
/*
    Whenever any file with references is saved,
    the search- and/or replacewords, that formed this 
    list of references, are saved too. Also saved is 
    whether this search and/or replace was casesensitive.
    There is one file where all data is saved in.
    A typical file might look like this:
    
    [/home/user/references]
    Word
    0
    [/home/user/bibledit/refs]
    Word1
    1
    word2
    0
    word3
    1
    
    First the name of the file with references gets stored,
    between square brackets.
    Next each search- and/or replaceword is stored, together with
    a 0 or 1 indicating whether the search/replace was case sensitive.
    Each filename can be followed by zero or more entries consisting of
    word and case sensitivity.
    
    This has been expanded with saving these three extra variables:
    - Whether globbing was done
    - Whether the search matched the beginning of a word
    - Whether the search matched the ending of a word.
    They are saved in a different way, see the actual file produced 
    by bibledit.
*/
  try
  {
    bool delete_this = false;
    vector < ustring > lines;
    ReadText rt (references_searchwords_filename (), true);
    for (unsigned int i = 0; i < rt.lines.size (); i++) {
      ustring s = trim (rt.lines[i]);
      ustring first;
      ustring last;
      if (s.length () > 0) {
        first = s[0];
        last = s[s.length () - 1];
      }
      if ((first == "[") && (last == "]"))
        delete_this = false;
      if (s.find (filename) != string::npos)
        delete_this = true;
      if (!delete_this)
        lines.push_back (s);
    }
    lines.push_back ("[" + filename + "]");
    for (unsigned int i = 0; i < searchwords.size (); i++) {
      lines.push_back (REFERENCES_SEARCHWORD + searchwords[i]);
      lines.push_back (REFERENCES_CASE + convert_to_string (casesensitives[i]));
      lines.push_back (REFERENCES_GLOB + convert_to_string (globs[i]));
      lines.push_back (REFERENCES_MATCH_BEGIN + convert_to_string (matchbeginnings[i]));
      lines.push_back (REFERENCES_MATCH_END + convert_to_string (matchendings[i]));
    }
    write_lines (references_searchwords_filename (), lines);
  }
  catch (exception & ex)
  {
    cerr << "Saving searchwords: " << ex.what () << endl;
  }
}


void References::get_references (vector <ustring>& refs, Session * session)
{
  refs.assign (references.begin (), references.end ());
  session->highlight_words.assign (searchwords.begin (), searchwords.end ());
  session->highlight_casesensitives.assign (casesensitives.begin (), casesensitives.end ());
  session->highlight_globbings.assign (globs.begin (), globs.end ());
  session->highlight_matchbegins.assign (matchbeginnings.begin (), matchbeginnings.end ());
  session->highlight_matchends.assign (matchendings.begin (), matchendings.end ());
}


void References::get_references (vector <ustring > &refs)
{
  refs.assign (references.begin (), references.end ());
}


void References::get_references (Session * session)
{
  session->highlight_words.assign (searchwords.begin (), searchwords.end ());
  session->highlight_casesensitives.assign (casesensitives.begin (), casesensitives.end ());
  session->highlight_globbings.assign (globs.begin (), globs.end ());
  session->highlight_matchbegins.assign (matchbeginnings.begin (), matchbeginnings.end ());
  session->highlight_matchends.assign (matchendings.begin (), matchendings.end ());
}


void References::set_references (vector <ustring> &refs, Session * session, Configuration * configuration)
{
  vector<ustring> comments_in;
  for (unsigned int i = 0; i < refs.size(); i++)
    comments_in.push_back ("Search result");
  references_hidden_ones_filter (refs, comments_in, configuration);
  references.assign (refs.begin (), refs.end ());
  comments.assign (comments_in.begin(), comments_in.end());
  searchwords.assign (session->highlight_words.begin (), session->highlight_words.end ());
  casesensitives.assign (session->highlight_casesensitives.begin (), session->highlight_casesensitives.end ());
  globs.assign (session->highlight_globbings.begin (), session->highlight_globbings.end ());
  matchbeginnings.assign (session->highlight_matchbegins.begin (), session->highlight_matchbegins.end ());
  matchendings.assign (session->highlight_matchends.begin (), session->highlight_matchends.end ());
}


void References::set_references (vector <ustring>& references_in, vector<ustring>& comments_in, Configuration * configuration)
{
  references_hidden_ones_filter (references_in, comments_in, configuration);
  references.assign (references_in.begin (), references_in.end ());
  comments.assign (comments_in.begin(), comments_in.end());
  searchwords.clear();
  casesensitives.clear();
  globs.clear();
  matchbeginnings.clear();
  matchendings.clear();
}
