/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include <glib.h>
#include "print_parallel_bible.h"
#include "utilities.h"
#include "bible.h"
#include "book.h"
#include "chapter.h"
#include "usfm.h"
#include "usfmtools.h"
#include <config.h>
#include "scripture.h"
#include "pdfviewer.h"
#include "xmlutils.h"
#include "paper.h"
#include "formatter.h"
#include "xmlfo-utils.h"
#include "constants.h"
#include "project.h"
#include "gwrappers.h"
#include "gtkwrappers.h"
#include "directories.h"
#include "fontutils.h"
#include "notecaller.h"
#include "xslfofootnote.h"
#include "xslfoendnote.h"
#include "xslfoxref.h"
#include "bookmetrics.h"
#include "mapping.h"
#include "portion_utils.h"


void view_parallel_bible_pdf (Configuration * configuration, Session * session, Progress * progress)
/*
Formats the references in "references", and highlights all words in
"session->highlights*" and shows them in a pdf viewer.
*/
{
  // Log.
  gw_message ("Printing Parallel Bible");
 
  // Open first scripture and book and check it.
  Scripture scripture (configuration->project);
  int book_index = scripture.get_index_of_book (configuration->book_opened);
  if (book_index < 0) {
    gtkw_dialog_info (NULL, configuration->book_opened + "does not exist in this project");
    return;
  }
  Book book (scripture.paths[book_index], false, "");
  BookMetrics bookmetrics (scripture.paths[book_index]);
  vector<unsigned int> chapters = bookmetrics.get_chapters ();
  
  // Progress system.
  progress->start();
  progress->set_text ("Collecting verses");
  progress->set_iterate (0, 1, chapters.size());
  
  // Prepare for the inline text markers.
  Project project (configuration);
  Usfm usfm (project.get_stylesheet());
  UsfmInlineMarkers usfm_inline_markers (usfm);

  // Prepare for leaving out footnotes, endnotes and crossreferences.
  XslFoFootnote xslfofootnote (usfm, configuration, false);
  XslFoEndnote xslfoendnote (usfm, configuration, false);
  XslFoXref xslfoxref (usfm, configuration, false);
  
  // Prepare for mapping.
  Mapping mapping (project.versification, configuration->book_opened);
  
  // Storage for final xsl-fo file.
  vector<ustring> xslfo_lines;

  // Start building the xml-fo file.
  // The various elements of this file are created using objects.
  // When the object goes out of scope, it writes the closing elements.
  {
    XmlFoRoot xmlforoot (&xslfo_lines, configuration);
    {
      XmlFoLayoutMasterSet layoutmasterset (&xslfo_lines, configuration, false);
    }      
    {
      XmlFoPageSequence pagesequence (&xslfo_lines, false);
      {
        XmlFoStaticContent staticcontent (&xslfo_lines, configuration);
      }
      {
        XmlFoFlow flow (&xslfo_lines);
        {
          // Messages to be printed first.
          vector<ustring> messages;
          // All the projects to be put in this parallel Bible.
          vector<ustring> project_s_raw = configuration->parallel_bible_projects_get ();
          vector<ustring> project_names;
          vector<Scripture> scripture_s;
          vector<int> book_index_s;
          vector<Book> book_s;
          vector<Mapping> mapping_s;
          vector<Chapter *> chapter_s;
          vector<unsigned int> chapter_numbers;
          for (unsigned int i = 0; i < project_s_raw.size(); i++) {
            Scripture scripture (project_s_raw[i]);
            Project project (project_s_raw[i]);
            int book_index = scripture.get_index_of_book (configuration->book_opened);
            if (book_index >= 0) {
              Book book (scripture.paths[book_index], false, "");
              Mapping mapping (project.versification, configuration->book_opened);
              project_names.push_back(project_s_raw[i]);
              scripture_s.push_back (scripture);
              book_index_s.push_back (book_index);
              book_s.push_back (book);
              mapping_s.push_back (mapping);
              chapter_s.push_back (NULL);
              chapter_numbers.push_back (1000);
            } else {
              messages.push_back ("Project "  + project_s_raw[i] + " was requested to be included, but it does not contain " + configuration->book_opened + ". It was left out.");
            }
          }
          // Print any messages.
          if (messages.size() > 0) {
            xslfo_lines.push_back ("<fo:block font-size=\"120%\" font-weight=\"bold\">");
            xslfo_lines.push_back ("Note");            
            xslfo_lines.push_back ("</fo:block>");
          }
          for (unsigned int i = 0; i < messages.size(); i++) {
            xslfo_lines.push_back ("<fo:block>");
            xslfo_lines.push_back (messages[i]);
            xslfo_lines.push_back ("</fo:block>");
          }
          // Variables for portion selection.
          bool portion_print = false;
          bool portion_print_next_verse_off = false;
          unsigned int portion_chapter_from, portion_chapter_to;
          ustring portion_verse_from, portion_verse_to;
          select_portion_get_values (bookmetrics, configuration->parallel_bible_chapters_verses,
                                     portion_chapter_from, portion_verse_from, 
                                     portion_chapter_to, portion_verse_to);
          // Produce chunks for the xsl-fo file for all references.
          for (unsigned int ch = 0; ch < chapters.size(); ch++) {
            // Update progress bar.
            progress->iterate();
            if (progress->cancel) {
              progress->finish();
              return;
            }
            // Get the chapter text of the first project.
            vector <ustring> chapter_lines;
            book.get_chapter (chapters[ch], chapter_lines);
            Chapter chapter (chapter_lines);
            // Go through the verse numbers in this chapter.
            vector<ustring> verses = bookmetrics.get_verses (chapters[ch]);
            for (unsigned int vs = 0; vs < verses.size(); vs++) {
              // See whether this chapter.verse is within our portion to print.
              if ((chapters[ch] == portion_chapter_from) && (verses[vs] == portion_verse_from))
                portion_print = true;
              if (portion_print_next_verse_off)
                portion_print = false;
              if ((chapters[ch] == portion_chapter_to) && (verses[vs] == portion_verse_to))
                portion_print_next_verse_off = true;
              if (!portion_print)
                continue;
              // See whether to print verses zero.
              if (!configuration->parallel_bible_include_verse_zero)
                if (verses[vs] == "0")
                  continue;
              // Add the block to the xsl-fo file.
              // Next line has keep-together.within-page="always", rather than
              // keep-together="always", as the latter one causes things to be 
              // kept together, in in a line, which causes the line to overflow
              // the right margin.
              if (configuration->parallel_bible_keep_verses_together)
                xslfo_lines.push_back ("<fo:block keep-together.within-page=\"always\">");
              else
                xslfo_lines.push_back ("<fo:block>");
              // XSLFormatter is better than FOP in that it does honour space conditionality,
              // which is initially set at "discard" for the beginning of a 
              // references area, as here. So to get the distance between the 
              // lines right, this is inserted: space-before.conditionality="retain".
              xslfo_lines.push_back ("  <fo:block space-before=\"2mm\" space-before.conditionality=\"retain\">");
              xslfo_lines.push_back (configuration->book_opened + " " + convert_to_string (chapters[ch]) + ":" + verses[vs]);
              xslfo_lines.push_back ("  </fo:block>");
              // Map this verse to the original, that is, to Hebrew or Greek.
              vector<int> hebrew_greek_chapters;
              vector<int> hebrew_greek_verses;
              mapping.me_to_original (chapters[ch], verses[vs], hebrew_greek_chapters, hebrew_greek_verses);
              // Get verse text for each version.
              for (unsigned int vsn = 0; vsn <= project_names.size(); vsn++) {
                // Get the verse text.
                vector <ustring> verse;
                if (vsn == 0) {
                  // First version.
                  chapter.get_verse (verses[vs], verse);
                } else {
                  // Other versions. 
                  // Get mapped chapters / verses.
                  vector<int> mychapters;
                  vector<int> myverses;
                  mapping_s[vsn - 1].original_to_me (hebrew_greek_chapters, hebrew_greek_verses, mychapters, myverses);
                  // Get text of any of the mapped verses.
                  for (unsigned int mp = 0; mp < mychapters.size(); mp++) {
                    // Chapter numbers differ?
                    if ((unsigned int) mychapters[mp] != chapter_numbers[vsn - 1]) {
                      // Free memory if need be.
                      if (chapter_s[vsn - 1])
                        delete chapter_s[vsn - 1];
                      // Get new data.
                      vector <ustring> chapter_lines;
                      book_s[vsn - 1].get_chapter (mychapters[mp], chapter_lines);
                      Chapter * chapter = new Chapter (chapter_lines);
                      chapter_s[vsn - 1] = chapter;
                      chapter_numbers[vsn - 1] = mychapters[mp];
                    }
                    // Get the verse and add it to our container.
                    vector <ustring> local_verses;
                    chapter_s[vsn - 1]->get_verse (convert_to_string (myverses[mp]), local_verses);
                    for (unsigned int i = 0; i < local_verses.size(); i++) {
                      verse.push_back (local_verses[i]);
                    }
                  }
                }
                // As a verse may consist of more lines, deal with each.
                ustring line;
                for (unsigned int i2 = 0; i2 < verse.size (); i2++) {
                  if (!line.empty())
                    line.append (" ");
                  line.append (verse[i2]);
                }
                // Take out verse number if it is the desired verse number, else leave it.
                usfm_extract_marker (line);
                if (line.find (verses[vs]) == 0)
                  line.erase (0, verses[vs].length() + 1);
                // Take out footnotes, endnotes, crossreferences.
                xslfofootnote.transform (NULL, line);
                xslfoendnote.transform (NULL, line);
                xslfoxref.transform (NULL, line);              
                // Change <, > and & to their corresponding entities.
                vector <size_t> highlight_positions;
                xml_handle_entities (line, highlight_positions);
                // Deal with inline markers, so that they do not appear in the output
                // as markers, but format the text instead.
                usfm_handle_inline_text (line, &usfm_inline_markers, NULL);
                // Extract any marker left in the line.
                {
                  ustring marker;
                  size_t position, length;
                  bool opening_marker;
                  while (usfm_search_marker (line, marker, position, length, opening_marker)) {
                    line.erase (position, length);
                  }
                }
                // Add this line to the xsl-fo file.
                xslfo_lines.push_back ("  <fo:block>");
                ustring project;
                if (vsn == 0)
                  project = configuration->project;
                else
                  project = project_names[vsn - 1];
                xslfo_lines.push_back ("<fo:inline font-size=\"70%\">" + project + "</fo:inline> ");
                xslfo_lines.push_back (line);
                xslfo_lines.push_back ("  </fo:block>");
              }
              // Close main block.
              xslfo_lines.push_back ("</fo:block>");
            }
          }
          // Clear memory for chapters.
          for (unsigned int i = 0; i < chapter_s.size(); i++) {
            if (chapter_s[i])
              delete chapter_s[i];
          }
        }
      }
    }
  }
  
  // Make a temporary directory where to put the working files and the resulting
  // .pdf file.information. This directory is not removed, because the pdf viewer 
  // needs the .pdf file to be there during viewing or printing.
  string working_directory = gw_build_filename(directories_get_temp(), "parallelbible");
  create_directory (working_directory);
  // Produce filename of .fo file.
  string fofilename = gw_build_filename(working_directory, "document.fo");
  // Write the document.
  write_lines (fofilename, xslfo_lines);
  // Tell user to be patient while we pdfxmltex runs.  
  progress->set_text ("Typesetting pages ...");
  if (progress->cancel) {
    progress->finish();
    return;
  }
  // Convert the xsl-fo document to .pdf.
  NoteCaller dummy (nntNumerical, "", false);
  string pdffilename = gw_build_filename(working_directory, "document.pdf");
  int conversion_result = formatter_convert_to_pdf (configuration, fofilename, pdffilename, progress, &dummy, &dummy);
  // Progressbar: ready.
  progress->set_fraction (1);
  progress->set_text ("Ready");
  if (progress->cancel) {
    progress->finish();
    return;
  }
  // View the .pdf document.
  pdfviewer (pdffilename);
  // Finalize progress bar.
  progress->finish();
  // Give message if there were errors.
  if (conversion_result != 0) {
    string message = "The formatter had some trouble.\n"
                     "See menu Help - System log for more details.\n"
                     "See the helpfile for a possible solution.";
    gtkw_dialog_error (NULL, message);
  };
  // Log: ready.
  gw_message ("Ready printing the Parallel Bible");
}
